package com.jiejing.fitness.finance.service.pay.impl;

import static java.util.stream.Collectors.toList;

import com.alibaba.fastjson.JSON;
import com.alipay.api.AlipayApiException;
import com.alipay.api.AlipayClient;
import com.alipay.api.DefaultAlipayClient;
import com.alipay.api.request.AlipaySystemOauthTokenRequest;
import com.alipay.api.response.AlipaySystemOauthTokenResponse;
import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.google.common.collect.Lists;
import com.jiejing.common.exception.BizException;
import com.jiejing.common.utils.collection.CollectionUtil;
import com.jiejing.common.utils.time.TimeUtil;
import com.jiejing.fitness.enums.auth.AuthDomainEnum;
import com.jiejing.fitness.enums.finance.BrandCashierTransStateEnum;
import com.jiejing.fitness.enums.tenant.TenantTypeEnum;
import com.jiejing.fitness.finance.repository.entity.GlobalConfig;
import com.jiejing.fitness.finance.repository.entity.MerchantSettleRecord;
import com.jiejing.fitness.finance.repository.entity.StudioCashierRecord;
import com.jiejing.fitness.finance.repository.entity.PartyToMerchant;
import com.jiejing.fitness.finance.repository.entity.StudioMerchantApply;
import com.jiejing.fitness.finance.repository.entity.StudioSettleRecord;
import com.jiejing.fitness.finance.repository.service.GlobalConfigRpService;
import com.jiejing.fitness.finance.repository.service.MerchantSettleRecordRpService;
import com.jiejing.fitness.finance.repository.service.StudioCashierRecordRpService;
import com.jiejing.fitness.finance.repository.service.PartyToMerchantRpService;
import com.jiejing.fitness.finance.repository.service.StudioCheckSettleRecordRpService;
import com.jiejing.fitness.finance.repository.service.StudioMerchantApplyRpService;
import com.jiejing.fitness.finance.repository.service.StudioSettleRecordRpService;
import com.jiejing.fitness.finance.service.config.PayChannelProperties;
import com.jiejing.fitness.finance.service.enums.FinanceErrorEnums;
import com.jiejing.fitness.finance.service.enums.GlobalConfigEnums;
import com.jiejing.fitness.finance.service.pay.PayService;
import com.jiejing.fitness.finance.service.pay.convert.PayConvert;
import com.jiejing.fitness.finance.service.pay.enums.PayFailMessageReplaceEnums;
import com.jiejing.fitness.finance.service.pay.params.BrandMerchantRefundParams;
import com.jiejing.fitness.finance.service.pay.params.NativePayParams;
import com.jiejing.fitness.finance.service.pay.params.StudioMerchantPayParams;
import com.jiejing.fitness.finance.service.rpc.MerchantRpcService;
import com.jiejing.fitness.finance.service.rpc.PayRpcService;
import com.jiejing.fitness.finance.service.rpc.PermissionRpcService;
import com.jiejing.fitness.finance.service.rpc.StudioRpcService;
import com.jiejing.fitness.finance.service.utils.DingUtil;
import com.jiejing.fitness.finance.service.utils.FeeUtil;
import com.jiejing.fitness.finance.service.utils.MoneyUtil;
import com.jiejing.message.enums.MsgChannelEnum;
import com.jiejing.message.event.SendCommonMsgEvent;
import com.jiejing.paycenter.common.enums.common.PayChannelEnums;
import com.jiejing.paycenter.common.enums.common.TransStateEnums;
import com.jiejing.paycenter.common.enums.pay.PayStateEnums;
import com.jiejing.paycenter.common.enums.pay.PayTypeEnums;
import com.jiejing.paycenter.common.model.vo.MerchantVO;
import com.jiejing.paycenter.api.pay.request.PayRequest;
import com.jiejing.paycenter.api.pay.request.RefundPayRequest;
import com.jiejing.paycenter.common.model.vo.PayVO;
import com.jiejing.paycenter.api.pay.vo.RefundVO;
import com.jiejing.paycenter.common.event.PayEvent;
import com.jiejing.paycenter.common.event.RefundEvent;
import com.jiejing.paycenter.common.model.vo.SettleVO;
import com.jiejing.studio.api.studio.vo.StudioVO;
import com.jiejing.wechat.WeChatAuthService;
import com.jiejing.wechat.vo.weChat.BaseAuthInfoVO;
import com.xiaomai.event.EventAgent;
import java.math.BigDecimal;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.support.TransactionTemplate;

/**
 * @author chengyubing
 * @since 2024/2/27 11:06
 */
@Slf4j
@Service
public class PayServiceImpl implements PayService {

  private static final BigDecimal MAX_AMOUNT = new BigDecimal("10000000");

  @Resource
  private PayChannelProperties config;

  @Resource
  private PartyToMerchantRpService partyToMerchantRpService;

  @Resource
  private PayRpcService payRpcService;

  @Resource
  private StudioRpcService studioRpcService;

  @Resource
  private StudioCashierRecordRpService studioCashierRecordRpService;

  @Resource
  private MerchantRpcService merchantRpcService;

  @Resource
  private StudioSettleRecordRpService studioSettleRecordRpService;

  @Resource
  private StudioMerchantApplyRpService studioMerchantApplyRpService;

  @Resource
  private StudioCheckSettleRecordRpService studioCheckSettleRecordRpService;

  @Resource
  private WeChatAuthService weChatAuthService;

  @Resource
  private GlobalConfigRpService globalConfigRpService;

  @Resource
  private TransactionTemplate transactionTemplate;

  @Resource
  private PermissionRpcService permissionRpcService;

  @Resource
  private MerchantSettleRecordRpService merchantSettleRecordRpService;

  @Resource(name = "financeThreadPool")
  private Executor executor;

  private final Map<String, DefaultAlipayClient> aliClientMap = new HashMap<>();

  @PostConstruct
  public void init() {
    GlobalConfig config = globalConfigRpService.getById(GlobalConfigEnums.CASHIER_ALI_INFO.getCode())
        .orElse(new GlobalConfig());
    AliInfo info = JSON.parseObject(config.getConfigValue(), AliInfo.class);
    aliClientMap.put(info.getAppId(),
        new DefaultAlipayClient(info.getBaseUrl(), info.getAppId(), info.getMerchantPrivateKey(), "json",
            "utf-8", info.getAlipayPublicKey(), "RSA2"));
  }

  @Override
  public String getOpenId(PayChannelEnums channel, String appId, String authCode) {
    switch (channel) {
      case WX:
        return Optional.ofNullable(weChatAuthService.getBaseAuthInfo(null, appId, authCode))
            .map(BaseAuthInfoVO::getOpenId).orElse(null);
      case ALI:
        try {
          AlipaySystemOauthTokenResponse response = aliClientMap.get(appId).execute(convert(authCode));
          log.info("get ali openId response {}, {}, {}", appId, authCode, JSON.toJSONString(response));
          return response.getOpenId();
        } catch (AlipayApiException e) {
          throw new BizException(e.getErrCode(), e.getMessage());
        }
      default:
        throw new BizException(FinanceErrorEnums.NOT_SUPPORT_TYPE);
    }
  }

  public AlipaySystemOauthTokenRequest convert(String authCode) {
    AlipaySystemOauthTokenRequest aliRequest = new AlipaySystemOauthTokenRequest();
    aliRequest.setCode(authCode);
    aliRequest.setGrantType("authorization_code");
    return aliRequest;
  }


  @Override
  public PayVO nativePay(NativePayParams params) {
    return payRpcService.pay(convertNativePay(params));
  }

  @Override
  public PayVO merchantPay(StudioMerchantPayParams params) {

    Pair<PartyToMerchant, StudioVO> pair = this.checkBeforeMerchantPay(params);

    MerchantVO merchant = merchantRpcService.getByMerchantId(pair.getLeft().getMerchantId());
    StudioCashierRecord record = PayConvert.convertPayInit(params, pair.getRight(), merchant);
    PayRequest request = PayConvert.convert(params, record);
    record.setFeeRate(payRpcService.getFeeRate(request));
    record.setFee(FeeUtil.calPayFee(record.getFeeRate(), params.getTransAmount()));
    record.setActualAmount(MoneyUtil.subtract(params.getTransAmount(), record.getFee()));
    studioCashierRecordRpService.insert(record);
    PayVO vo = payRpcService.pay(request);
    this.payCallback(PayConvert.convertEvent(request, vo));

    if (PayStateEnums.FAILED == PayStateEnums.getByCode(vo.getPayState())) {
      throw new BizException(replaceFailMessage(vo.getFailMsg()));
    }
    return vo;
  }

  @Override
  public void payCallback(PayEvent event) {
    if (!event.getChannelNo().equals(config.getCashier())) {
      return;
    }

    Boolean result = transactionTemplate.execute(action -> {
      Long id = Long.parseLong(event.getTransNo());
      StudioCashierRecord record = studioCashierRecordRpService.getByIdForUpdate(id).orElse(null);
      if (null == record) {
        return false;
      }

      BrandCashierTransStateEnum originalState = BrandCashierTransStateEnum.valueOf(record.getTransState());
      BrandCashierTransStateEnum targetState = PayConvert.convertTransState(event.getPayState());
      if (targetState == originalState) {
        return false;
      }

      if (isPaying(targetState) && isPayFinished(originalState)) {
        return false;
      }

      StudioCashierRecord toModify = PayConvert.convertPay(record, event);
      studioCashierRecordRpService.updateById(toModify);
      return true;
    });

    if (null != result && result) {
      if (PayStateEnums.SUCCESS.getCode().equals(event.getPayState())) {
        executor.execute(() -> this.sendPaySuccessMessage(event));
      }
    }

  }

  @Override
  public RefundVO merchantRefund(BrandMerchantRefundParams params) {
    StudioCashierRecord pay = studioCashierRecordRpService.getById(Long.parseLong(params.getPayTransNo()))
        .orElseThrow(() -> new BizException(FinanceErrorEnums.NOT_EXIST));

    BigDecimal historyRefundActualAmount = studioCashierRecordRpService.sumRefundActualAmountByPayTransNo(
        params.getPayTransNo());

    StudioCashierRecord record = PayConvert.convertRefundInit(params, pay, historyRefundActualAmount);
    studioCashierRecordRpService.insert(record);

    RefundPayRequest request = PayConvert.convert(params, record);
    RefundVO vo = payRpcService.refund(request);

    StudioCashierRecord toModify = PayConvert.convertRefund(record, vo);
    studioCashierRecordRpService.updateById(toModify);
    return vo;
  }

  @Override
  public void refundCallback(RefundEvent event) {
    StudioCashierRecord record = studioCashierRecordRpService.getById(Long.parseLong(event.getTransNo()))
        .orElse(null);
    if (null == record) {
      return;
    }

    StudioCashierRecord toModify = PayConvert.convertRefund(record, event);
    studioCashierRecordRpService.updateById(toModify);
  }

  @Override
  public void checkSettle(Long merchantId, Date settleDate) {
    Date endDate = null == settleDate ? TimeUtil.local().startOfDay(new Date()) : settleDate;
    AtomicInteger failCount = new AtomicInteger(0);
    this.pageAndConsumer(merchantId, 200, apply -> {

      // 同步历史处理中的结算记录
      Date startDate = this.syncHistorySettle(apply.getMerchantId(), endDate);

      MerchantSettleRecord exist = merchantSettleRecordRpService.getByMerchantIdAndSettleDate(
          apply.getMerchantId(), endDate);
      if (null != exist) {
        return;
      }

      // 对账
      SettleVO vo = payRpcService.syncSettle(apply.getMerchantId(), endDate);
      BigDecimal totalAmount = studioCashierRecordRpService.sumMerchantPaySuccess(apply.getMerchantId(),
          startDate, endDate);
      log.info("settle vo is {}, local total amount is {}", JSON.toJSONString(vo), totalAmount);
      if (BigDecimal.ZERO.compareTo(vo.getTransAmount()) == 0
          && BigDecimal.ZERO.compareTo(totalAmount) == 0) {
        // 没有交易
        return;
      }

      if (vo.getTransAmount().compareTo(totalAmount) == 0) {
        // 对账成功
        Map<Long, BigDecimal> studioTransAmountMap = studioCashierRecordRpService.sumMerchantPaySuccessGroupByStudioId(
            apply.getMerchantId(), startDate, endDate);
        Map<Long, StudioVO> studioMap = studioRpcService.mapStudio(
            Lists.newArrayList(studioTransAmountMap.keySet()));
        transactionTemplate.executeWithoutResult(action -> {
          MerchantSettleRecord record = PayConvert.convertMerchantSettle(apply, vo);
          merchantSettleRecordRpService.insert(record);
          studioSettleRecordRpService.insertAll(
              PayConvert.convertStudioSettle(record, studioTransAmountMap, studioMap));
        });
        if (TransStateEnums.SUCCESS == TransStateEnums.getByCode(vo.getTransState())) {
          // 结算成功，则更新收银流水状态为记录为入账成功
          this.updatePayIn(apply.getMerchantId(), startDate, endDate);
        }
      } else {
        // 对账失败
        failCount.incrementAndGet();
        studioCheckSettleRecordRpService.insert(PayConvert.convertCheckSettle(apply, vo, totalAmount));
      }
    });

    // 钉钉消息
    if (failCount.get() > 0) {
      DingUtil.sendCheckSettleFail(failCount.get());
    } else {
      DingUtil.sendCheckSettleSuccess();
    }

  }

  @Override
  public void syncSettle(Long merchantId, Date settleDate) {
    Date endDate = null == settleDate ? TimeUtil.local().startOfDay(new Date()) : settleDate;
    this.pageAndConsumer(merchantId, 200,
        apply -> this.syncHistorySettle(apply.getMerchantId(), endDate));
  }

  private Date syncHistorySettle(Long merchantId, Date endDate) {
    Date startDate = TimeUtil.local().plus(endDate, -15, ChronoUnit.DAYS);
    List<MerchantSettleRecord> list = merchantSettleRecordRpService.listInitAndProcessByMerchantIdAndBeforeOrEqualEndDate(
        merchantId, endDate);
    if (CollectionUtil.isEmpty(list)) {
      return startDate;
    }

    for (int i = 0; i < list.size(); i++) {
      MerchantSettleRecord history = list.get(i);
      SettleVO vo = payRpcService.syncSettle(merchantId, history.getSettleDate());
      if (vo == null) {
        if (i == 0) {
          return TimeUtil.local().plus(history.getSettleDate(), -15, ChronoUnit.DAYS);
        } else {
          return list.get(i - 1).getSettleDate();
        }
      }

      TransStateEnums state = TransStateEnums.getByCode(vo.getTransState());
      if (TransStateEnums.INIT == state || TransStateEnums.PROCESS == state) {
        if (i == 0) {
          return TimeUtil.local().plus(history.getSettleDate(), -15, ChronoUnit.DAYS);
        } else {
          return list.get(i - 1).getSettleDate();
        }
      }

      transactionTemplate.executeWithoutResult(action -> {
        merchantSettleRecordRpService.updateById(PayConvert.convertMerchantSettle(history, vo));
        studioSettleRecordRpService.updateByParentId(PayConvert.convertStudioSettle(history, vo));
      });
      if (TransStateEnums.SUCCESS == state) {
        this.updatePayIn(merchantId, null, history.getSettleDate());
      }
      startDate = history.getSettleDate();
    }
    return startDate;
  }

  private void updatePayIn(Long merchantId, Date startTime, Date endTime) {
    Integer current = 0;
    Integer size = 500;

    do {
      Page<StudioCashierRecord> page = studioCashierRecordRpService.pageMerchantPaySuccess(merchantId,
          startTime, endTime, current, size);
      if (CollectionUtil.isEmpty(page.getContent())) {
        break;
      }

      try {
        List<Long> ids = page.getContent().stream().map(StudioCashierRecord::getId).collect(toList());
        studioCashierRecordRpService.updateByIds(
            StudioCashierRecord.builder().transState(BrandCashierTransStateEnum.PAY_IN.getCode())
                .inTime(endTime)
                .updateTime(new Date()).build(), ids);
      } finally {
        current++;
      }

      if (!page.hasNext()) {
        break;
      }
    } while (true);
  }


  public void pageAndConsumer(Long merchantId, Integer batch, Consumer<StudioMerchantApply> consumer) {
    Long minId = 0L;
    log.info("consumer merchant apply start");
    do {
      List<StudioMerchantApply> list = studioMerchantApplyRpService.listByMinId(minId, merchantId, batch);
      if (CollectionUtil.isEmpty(list)) {
        break;
      }
      try {

        list.forEach(apply -> {
          try {
            log.info("start consumer merchant apply {}, {}", apply.getMerchantId(), apply.getMerchantNo());
            consumer.accept(apply);
            log.info("end consumer merchant apply {}, {}", apply.getMerchantId(), apply.getMerchantNo());
          } catch (Exception e) {
            log.error("consumer apply fail apply {}, {}", apply.getMerchantId(), apply.getMerchantNo(), e);
          }
        });

      } finally {
        minId = list.get(list.size() - 1).getId();
      }
    } while (true);
    log.info("consumer merchant apply finished");
  }


  private String getChannelNo(NativePayParams params) {
    switch (params.getChannel()) {
      case WX:
        return config.getWxNative();
      case ALI:
        return config.getAliNative();
      default:
        throw new BizException(FinanceErrorEnums.NOT_SUPPORT_TYPE);
    }
  }

  private PayRequest convertNativePay(NativePayParams params) {
    PayRequest req = new PayRequest();
    req.setChannelNo(getChannelNo(params));
    req.setTransNo(IdWorker.getIdStr());
    req.setAmount(params.getAmount());
    req.setPayType(PayTypeEnums.NATIVE);
    req.setGoods(params.getGoods());
    req.setOrderNo(params.getOrderNo());
    req.setOrderType(params.getOrderType().getCode());
    req.setTimeExpire(params.getTimeExpire());
    req.setTradingTime(new Date());
    req.setExtra(params.getExtra());
    return req;
  }

  private void sendPaySuccessMessage(PayEvent e) {
    Long studioId = e.getExtra().getLong("studioId");
    String buyerName = e.getExtra().getString("buyerName");

    // 有乐动收银权限的账户
    Set<Long> targetIds = permissionRpcService.getUserIdByCode(AuthDomainEnum.FITNESS_ADMIN.getCode(),
        TenantTypeEnum.STUDIO.buildKey(studioId), "FitXmPay", false);
    if (CollectionUtil.isEmpty(targetIds)) {
      return;
    }

    List<Map<String, Object>> paramList = targetIds.stream().map(targetId -> {
      Map<String, Object> paramMap = new HashMap<>(1);
      paramMap.put("targetId", targetId);
      paramMap.put("studioId", studioId);
      paramMap.put("buyerName", buyerName);
      paramMap.put("amount", e.getAmount());
      return paramMap;
    }).collect(Collectors.toList());

    SendCommonMsgEvent event = new SendCommonMsgEvent();
    event.setChannelEnums(Lists.newArrayList(MsgChannelEnum.APP_PUSH));
    event.setCovertTarget(true);
    event.setEventId(IdWorker.getId());
    event.setSourceId(studioId);
    event.setBizType("CASHIER_PAY_SUCCESS");
    event.setParams(paramList);
    EventAgent.of(SendCommonMsgEvent.class).triggerEvent(event);
  }

  private boolean isPaying(BrandCashierTransStateEnum state) {
    return BrandCashierTransStateEnum.PAYING == state || BrandCashierTransStateEnum.PAY_INIT == state;
  }

  private boolean isPayFinished(BrandCashierTransStateEnum state) {
    return isPaySuccess(state) || isPayFail(state);
  }

  private boolean isPayFail(BrandCashierTransStateEnum state) {
    return BrandCashierTransStateEnum.PAY_FAIL == state;
  }

  private boolean isPaySuccess(BrandCashierTransStateEnum state) {
    return BrandCashierTransStateEnum.PAY_SUCCESS == state || BrandCashierTransStateEnum.PAY_IN == state;
  }

  private Pair<PartyToMerchant, StudioVO> checkBeforeMerchantPay(StudioMerchantPayParams params) {
    if (params.getTransAmount().compareTo(BigDecimal.ZERO) <= 0) {
      throw new BizException(FinanceErrorEnums.PAY_FAIL_8);
    }
    if (params.getTransAmount().compareTo(MAX_AMOUNT) > 0) {
      throw new BizException(FinanceErrorEnums.PAY_FAIL_9);
    }
    StudioVO studio = studioRpcService.getStudio(params.getStudioId());
    PartyToMerchant relation = partyToMerchantRpService.getByStudioId(studio.getId(), config.getCashier());
    if (null == relation) {
      throw new BizException(FinanceErrorEnums.PAY_FAIL_1);
    }
    return Pair.of(relation, studio);
  }

  private FinanceErrorEnums replaceFailMessage(String failMsg) {
    return PayFailMessageReplaceEnums.convertBySource(failMsg);
  }

  @Data
  private static class AliInfo {

    private String appId;

    private String baseUrl;

    private String alipayPublicKey;

    private String merchantPrivateKey;

  }

}
