package com.jiejing.fitness.finance.service.utils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Objects;

/**
 * @author chengyubing
 * @since 2024/2/27 14:16
 */
public class FeeUtil {

  /**
   * 【支付】计算支付手续费，保留小数点后两位，四舍五入
   *
   * @param feeRate 费率（%）
   * @param amount  支付金额
   * @return 手续费
   */
  public static BigDecimal calPayFee(BigDecimal feeRate, BigDecimal amount) {
    return MoneyUtil.defaultRound(MoneyUtil.multiply(
        MoneyUtil.divide(feeRate, new BigDecimal("100"), 4, RoundingMode.HALF_UP.ordinal()),
        amount));
  }

  /**
   * 【退款】计算退款手续费
   *
   * <p>实退手续费计算公式：退款手续费=向下取整(退款金额*原交易手续费金额/原交易金额)</p>
   *
   * @param refundTransAmount         退款申请金额
   * @param payTransAmount            原支付交易金额
   * @param payActualAmount           用户实收金额
   * @param payFee                    平台实收手续费
   * @param historyRefundActualAmount 历史实退金额
   * @return 本次退款应退手续费
   */
  public static BigDecimal calculateRefundFee(BigDecimal refundTransAmount, BigDecimal payTransAmount,
      BigDecimal payActualAmount, BigDecimal payFee, BigDecimal historyRefundActualAmount) {
    // 机构剩余实收金额
    BigDecimal leftPayActualAmount = MoneyUtil.subtract(payActualAmount, historyRefundActualAmount);
    // 试算手续费
    BigDecimal trialFee = MoneyUtil.divide(refundTransAmount.multiply(payFee), payTransAmount, 0,
        RoundingMode.FLOOR);
    // 试算实退金额
    BigDecimal trialActualAmount = MoneyUtil.subtract(refundTransAmount, trialFee);
    // 实退手续费
    return leftPayActualAmount.compareTo(trialActualAmount) >= 0 ? trialFee
        : MoneyUtil.add(trialFee, MoneyUtil.subtract(trialActualAmount, leftPayActualAmount));
  }

  /**
   * 计算补贴回退金额（向上取整ceil）
   *
   * @param refundTransAmount 退款申请金额
   * @param payTransAmount    原支付交易金额
   * @param subsidyAmount     补贴金额
   * @param historyBackAmount 历史补贴回退金额
   * @return 补贴回退金额
   */
  public static BigDecimal calculateSubsidyBackAmount(BigDecimal refundTransAmount,
      BigDecimal payTransAmount, BigDecimal subsidyAmount, BigDecimal historyBackAmount) {
    if (Objects.isNull(subsidyAmount)) {
      return BigDecimal.ZERO;
    }
    // 剩余可回退补贴金额
    BigDecimal leftSubsidyBackAmount = MoneyUtil.subtract(subsidyAmount, historyBackAmount);
    // 试算补贴回退金额
    BigDecimal trialAmount = MoneyUtil.divide(refundTransAmount.multiply(subsidyAmount),
        payTransAmount, 0, RoundingMode.CEILING);
    return leftSubsidyBackAmount.compareTo(trialAmount) >= 0 ? trialAmount : leftSubsidyBackAmount;
  }

  public static BigDecimal calculateAdvanceAmount(BigDecimal refundAmount, BigDecimal qyqbhTransitBalance,
      BigDecimal yxyjhTransitBalance) {
    BigDecimal transitBalance = MoneyUtil.subtract(qyqbhTransitBalance, yxyjhTransitBalance);
    if (refundAmount.compareTo(transitBalance) <= 0) {
      // 若：退款金额 <= 业务钱包入账中余额，则无需佣金垫付记录记账
      return null;
    }

    // 若：退款金额 > 业务钱包入账中余额，则添加佣金垫付记录记账
    // 垫付金额 = (退款金额 - 业务入帐中金额) > 佣金入账中 ? 佣金入账中 : (退款金额 - 入帐中金额)
    BigDecimal advanceAmount = MoneyUtil.subtract(refundAmount, transitBalance);
    if (advanceAmount.compareTo(yxyjhTransitBalance) > 0) {
      advanceAmount = yxyjhTransitBalance;
    }

    return advanceAmount;
  }

  public static void main(String[] args) {
    // 1. 首次退款
//    BigDecimal refundTransAmount = new BigDecimal("498");
//    BigDecimal payTransAmount = new BigDecimal("500");
//    BigDecimal payActualAmount = new BigDecimal("498");
//    BigDecimal payFee = new BigDecimal("2");
//    BigDecimal historyRefundActualAmount = new BigDecimal("0");
    // 预计输出：1
    // 实退金额：497, 实退手续费：1。剩余可退金额：2（剩余实退金额：1，剩余实退手续费：1），历史实退金额：497，历史实退手续费：1

    // 2. 二次退款
//    BigDecimal refundTransAmount = new BigDecimal("1");
//    BigDecimal payTransAmount = new BigDecimal("500");
//    BigDecimal payActualAmount = new BigDecimal("498");
//    BigDecimal payFee = new BigDecimal("2");
//    BigDecimal historyRefundActualAmount = new BigDecimal("497");
    // 预计输出：0
    // 实退金额：1，实退手续费：0。剩余可退金额：1（剩余实退金额：0，剩余实退手续费：1），历史实退金额：498，历史实退手续费：1

    // 3. 三次退款
    BigDecimal refundTransAmount = new BigDecimal("40");
    BigDecimal payTransAmount = new BigDecimal("10000");
    BigDecimal payActualAmount = new BigDecimal("9943");
    BigDecimal payFee = new BigDecimal("57");
    BigDecimal historyRefundActualAmount = new BigDecimal("9904");
    // 预计输出：1
    // 实退金额：0，实退手续费：1。剩余可退金额：0，历史实退金额：498，历史实退手续费：2
    System.out.println(
        FeeUtil.calculateRefundFee(refundTransAmount, payTransAmount, payActualAmount, payFee,
            historyRefundActualAmount));

  }

}
