package com.jiejing.fitness.finance.service.pay.impl;

import static java.util.stream.Collectors.toList;

import com.alibaba.fastjson.JSON;
import com.alipay.api.AlipayApiException;
import com.alipay.api.DefaultAlipayClient;
import com.alipay.api.response.AlipaySystemOauthTokenResponse;
import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.google.common.collect.Lists;
import com.jiejing.common.exception.BizException;
import com.jiejing.common.utils.collection.CollectionUtil;
import com.jiejing.fitness.enums.auth.AuthDomainEnum;
import com.jiejing.fitness.enums.finance.BrandCashierTransStateEnum;
import com.jiejing.fitness.enums.tenant.TenantTypeEnum;
import com.jiejing.fitness.finance.repository.entity.GlobalConfig;
import com.jiejing.fitness.finance.repository.entity.StudioCashierRecord;
import com.jiejing.fitness.finance.repository.entity.PartyToMerchant;
import com.jiejing.fitness.finance.repository.service.GlobalConfigRpService;
import com.jiejing.fitness.finance.repository.service.StudioCashierRecordRpService;
import com.jiejing.fitness.finance.repository.service.PartyToMerchantRpService;
import com.jiejing.fitness.finance.service.config.AppUrlProperties;
import com.jiejing.fitness.finance.service.config.PayChannelProperties;
import com.jiejing.fitness.finance.service.enums.FinanceErrorEnums;
import com.jiejing.fitness.finance.service.enums.GlobalConfigEnums;
import com.jiejing.fitness.finance.service.pay.PayService;
import com.jiejing.fitness.finance.service.pay.convert.PayConvert;
import com.jiejing.fitness.finance.service.pay.params.AppPayParams;
import com.jiejing.fitness.finance.service.pay.params.NativePayParams;
import com.jiejing.fitness.finance.service.pay.params.StudioMerchantPayParams;
import com.jiejing.fitness.finance.service.rpc.MerchantRpcService;
import com.jiejing.fitness.finance.service.rpc.PayRpcService;
import com.jiejing.fitness.finance.service.rpc.PermissionRpcService;
import com.jiejing.fitness.finance.service.rpc.StudioRpcService;
import com.jiejing.fitness.finance.service.utils.FeeUtil;
import com.jiejing.fitness.finance.service.utils.MoneyUtil;
import com.jiejing.message.enums.MsgChannelEnum;
import com.jiejing.message.event.SendCommonMsgEvent;
import com.jiejing.paycenter.common.enums.common.PayChannelEnums;
import com.jiejing.paycenter.common.enums.pay.PayStateEnums;
import com.jiejing.paycenter.common.model.vo.MerchantVO;
import com.jiejing.paycenter.api.pay.request.PayRequest;
import com.jiejing.paycenter.common.model.vo.PayVO;
import com.jiejing.paycenter.common.event.PayEvent;
import com.jiejing.studio.api.studio.vo.StudioVO;
import com.jiejing.wechat.WeChatAuthService;
import com.jiejing.wechat.vo.weChat.BaseAuthInfoVO;
import com.xiaomai.event.EventAgent;
import java.math.BigDecimal;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;
import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.stereotype.Service;
import org.springframework.transaction.support.TransactionTemplate;

/**
 * @author chengyubing
 * @since 2024/2/27 11:06
 */
@Slf4j
@Service
public class PayServiceImpl implements PayService {

  private static final BigDecimal MAX_AMOUNT = new BigDecimal("10000000");

  @Resource
  private PayChannelProperties config;

  @Resource
  private AppUrlProperties appUrlProperties;

  @Resource
  private PartyToMerchantRpService partyToMerchantRpService;

  @Resource
  private PayRpcService payRpcService;

  @Resource
  private StudioRpcService studioRpcService;

  @Resource
  private StudioCashierRecordRpService studioCashierRecordRpService;

  @Resource
  private MerchantRpcService merchantRpcService;

  @Resource
  private WeChatAuthService weChatAuthService;

  @Resource
  private GlobalConfigRpService globalConfigRpService;

  @Resource
  private TransactionTemplate transactionTemplate;

  @Resource
  private PermissionRpcService permissionRpcService;

  @Resource(name = "financeThreadPool")
  private Executor executor;

  private final Map<String, DefaultAlipayClient> aliClientMap = new HashMap<>();

  @PostConstruct
  public void init() {
    GlobalConfig config = globalConfigRpService.getById(GlobalConfigEnums.CASHIER_ALI_INFO.getCode())
        .orElse(new GlobalConfig());
    AliInfo info = JSON.parseObject(config.getConfigValue(), AliInfo.class);
    aliClientMap.put(info.getAppId(),
        new DefaultAlipayClient(info.getBaseUrl(), info.getAppId(), info.getMerchantPrivateKey(), "json",
            "utf-8", info.getAlipayPublicKey(), "RSA2"));
  }

  @Override
  public String getOpenId(PayChannelEnums channel, String appId, String authCode) {
    switch (channel) {
      case WX:
        return Optional.ofNullable(weChatAuthService.getBaseAuthInfo(null, appId, authCode))
            .map(BaseAuthInfoVO::getOpenId).orElse(null);
      case ALI:
        try {
          AlipaySystemOauthTokenResponse response = aliClientMap.get(appId)
              .execute(PayConvert.convertAlipaySystemOauthTokenRequest(authCode));
          log.info("get ali openId response {}, {}, {}", appId, authCode, JSON.toJSONString(response));
          return response.getOpenId();
        } catch (AlipayApiException e) {
          throw new BizException(e.getErrCode(), e.getMessage());
        }
      default:
        throw new BizException(FinanceErrorEnums.NOT_SUPPORT_TYPE);
    }
  }

  @Override
  public PayVO nativePay(NativePayParams params) {
    return payRpcService.pay(PayConvert.convertNativePay(params, config));
  }

  @Override
  public PayVO appPay(AppPayParams params) {
    return payRpcService.pay(PayConvert.convertAppPay(params, config));
  }

  @Override
  public PayVO merchantPay(StudioMerchantPayParams params) {

    Pair<PartyToMerchant, StudioVO> pair = this.checkBeforeMerchantPay(params);

    MerchantVO merchant = merchantRpcService.getByMerchantId(pair.getLeft().getMerchantId());
    StudioCashierRecord record = PayConvert.convertPayInit(params, pair.getRight(), merchant);
    PayRequest request = PayConvert.convert(params, record);
    record.setFeeRate(payRpcService.getFeeRate(request));
    record.setFee(FeeUtil.calPayFee(record.getFeeRate(), params.getTransAmount()));
    record.setActualAmount(MoneyUtil.subtract(params.getTransAmount(), record.getFee()));
    studioCashierRecordRpService.insert(record);
    PayVO vo = payRpcService.pay(request);
    this.payCallback(PayConvert.convertEvent(request, vo));

    if (PayStateEnums.FAILED == PayStateEnums.getByCode(vo.getPayState())) {
      throw new BizException(PayConvert.replaceFailMessage(vo.getFailMsg()));
    }
    return vo;
  }

  @Override
  public void payCallback(PayEvent event) {
    if (!event.getChannelNo().equals(config.getCashier())) {
      return;
    }

    Boolean result = transactionTemplate.execute(action -> {
      Long id = Long.parseLong(event.getTransNo());
      StudioCashierRecord record = studioCashierRecordRpService.getByIdForUpdate(id).orElse(null);
      if (null == record) {
        return false;
      }

      BrandCashierTransStateEnum originalState = BrandCashierTransStateEnum.valueOf(record.getTransState());
      BrandCashierTransStateEnum targetState = PayConvert.convertTransState(event.getPayState());
      if (targetState == originalState) {
        return false;
      }

      if (isPaying(targetState) && isPayFinished(originalState)) {
        return false;
      }

      StudioCashierRecord toModify = PayConvert.convertPay(record, event);
      studioCashierRecordRpService.updateById(toModify);
      return true;
    });

    if (null != result && result) {
      if (PayStateEnums.SUCCESS.getCode().equals(event.getPayState())) {
        executor.execute(() -> this.sendPaySuccessMessage(event));
      }
    }

  }

  private List<Long> getAdminIds(Long studioId) {
    List<Long> userIds = this.getUserIds(studioId);
    if (CollectionUtil.isEmpty(userIds)) {
      return Lists.newArrayList();
    }
    return studioRpcService.listAdminIdsByUserIds(studioId, Lists.newArrayList(userIds));
  }

  private List<Long> getUserIds(Long studioId) {
    // 有乐动收银查看和操作权限的账户
    Set<Long> seeUserIds = permissionRpcService.getUserIdByCode(AuthDomainEnum.FITNESS_ADMIN.getCode(),
        TenantTypeEnum.STUDIO.buildKey(studioId), "FitSeeXmPay", false);
    Set<Long> operatorUserIds = permissionRpcService.getUserIdByCode(AuthDomainEnum.FITNESS_ADMIN.getCode(),
        TenantTypeEnum.STUDIO.buildKey(studioId), "FitManageXmPay", false);
    return seeUserIds.stream().filter(operatorUserIds::contains).collect(toList());
  }


  private void sendPaySuccessMessage(PayEvent e) {
    Long studioId = e.getExtra().getLong("studioId");
    String buyerName = e.getExtra().getString("buyerName");

    List<Long> targetIds = this.getAdminIds(studioId);
    if (CollectionUtil.isEmpty(targetIds)) {
      return;
    }

    List<Map<String, Object>> paramList = targetIds.stream().map(targetId -> {
      Map<String, Object> paramMap = new HashMap<>(1);
      paramMap.put("targetId", targetId);
      paramMap.put("studioId", studioId);
      paramMap.put("buyerName", buyerName);
      paramMap.put("amount", e.getAmount());
      paramMap.put("appUrl", appUrlProperties.getTransDetail() + e.getTransNo());
      return paramMap;
    }).collect(Collectors.toList());

    SendCommonMsgEvent event = new SendCommonMsgEvent();
    event.setChannelEnums(Lists.newArrayList(MsgChannelEnum.APP_PUSH));
    event.setCovertTarget(true);
    event.setEventId(IdWorker.getId());
    event.setSourceId(studioId);
    event.setBizType("CASHIER_PAY_SUCCESS");
    event.setParams(paramList);
    EventAgent.of(SendCommonMsgEvent.class).triggerEvent(event);
  }

  private boolean isPaying(BrandCashierTransStateEnum state) {
    return BrandCashierTransStateEnum.PAYING == state || BrandCashierTransStateEnum.PAY_INIT == state;
  }

  private boolean isPayFinished(BrandCashierTransStateEnum state) {
    return isPaySuccess(state) || isPayFail(state);
  }

  private boolean isPayFail(BrandCashierTransStateEnum state) {
    return BrandCashierTransStateEnum.PAY_FAIL == state;
  }

  private boolean isPaySuccess(BrandCashierTransStateEnum state) {
    return BrandCashierTransStateEnum.PAY_SUCCESS == state || BrandCashierTransStateEnum.PAY_IN == state;
  }

  private Pair<PartyToMerchant, StudioVO> checkBeforeMerchantPay(StudioMerchantPayParams params) {
    if (params.getTransAmount().compareTo(BigDecimal.ZERO) <= 0) {
      throw new BizException(FinanceErrorEnums.PAY_FAIL_8);
    }
    if (params.getTransAmount().compareTo(MAX_AMOUNT) > 0) {
      throw new BizException(FinanceErrorEnums.PAY_FAIL_9);
    }
    StudioVO studio = studioRpcService.getStudio(params.getStudioId());
    PartyToMerchant relation = partyToMerchantRpService.getByStudioId(studio.getId(), config.getCashier());
    if (null == relation) {
      throw new BizException(FinanceErrorEnums.PAY_FAIL_10);
    }
    return Pair.of(relation, studio);
  }


  @Data
  private static class AliInfo {

    private String appId;

    private String baseUrl;

    private String alipayPublicKey;

    private String merchantPrivateKey;

  }

}
