package com.jiejing.fitness.finance.service.event;

import static org.springframework.integration.IntegrationMessageHeaderAccessor.DELIVERY_ATTEMPT;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.jiejing.fitness.enums.delay.DelayTaskCodeEnum;
import com.jiejing.fitness.enums.finance.PartyTypeEnum;
import com.jiejing.fitness.finance.service.axf.AxfService;
import com.jiejing.fitness.finance.service.enums.MerchantBizCodeEnums;
import com.jiejing.fitness.finance.service.merchant.StudioMerchantService;
import com.jiejing.fitness.finance.service.pay.PayService;
import com.jiejing.fitness.finance.service.pay.RefundService;
import com.jiejing.paycenter.common.event.CommodityOrderEvent;
import com.jiejing.paycenter.common.event.MerchantEvent;
import com.jiejing.paycenter.common.event.MerchantSubChannelEvent;
import com.jiejing.paycenter.common.event.OrderDeductionEvent;
import com.jiejing.paycenter.common.event.PayEvent;
import com.jiejing.paycenter.common.event.RefundEvent;
import com.jiejing.paycenter.common.event.StoreEvent;
import com.jiejing.workflow.event.DelayTaskEvent;
import com.xiaomai.event.annotation.EventHandler;
import java.util.ArrayList;
import java.util.Optional;
import javax.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.messaging.handler.annotation.Header;
import org.springframework.stereotype.Service;

/**
 * @author chengyubing
 * @since 2024/2/21 14:30
 */
@Slf4j
@Service
public class ListenerService {

  private static final int MAX_RETRY = 3;

  @Resource
  private StudioMerchantService studioMerchantService;

  @Resource
  private PayService payService;

  @Resource
  private RefundService refundService;

  @Resource
  private AxfService axfService;

  @EventHandler(value = MerchantEvent.class, binder = "biz-kafka", maxAttempts = MAX_RETRY)
  public void handleMerchantEvent(MerchantEvent event, @Header(DELIVERY_ATTEMPT) int retryNum) {
    try {
      log.info("start process merchant event {}", JSON.toJSONString(event));
      event.setSubChannels(Optional.ofNullable(event.getSubChannels()).orElse(new ArrayList<>(1)));
      event.setSubChannelAuths(Optional.ofNullable(event.getSubChannelAuths()).orElse(new ArrayList<>(1)));

      MerchantBizCodeEnums type = MerchantBizCodeEnums.getByCode(event.getBizCode());
      switch (type) {
        case STUDIO:
          studioMerchantService.callback(event);
          break;
        case AXF:
          axfService.callback(event);
          break;
        default:
          break;
      }
    } catch (Exception e) {
      log.error("merchant event callback process fail {}", JSON.toJSONString(event), e);
    }
  }


  @EventHandler(value = PayEvent.class, binder = "biz-kafka", maxAttempts = MAX_RETRY)
  public void payEventCallback(PayEvent event, @Header(DELIVERY_ATTEMPT) int retryNum) {
    try {
      log.info("start process pay event {}", JSON.toJSONString(event));
      payService.payCallback(event);
    } catch (Exception e) {
      log.info("process pay event fail {}", event.getTransNo(), e);
    }
  }

  @EventHandler(value = RefundEvent.class, binder = "biz-kafka", maxAttempts = MAX_RETRY)
  public void refundEventCallback(RefundEvent event, @Header(DELIVERY_ATTEMPT) int retryNum) {
    try {
      log.info("start process refund event {}", JSON.toJSONString(event));
      refundService.refundCallback(event);
    } catch (Exception e) {
      log.info("process refund event fail {}", event.getTransNo(), e);
    }
  }

  @EventHandler(value = MerchantSubChannelEvent.class, binder = "biz-kafka", maxAttempts = MAX_RETRY)
  public void merchantSubChannelEventCallback(MerchantSubChannelEvent event,
      @Header(DELIVERY_ATTEMPT) int retryNum) {
    try {
      log.info("start process merchantSubChannelEventCallback event {}", JSON.toJSONString(event));
      studioMerchantService.merchantSubChannelEventCallback(event);
    } catch (Exception e) {
      log.info("process merchantSubChannelEventCallback event fail {}", event.getId(), e);
    }
  }

  @EventHandler(value = DelayTaskEvent.class, binder = "biz-kafka", maxAttempts = MAX_RETRY)
  public void delayTaskEventCallback(DelayTaskEvent event, @Header(DELIVERY_ATTEMPT) int retryNum) {
    try {
      log.info("start process delayTaskEventCallback {}", JSON.toJSONString(event));

      switch (DelayTaskCodeEnum.getByCode(event.getBizCode())) {
        case EMBEDDED_XCX:
          studioMerchantService.syncEmbeddedXcx(event.getId());
          break;
        case CANCEL_AXF_ORDER_FAIL:
          this.processCancelAxfOrderFail(event);
          break;
        default:
          return;
      }
    } catch (Exception e) {
      log.info("process delayTaskEventCallback fail {}", event, e);
    }
  }

  private void processCancelAxfOrderFail(DelayTaskEvent event) {
    Long id = event.getBizId();
    JSONObject body = JSON.parseObject(event.getTaskBody());
    axfService.cancelFail(id, body.getLong("opId"), body.getDate("cancelTime"));
  }

  @EventHandler(value = StoreEvent.class, binder = "biz-kafka", maxAttempts = 5)
  public void storeEvent(StoreEvent event, @Header(DELIVERY_ATTEMPT) int retryNum) {
    try {
      log.info("start process store event {}", JSON.toJSONString(event));
      axfService.storeCallback(event);
    } catch (Exception e) {
      log.error("store event callback process fail {}", JSON.toJSONString(event), e);
    }
  }

  @EventHandler(value = CommodityOrderEvent.class, binder = "biz-kafka", maxAttempts = 5)
  public void commodityOrderEvent(CommodityOrderEvent event) {
    try {
      log.info("start process CommodityOrderEvent {}", JSON.toJSONString(event));
      axfService.orderCallback(event);
    } catch (Exception e) {
      log.info("process CommodityOrderEvent fail {}", event.getOrderNo(), e);
    }
  }

  @EventHandler(value = OrderDeductionEvent.class, binder = "biz-kafka", maxAttempts = 5)
  public void orderDeductionEvent(OrderDeductionEvent event) {
    try {
      log.info("start process OrderDeductionEvent {}", JSON.toJSONString(event));
      axfService.orderDeductionCallback(event);
    } catch (Exception e) {
      log.info("process OrderDeductionEvent fail {}, {}", event.getOrderNo(), event.getThirdTransNo(), e);
    }
  }


}
