package com.jiejing.fitness.finance.service.merchant.convert;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.jiejing.common.utils.convert.BeanUtil;
import com.jiejing.common.utils.crypt.AesUtil;
import com.jiejing.filecenter.api.resource.vo.ResourceInfoVO;
import com.jiejing.fitness.finance.api.enums.ApplyTypeEnums;
import com.jiejing.fitness.finance.api.enums.MerchantTypeEnums;
import com.jiejing.fitness.finance.repository.entity.BrandMerchant;
import com.jiejing.fitness.finance.repository.entity.BrandMerchantApply;
import com.jiejing.fitness.finance.service.merchant.params.ApplyBrandMerchantParams;
import com.jiejing.paycenter.api.merchant.request.ApplyMerchantRequest;
import com.jiejing.paycenter.api.merchant.request.OpenSingleSubChannelRequest;
import com.jiejing.paycenter.api.merchant.request.UploadRequest;
import com.jiejing.paycenter.common.enums.common.OpenStateEnums;
import com.jiejing.paycenter.common.enums.merchant.BusinessTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.LicenseTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.MccEnums;
import com.jiejing.paycenter.common.enums.merchant.PaySceneEnums;
import com.jiejing.paycenter.common.enums.merchant.ResourceTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SceneTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelAuthTypeEnums;
import com.jiejing.paycenter.common.event.MerchantEvent;
import com.jiejing.paycenter.common.model.Address;
import com.jiejing.paycenter.common.model.BankCard;
import com.jiejing.paycenter.common.model.BusinessInfo;
import com.jiejing.paycenter.common.model.Contact;
import com.jiejing.paycenter.common.model.Legal;
import com.jiejing.paycenter.common.model.License;
import com.jiejing.paycenter.common.model.SubChannelAuthInfo;
import com.jiejing.paycenter.common.model.SubChannelInfo;
import com.jiejing.studio.api.studio.vo.StudioVO;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @author chengyubing
 * @since 2024/2/20 16:46
 */
public class MerchantConvert {

  private final static BigDecimal ALI_OFFLINE_FEE_RATE = new BigDecimal("0.38");

  private final static BigDecimal WX_OFFLINE_FEE_RATE = new BigDecimal("0.38");

  private final static List<OpenSingleSubChannelRequest> DEFAULT_SUB_CHANNELS = Lists.newArrayList(
      OpenSingleSubChannelRequest.builder().scene(PaySceneEnums.ALI_OFFLINE).feeRate(ALI_OFFLINE_FEE_RATE)
          .build(),
      OpenSingleSubChannelRequest.builder().scene(PaySceneEnums.WX_GZH_OFFLINE).feeRate(WX_OFFLINE_FEE_RATE)
          .build(),
      OpenSingleSubChannelRequest.builder().scene(PaySceneEnums.WX_XCX_OFFLINE).feeRate(WX_OFFLINE_FEE_RATE)
          .build()
  );

  public static ApplyMerchantRequest convertRequest(String channelNo, String applyNo,
      ApplyBrandMerchantParams params) {
    return ApplyMerchantRequest.builder()
        .bizCode(MerchantTypeEnums.BRAND.getCode())
        .applyNo(applyNo)
        .channelNo(channelNo)
        .companyType(params.getCompanyType())
        .merchantName(params.getMerchantName())
        .shortName(params.getShortName())
        .legal(convertLegal(params))
        .bankCard(convertBankCard(params))
        .license(convertLicense(params))
        .business(convertBusiness(params))
        .contact(convertContact(params))
        .subChannels(DEFAULT_SUB_CHANNELS)
        .resourceMap(params.getResourceInfo())
        .build();
  }


  private static Contact convertContact(ApplyBrandMerchantParams params) {
    return Contact.builder()
        .contactName(params.getLegalName())
        .contactPhone(params.getLegalPhone())
        .contactEmail(params.getContactEmail())
        .servicePhone(params.getLegalPhone())
        .build();
  }

  private static BusinessInfo convertBusiness(ApplyBrandMerchantParams params) {
    return BusinessInfo.builder()
        .mcc(MccEnums.MCC_7941)
        .businessType(BusinessTypeEnums.REALITY)
        .sceneType(SceneTypeEnums.OFFLINE)
        .businessAddress(Address.builder()
            .province(params.getBusinessProvince())
            .city(params.getBusinessCity())
            .district(params.getBusinessDistrict())
            .address(params.getBusinessAddress())
            .build())
        .build();
  }

  private static License convertLicense(ApplyBrandMerchantParams params) {
    return License.builder()
        .licenseCode(params.getLicenseCode())
        .licenseType(LicenseTypeEnums.NATIONAL_LEGAL_MERGE)
        .licenseExpireType(params.getLicenseExpireType())
        .licenseStartDate(params.getLicenseStartDate())
        .licenseEndDate(params.getLicenseEndDate())
        .businessScope(params.getLicenseBusinessScope())
        .foundDate(params.getLicenseFoundDate())
        .registeredCapital("10000")
        .licenseAddress(Address.builder()
            .province(params.getLicenseProvince())
            .city(params.getLicenseCity())
            .district(params.getLicenseDistrict())
            .address(params.getLicenseAddress())
            .build())
        .build();
  }

  private static BankCard convertBankCard(ApplyBrandMerchantParams params) {
    return BankCard.builder()
        .cardType(params.getBankCardType())
        .cardNo(params.getBankCardNo())
        .cardName(params.getBankCardName())
        .phone(params.getBankPhone())
        .certNo(params.getBankCertNo())
        .certType(params.getBankCertType())
        .baseDepositNo(params.getBankBaseDepositNo())
        .certExpireType(params.getBankCertExpireType())
        .certStartDate(params.getBankCertStartDate())
        .certEndDate(params.getBankCertEndDate())
        .bankCode(params.getBankCode())
        .bankName(params.getBankName())
        .branchName(params.getBankBranchName())
        .branchCode(params.getBankBranchCode())
        .cardAddress(Address.builder()
            .province(params.getBankCardProvince())
            .city(params.getBankCardCity())
            .district(params.getBankCardDistrict())
            .address(params.getBankCardAddress())
            .build())
        .build();
  }

  private static Legal convertLegal(ApplyBrandMerchantParams params) {
    return Legal.builder()
        .legalName(params.getLegalName())
        .legalPhone(params.getLegalPhone())
        .legalCertNo(params.getLegalCertNo())
        .legalCertType(params.getLegalCertType())
        .legalCertExpireType(params.getLegalCertExpireType())
        .legalCertStartDate(params.getLegalCertStartDate())
        .legalCertEndDate(params.getLegalCertEndDate())
        .legalAddress(Address.builder()
            .province(params.getLegalProvince())
            .city(params.getLegalCity())
            .district(params.getLegalDistrict())
            .address(params.getLegalAddress())
            .build())
        .build();
  }

  public static BrandMerchantApply convertApply(Long id, ApplyBrandMerchantParams params, StudioVO studio,
      BrandMerchantApply exist, String channel) {
    String salt = AesUtil.getSalt(8);
    ApplyBrandMerchantParams encrypt = params.encrypt(salt);
    return BrandMerchantApply.builder()
        .id(id)
        .applyNo(Long.toString(id))
        .brandId(studio.getBrandId())
        .studioId(encrypt.getStudioId())
        .applyType(exist == null ? ApplyTypeEnums.OPEN : ApplyTypeEnums.RE_OPEN)
        .channelNo(channel)
        .merchantName(encrypt.getMerchantName())
        .shortName(encrypt.getShortName())
        .companyType(encrypt.getCompanyType())
        .openState(OpenStateEnums.INIT)
        .businessProvince(encrypt.getBusinessProvince())
        .businessCity(encrypt.getBusinessCity())
        .businessDistrict(encrypt.getBusinessDistrict())
        .businessAddress(encrypt.getBusinessAddress())
        .licenseCode(encrypt.getLicenseCode())
        .licenseExpireType(encrypt.getLicenseExpireType())
        .licenseStartDate(encrypt.getLicenseStartDate())
        .licenseEndDate(encrypt.getLicenseEndDate())
        .licenseBusinessScope(encrypt.getLicenseBusinessScope())
        .licenseFoundDate(encrypt.getLicenseFoundDate())
        .licenseProvince(encrypt.getLicenseProvince())
        .licenseCity(encrypt.getLicenseCity())
        .licenseDistrict(encrypt.getLicenseDistrict())
        .licenseAddress(encrypt.getLicenseAddress())
        .legalName(encrypt.getLegalName())
        .legalCertNo(encrypt.getLegalCertNo())
        .legalCertType(encrypt.getLegalCertType())
        .legalCertExpireType(encrypt.getLegalCertExpireType())
        .legalCertStartDate(encrypt.getLegalCertStartDate())
        .legalCertEndDate(encrypt.getLegalCertEndDate())
        .legalPhone(encrypt.getLegalPhone())
        .legalProvince(encrypt.getLegalProvince())
        .legalCity(encrypt.getLegalCity())
        .legalDistrict(encrypt.getLegalDistrict())
        .legalAddress(encrypt.getLegalAddress())
        .contactEmail(encrypt.getContactEmail())
        .bankCardType(encrypt.getBankCardType())
        .bankCardNo(encrypt.getBankCardNo())
        .bankCardName(encrypt.getBankCardName())
        .bankPhone(encrypt.getBankPhone())
        .bankCertNo(encrypt.getBankCertNo())
        .bankCertType(encrypt.getBankCertType())
        .bankCertExpireType(encrypt.getBankCertExpireType())
        .bankCertStartDate(encrypt.getBankCertStartDate())
        .bankCertEndDate(encrypt.getBankCertEndDate())
        .bankBaseDepositNo(encrypt.getBankBaseDepositNo())
        .bankCode(encrypt.getBankCode())
        .bankName(encrypt.getBankName())
        .bankBranchName(encrypt.getBankBranchName())
        .bankBranchCode(encrypt.getBankBranchCode())
        .bankCardProvince(encrypt.getBankCardProvince())
        .bankCardCity(encrypt.getBankCardCity())
        .bankCardDistrict(encrypt.getBankCardDistrict())
        .bankCardAddress(encrypt.getBankCardAddress())
        .resourceInfo(JSON.toJSONString(encrypt.getResourceInfo()))
        .salt(salt)
        .createTime(new Date())
        .updateTime(new Date())
        .build();
  }

  public static UploadRequest convertUploadRequest(String channel, ResourceTypeEnums type,
      ResourceInfoVO vo) {
    return UploadRequest.builder()
        .url(vo.getUrl())
        .fileSize(vo.getFileSize())
        .suffix(vo.getSuffix())
        .channelNo(channel)
        .resourceType(type)
        .build();
  }

  public static BrandMerchantApply convertApply(BrandMerchantApply apply, MerchantEvent event) {
    // 品牌商户规则：微信线下公众号、微信线下小程序、支付宝线下通道全部开通成功才算成功
    Map<PaySceneEnums, SubChannelInfo> channelMap = convertSubChannelMap(event.getSubChannels());
    OpenStateEnums state = convertOpenState(channelMap);
    String openFailMessage = OpenStateEnums.FAIL == state ? convertFailMessage(event.getSubChannels()) : null;
    SubChannelInfo ali = channelMap.get(PaySceneEnums.ALI_OFFLINE);
    SubChannelInfo wxGzhOffline = channelMap.get(PaySceneEnums.WX_GZH_OFFLINE);
    OpenStateEnums wxOfflineState = convertWxOfflineState(channelMap);
    String wxOfflineFailMessage = convertWxOfflineFailMessage(channelMap);
    return BrandMerchantApply.builder()
        .id(apply.getId())
        .brandId(apply.getBrandId())
        .studioId(apply.getStudioId())
        .applyNo(apply.getApplyNo())
        .applyType(apply.getApplyType())
        .channelNo(apply.getChannelNo())
        .merchantId(event.getMerchantId())
        .merchantNo(event.getMerchantNo())
        .merchantName(event.getMerchantName())
        .shortName(event.getShortName())
        .companyType(event.getCompanyType())
        .openState(state)
        .openFailMessage(openFailMessage)
        .aliOpenState(ali.getOpenState())
        .aliAuthState(convertAliAuthState(event.getSubChannelAuths()))
        .aliMerchantNo(JSON.toJSONString(ali.getSubMerchantNos()))
        .aliOpenFailMessage(ali.getFailMessage())
        .wxOfflineOpenState(wxOfflineState)
        .wxOfflineMerchantNo(JSON.toJSONString(wxGzhOffline.getSubMerchantNos()))
        .wxOfflineAuthState(convertWxOfflineAuthState(event.getSubChannelAuths()))
        .wxOfflineOpenFailMessage(wxOfflineFailMessage)
        .businessProvince(event.getBusiness().getBusinessAddress().getProvince())
        .businessCity(event.getBusiness().getBusinessAddress().getCity())
        .businessDistrict(event.getBusiness().getBusinessAddress().getDistrict())
        .businessAddress(event.getBusiness().getBusinessAddress().getAddress())
        .licenseCode(event.getLicense().getLicenseCode())
        .licenseExpireType(event.getLicense().getLicenseExpireType())
        .licenseStartDate(event.getLicense().getLicenseStartDate())
        .licenseEndDate(event.getLicense().getLicenseEndDate())
        .licenseBusinessScope(event.getLicense().getBusinessScope())
        .licenseFoundDate(event.getLicense().getFoundDate())
        .licenseProvince(event.getLicense().getLicenseAddress().getProvince())
        .licenseCity(event.getLicense().getLicenseAddress().getCity())
        .licenseDistrict(event.getLicense().getLicenseAddress().getDistrict())
        .licenseAddress(event.getLicense().getLicenseAddress().getAddress())
        .legalName(event.getLegal().getLegalName())
        .legalCertNo(event.getLegal().getLegalCertNo())
        .legalCertType(event.getLegal().getLegalCertType())
        .legalCertExpireType(event.getLegal().getLegalCertExpireType())
        .legalCertStartDate(event.getLegal().getLegalCertStartDate())
        .legalCertEndDate(event.getLegal().getLegalCertEndDate())
        .legalPhone(event.getLegal().getLegalPhone())
        .legalProvince(event.getLegal().getLegalAddress().getProvince())
        .legalCity(event.getLegal().getLegalAddress().getCity())
        .legalDistrict(event.getLegal().getLegalAddress().getDistrict())
        .legalAddress(event.getLegal().getLegalAddress().getAddress())
        .contactEmail(event.getContact().getContactEmail())
        .bankCardType(event.getBankCard().getCardType())
        .bankCardNo(event.getBankCard().getCardNo())
        .bankCardName(event.getBankCard().getCardName())
        .bankPhone(event.getBankCard().getPhone())
        .bankCertNo(event.getBankCard().getCertNo())
        .bankCertType(event.getBankCard().getCertType())
        .bankCertExpireType(event.getBankCard().getCertExpireType())
        .bankCertStartDate(event.getBankCard().getCertStartDate())
        .bankCertEndDate(event.getBankCard().getCertEndDate())
        .bankBaseDepositNo(event.getBankCard().getBaseDepositNo())
        .bankCode(event.getBankCard().getBankCode())
        .bankName(event.getBankCard().getBankName())
        .bankBranchName(event.getBankCard().getBranchName())
        .bankBranchCode(event.getBankCard().getBranchCode())
        .bankCardProvince(event.getBankCard().getCardAddress().getProvince())
        .bankCardCity(event.getBankCard().getCardAddress().getCity())
        .bankCardDistrict(event.getBankCard().getCardAddress().getDistrict())
        .bankCardAddress(event.getBankCard().getCardAddress().getAddress())
        .resourceInfo(JSON.toJSONString(event.getResourceMap()))
        .finishTime(OpenStateEnums.SUCCESS == state ? new Date() : null)
        .updateTime(new Date())
        .build();
  }

  private static OpenStateEnums convertWxOfflineAuthState(List<SubChannelAuthInfo> auths) {
    return auths.stream()
        .filter(e -> e.getAuthType() == SubChannelAuthTypeEnums.WX_OFFLINE)
        .findFirst()
        .map(SubChannelAuthInfo::getAuthState)
        .orElse(null);
  }

  private static OpenStateEnums convertAliAuthState(List<SubChannelAuthInfo> auths) {
    return auths.stream()
        .filter(e -> e.getAuthType() == SubChannelAuthTypeEnums.ALI_OFFLINE)
        .findFirst()
        .map(SubChannelAuthInfo::getAuthState)
        .orElse(null);
  }

  private static String convertWxOfflineFailMessage(Map<PaySceneEnums, SubChannelInfo> channelMap) {
    SubChannelInfo wxGzhOffline = channelMap.get(PaySceneEnums.WX_GZH_OFFLINE);
    SubChannelInfo wxXcxOffline = channelMap.get(PaySceneEnums.WX_XCX_OFFLINE);
    if (OpenStateEnums.FAIL == wxGzhOffline.getOpenState()) {
      return wxGzhOffline.getFailMessage();
    }
    if (OpenStateEnums.FAIL == wxXcxOffline.getOpenState()) {
      return wxXcxOffline.getFailMessage();
    }
    return null;
  }

  private static OpenStateEnums convertWxOfflineState(Map<PaySceneEnums, SubChannelInfo> channelMap) {
    SubChannelInfo wxGzhOffline = channelMap.get(PaySceneEnums.WX_GZH_OFFLINE);
    SubChannelInfo wxXcxOffline = channelMap.get(PaySceneEnums.WX_XCX_OFFLINE);
    if (OpenStateEnums.SUCCESS == wxGzhOffline.getOpenState()
        && OpenStateEnums.SUCCESS == wxXcxOffline.getOpenState()) {
      return OpenStateEnums.SUCCESS;
    }
    if (OpenStateEnums.FAIL == wxGzhOffline.getOpenState()) {
      return OpenStateEnums.FAIL;
    }
    if (OpenStateEnums.FAIL == wxXcxOffline.getOpenState()) {
      return OpenStateEnums.FAIL;
    }
    return OpenStateEnums.PROCESS;
  }

  private static String convertFailMessage(List<SubChannelInfo> subChannels) {
    return subChannels.stream()
        .filter(e -> OpenStateEnums.FAIL == e.getOpenState())
        .map(SubChannelInfo::getFailMessage)
        .findFirst()
        .orElse(null);
  }

  private static OpenStateEnums convertOpenState(Map<PaySceneEnums, SubChannelInfo> channelMap) {

    List<PaySceneEnums> sceneList = getDefaultScene();

    OpenStateEnums state = OpenStateEnums.SUCCESS;
    for (PaySceneEnums scene : sceneList) {
      SubChannelInfo subChannel = channelMap.get(scene);
      if (OpenStateEnums.PROCESS == subChannel.getOpenState()) {
        state = OpenStateEnums.PROCESS;
      }
      if (OpenStateEnums.FAIL == subChannel.getOpenState()) {
        return OpenStateEnums.FAIL;
      }
    }
    return state;
  }

  private static Map<PaySceneEnums, SubChannelInfo> convertSubChannelMap(List<SubChannelInfo> subChannels) {
    Map<PaySceneEnums, SubChannelInfo> map = subChannels.stream()
        .collect(Collectors.toMap(SubChannelInfo::getScene, e -> e));
    for (OpenSingleSubChannelRequest defaultSubChannel : DEFAULT_SUB_CHANNELS) {
      if (null == map.get(defaultSubChannel.getScene())) {
        map.put(defaultSubChannel.getScene(),
            SubChannelInfo.builder().scene(defaultSubChannel.getScene()).openState(OpenStateEnums.PROCESS)
                .build());
      }
    }
    return map;
  }

  public static List<PaySceneEnums> getDefaultScene() {
    return DEFAULT_SUB_CHANNELS.stream().map(OpenSingleSubChannelRequest::getScene)
        .collect(Collectors.toList());
  }

  public static BrandMerchant convertMerchant(BrandMerchantApply apply) {
    BrandMerchant merchant = BeanUtil.map(apply, BrandMerchant.class);
    merchant.setId(apply.getMerchantId());
    merchant.setCreateTime(new Date());
    merchant.setUpdateTime(new Date());
    return merchant;
  }

}
