package com.jiejing.fitness.finance.service.pay.impl;

import static java.util.stream.Collectors.toList;

import com.alibaba.fastjson.JSON;
import com.alipay.api.AlipayApiException;
import com.alipay.api.AlipayConfig;
import com.alipay.api.AlipayRequest;
import com.alipay.api.DefaultAlipayClient;
import com.alipay.api.request.AlipaySystemOauthTokenRequest;
import com.alipay.api.response.AlipaySystemOauthTokenResponse;
import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.google.common.collect.Lists;
import com.jiejing.common.exception.BizException;
import com.jiejing.common.utils.collection.CollectionUtil;
import com.jiejing.common.utils.time.TimeUtil;
import com.jiejing.fitness.enums.finance.BrandCashierTransStateEnum;
import com.jiejing.fitness.enums.finance.PartyTypeEnum;
import com.jiejing.fitness.finance.api.pay.request.NativePayRequest;
import com.jiejing.fitness.finance.repository.entity.GlobalConfig;
import com.jiejing.fitness.finance.repository.entity.StudioCashierRecord;
import com.jiejing.fitness.finance.repository.entity.PartyToMerchant;
import com.jiejing.fitness.finance.repository.entity.StudioCheckSettleRecord;
import com.jiejing.fitness.finance.repository.entity.StudioMerchantApply;
import com.jiejing.fitness.finance.repository.entity.StudioSettleRecord;
import com.jiejing.fitness.finance.repository.service.GlobalConfigRpService;
import com.jiejing.fitness.finance.repository.service.StudioCashierRecordRpService;
import com.jiejing.fitness.finance.repository.service.PartyToMerchantRpService;
import com.jiejing.fitness.finance.repository.service.StudioCheckSettleRecordRpService;
import com.jiejing.fitness.finance.repository.service.StudioMerchantApplyRpService;
import com.jiejing.fitness.finance.repository.service.StudioSettleRecordRpService;
import com.jiejing.fitness.finance.service.enums.FinanceErrorEnums;
import com.jiejing.fitness.finance.service.enums.GlobalConfigEnums;
import com.jiejing.fitness.finance.service.pay.PayService;
import com.jiejing.fitness.finance.service.pay.convert.PayConvert;
import com.jiejing.fitness.finance.service.pay.params.BrandMerchantRefundParams;
import com.jiejing.fitness.finance.service.pay.params.NativePayParams;
import com.jiejing.fitness.finance.service.pay.params.StudioMerchantPayParams;
import com.jiejing.fitness.finance.service.rpc.MerchantRpcService;
import com.jiejing.fitness.finance.service.rpc.PayRpcService;
import com.jiejing.fitness.finance.service.rpc.StudioRpcService;
import com.jiejing.fitness.finance.service.utils.DingUtil;
import com.jiejing.fitness.finance.service.utils.FeeUtil;
import com.jiejing.paycenter.api.pay.PayApi;
import com.jiejing.paycenter.common.enums.common.PayChannelEnums;
import com.jiejing.paycenter.common.enums.common.TransStateEnums;
import com.jiejing.paycenter.common.enums.pay.PayTypeEnums;
import com.jiejing.paycenter.common.model.vo.MerchantVO;
import com.jiejing.paycenter.api.pay.request.PayRequest;
import com.jiejing.paycenter.api.pay.request.RefundPayRequest;
import com.jiejing.paycenter.common.model.vo.PayVO;
import com.jiejing.paycenter.api.pay.vo.RefundVO;
import com.jiejing.paycenter.common.event.PayEvent;
import com.jiejing.paycenter.common.event.RefundEvent;
import com.jiejing.paycenter.common.model.vo.SettleVO;
import com.jiejing.studio.api.studio.vo.StudioVO;
import com.jiejing.wechat.WeChatAuthService;
import com.jiejing.wechat.vo.weChat.BaseAuthInfoVO;
import java.math.BigDecimal;
import java.time.temporal.ChronoUnit;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;

/**
 * @author chengyubing
 * @since 2024/2/27 11:06
 */
@Slf4j
@Service
public class PayServiceImpl implements PayService {

  @Value("${finance.brand.merchant.channel}")
  private String channel;

  @Resource
  private PartyToMerchantRpService partyToMerchantRpService;

  @Resource
  private PayRpcService payRpcService;

  @Resource
  private StudioRpcService studioRpcService;

  @Resource
  private StudioCashierRecordRpService studioCashierRecordRpService;

  @Resource
  private MerchantRpcService merchantRpcService;

  @Resource
  private StudioSettleRecordRpService studioSettleRecordRpService;

  @Resource
  private StudioMerchantApplyRpService studioMerchantApplyRpService;

  @Resource
  private StudioCheckSettleRecordRpService studioCheckSettleRecordRpService;

  @Resource
  private WeChatAuthService weChatAuthService;

  @Resource
  private GlobalConfigRpService globalConfigRpService;

  private final Map<String, DefaultAlipayClient> aliClientMap = new HashMap<>();

  @PostConstruct
  public void init() {
    GlobalConfig config = globalConfigRpService.getById(GlobalConfigEnums.CASHIER_ALI_INFO.getCode())
        .orElse(new GlobalConfig());
    AliInfo info = JSON.parseObject(config.getConfigValue(), AliInfo.class);
    aliClientMap.put(info.getAppId(),
        new DefaultAlipayClient(info.getBaseUrl(), info.getAppId(), info.getMerchantPrivateKey(), "json",
            "utf-8"));
  }

  @Override
  public String getOpenId(PayChannelEnums channel, String appId, String authCode) {
    switch (channel) {
      case WX:
        return Optional.ofNullable(weChatAuthService.getBaseAuthInfo(null, appId, authCode))
            .map(BaseAuthInfoVO::getOpenId).orElse(null);
      case ALI:
        try {
          AlipaySystemOauthTokenResponse response = aliClientMap.get(appId).execute(convert(authCode));
          return response.getUserId();
        } catch (AlipayApiException e) {
          throw new BizException(e.getErrCode(), e.getMessage());
        }
      default:
        throw new BizException(FinanceErrorEnums.NOT_SUPPORT_TYPE);
    }
  }

  public AlipaySystemOauthTokenRequest convert(String authCode) {
    AlipaySystemOauthTokenRequest aliRequest = new AlipaySystemOauthTokenRequest();
    aliRequest.setCode(authCode);
    aliRequest.setGrantType("authorization_code");
    return aliRequest;
  }


  @Override
  public PayVO nativePay(NativePayParams params) {
    return payRpcService.pay(convertNativePay(params));
  }

  @Override
  public PayVO merchantPay(StudioMerchantPayParams params) {
    StudioVO studio = studioRpcService.getStudio(params.getStudioId());
    PartyToMerchant relation = partyToMerchantRpService.getByStudioId(studio.getId(), channel);
    if (null == relation) {
      throw new BizException(FinanceErrorEnums.MERCHANT_NOT_OPEN);
    }

    MerchantVO merchant = merchantRpcService.getByMerchantId(relation.getMerchantId());
    StudioCashierRecord record = PayConvert.convertPayInit(params, studio, merchant);
    PayRequest request = PayConvert.convert(params, record);
    record.setFeeRate(payRpcService.getFeeRate(request));
    record.setFee(FeeUtil.calPayFee(record.getFeeRate(), params.getTransAmount()));

    studioCashierRecordRpService.insert(record);
    return payRpcService.pay(request);
  }

  @Override
  public void payCallback(PayEvent event) {
    if (!event.getChannelNo().equals(channel)) {
      return;
    }
    StudioCashierRecord record = studioCashierRecordRpService.getById(Long.parseLong(event.getTransNo()))
        .orElse(null);
    if (null == record) {
      return;
    }

    StudioCashierRecord toModify = PayConvert.convertPay(record, event);
    studioCashierRecordRpService.updateById(toModify);
  }

  @Override
  public RefundVO merchantRefund(BrandMerchantRefundParams params) {
    StudioCashierRecord pay = studioCashierRecordRpService.getById(Long.parseLong(params.getPayTransNo()))
        .orElseThrow(() -> new BizException(FinanceErrorEnums.NOT_EXIST));

    BigDecimal historyRefundActualAmount = studioCashierRecordRpService.sumRefundActualAmountByPayTransNo(
        params.getPayTransNo());

    StudioCashierRecord record = PayConvert.convertRefundInit(params, pay, historyRefundActualAmount);
    studioCashierRecordRpService.insert(record);

    RefundPayRequest request = PayConvert.convert(params, record);
    RefundVO vo = payRpcService.refund(request);

    StudioCashierRecord toModify = PayConvert.convertRefund(record, vo);
    studioCashierRecordRpService.updateById(toModify);
    return vo;
  }

  @Override
  public void refundCallback(RefundEvent event) {
    StudioCashierRecord record = studioCashierRecordRpService.getById(Long.parseLong(event.getTransNo()))
        .orElse(null);
    if (null == record) {
      return;
    }

    StudioCashierRecord toModify = PayConvert.convertRefund(record, event);
    studioCashierRecordRpService.updateById(toModify);
  }

  @Override
  public void checkSettle(Long merchantId, Date settleDate) {
    Date endDate = null == settleDate ? TimeUtil.local().startOfDay(new Date()) : settleDate;
    Date startDate = TimeUtil.local().plus(endDate, -15, ChronoUnit.DAYS);
    AtomicInteger failCount = new AtomicInteger(0);
    this.pageAndConsumer(merchantId, 200, apply -> {
      StudioSettleRecord exist = studioSettleRecordRpService.getByMerchantIdAndSettleDate(
          apply.getMerchantId(), endDate);
      if (exist != null) {
        return;
      }
      SettleVO vo = payRpcService.syncSettle(apply.getMerchantId(), endDate);
      // 对账
      BigDecimal totalAmount = studioCashierRecordRpService.sumMerchantPaySuccess(apply.getMerchantId(),
          startDate, endDate);
      if (BigDecimal.ZERO.compareTo(vo.getTransAmount()) == 0
          && BigDecimal.ZERO.compareTo(totalAmount) == 0) {
        // 没有交易
        return;
      }

      if (vo.getTransAmount().compareTo(totalAmount) == 0) {
        // 对账成功
        Map<Long, BigDecimal> studioTransAmountMap = studioCashierRecordRpService.sumMerchantPaySuccessGroupByStudioId(
            apply.getMerchantId(), startDate, endDate);
        Map<Long, StudioVO> studioMap = studioRpcService.mapStudio(
            Lists.newArrayList(studioTransAmountMap.keySet()));
        studioSettleRecordRpService.insertAll(
            PayConvert.convertSettle(apply, studioTransAmountMap, vo, studioMap));
        if (TransStateEnums.SUCCESS == TransStateEnums.getByCode(vo.getTransState())) {
          // 结算成功，则更新收银流水状态为记录为入账成功
          this.updatePayIn(apply.getMerchantId(), startDate, endDate);
        }
      } else {
        // 对账失败
        failCount.incrementAndGet();
        studioCheckSettleRecordRpService.insert(PayConvert.convertCheckSettle(apply, vo, totalAmount));
      }
    });

    // 钉钉消息
    if (failCount.get() > 0) {
      DingUtil.sendCheckSettleFail(failCount.get());
    } else {
      DingUtil.sendCheckSettleSuccess();
    }

  }

  private void updatePayIn(Long merchantId, Date startTime, Date endTime) {
    Integer current = 0;
    Integer size = 500;

    do {
      Page<StudioCashierRecord> page = studioCashierRecordRpService.pageMerchantPaySuccess(merchantId,
          startTime, endTime, current, size);
      if (CollectionUtil.isEmpty(page.getContent())) {
        break;
      }

      try {
        List<Long> ids = page.getContent().stream().map(StudioCashierRecord::getId).collect(toList());
        studioCashierRecordRpService.updateByIds(
            StudioCashierRecord.builder().transState(BrandCashierTransStateEnum.PAY_IN.getCode())
                .inTime(endTime)
                .updateTime(new Date()).build(), ids);

      } finally {
        current++;
      }

      if (!page.hasNext()) {
        break;
      }
    } while (true);
  }


  public void pageAndConsumer(Long merchantId, Integer batch, Consumer<StudioMerchantApply> consumer) {
    Long minId = 0L;
    log.info("consumer merchant apply start");
    do {
      List<StudioMerchantApply> list = studioMerchantApplyRpService.listByMinId(minId, merchantId, batch);
      if (CollectionUtil.isEmpty(list)) {
        break;
      }
      try {

        list.forEach(apply -> {
          try {
            log.info("start consumer merchant apply {}, {}", apply.getMerchantId(), apply.getMerchantNo());
            consumer.accept(apply);
            log.info("end consumer merchant apply {}, {}", apply.getMerchantId(), apply.getMerchantNo());
          } catch (Exception e) {
            log.error("consumer apply fail apply {}, {}", apply.getMerchantId(), apply.getMerchantNo(), e);
          }
        });

      } finally {
        minId = list.get(list.size() - 1).getId();
      }
    } while (true);
    log.info("consumer merchant apply finished");
  }


  private String getChannelNo(NativePayParams params) {
    switch (params.getChannel()) {
      case WX:
        return "003";
      case ALI:
        return "002";
      default:
        throw new BizException(FinanceErrorEnums.NOT_SUPPORT_TYPE);
    }
  }

  private PayRequest convertNativePay(NativePayParams params) {
    PayRequest req = new PayRequest();
    req.setChannelNo(getChannelNo(params));
    req.setTransNo(IdWorker.getIdStr());
    req.setAmount(params.getAmount());
    req.setPayType(PayTypeEnums.NATIVE);
    req.setGoods(params.getGoods());
    req.setOrderNo(params.getOrderNo());
    req.setOrderType(params.getOrderType().getCode());
    req.setTimeExpire(params.getTimeExpire());
    req.setTradingTime(new Date());
    req.setExtra(params.getExtra());
    return req;
  }


  @Data
  private static class AliInfo {

    private String appId;

    private String baseUrl;

    private String alipayPublicKey;

    private String merchantPrivateKey;

  }

}
