package com.jiejing.fitness.finance.service.utils;

import java.math.BigDecimal;
import java.math.RoundingMode;

/**
 * @author chengyubing
 * @since 2024/2/27 14:16
 */
public class FeeUtil {

  /**
   * 【支付】计算支付手续费，保留小数点后两位，四舍五入
   *
   * @param feeRate 费率（%）
   * @param amount  支付金额（元）
   * @return 手续费
   */
  public static BigDecimal calPayFee(BigDecimal feeRate, BigDecimal amount) {
    return MoneyUtil.defaultRound(MoneyUtil.multiply(
        MoneyUtil.divide(feeRate, new BigDecimal("100"), 4, RoundingMode.HALF_UP.ordinal()),
        amount));
  }

  /**
   * 【退款】计算退款手续费（元）
   *
   * <code>试算手续费 = 向下取整(退款金额/原交易金额*原交易手续费金额)</code>
   * <code>试算实退金额 = 退款申请金额-试算手续费</code>
   * <code>剩余实收金额 = 实收金额-已实退金额</code>
   * <code>若：剩余实收金额 >= 试算实退金额，则：应退手续费 = 试算手续费</code>
   * <code>若：剩余实收金额 < 试算实退金额，则：应退手续费 = 试算手续费 + (试算实退金额 - 剩余实收金额)</code>
   *
   * @param refundTransAmount         退款申请金额（元）
   * @param payTransAmount            原支付交易金额（元）
   * @param payActualAmount           用户实收金额（元）
   * @param payFee                    平台实收手续费（元）
   * @param historyRefundActualAmount 历史实退金额（元）
   * @return 本次退款应退手续费（元）
   */
  public static BigDecimal calculateRefundFee(BigDecimal refundTransAmount, BigDecimal payTransAmount,
      BigDecimal payActualAmount, BigDecimal payFee, BigDecimal historyRefundActualAmount) {
    // 机构剩余实收金额
    BigDecimal leftPayActualAmount = MoneyUtil.subtract(payActualAmount, historyRefundActualAmount);
    // 试算手续费：向下取整（退款金额/原交易金额*原支付交易手续费金额）
    BigDecimal trialFee = calculateTrialFee(refundTransAmount, payTransAmount, payFee);
    // 试算实退金额
    BigDecimal trialActualAmount = MoneyUtil.subtract(refundTransAmount, trialFee);
    // 实退手续费
    return leftPayActualAmount.compareTo(trialActualAmount) >= 0 ? trialFee
        : MoneyUtil.add(trialFee, MoneyUtil.subtract(trialActualAmount, leftPayActualAmount));
  }

  /**
   * 计算试算手续费
   * <p>
   * 试算手续费 = 向下取整(退款金额/原交易金额*原交易手续费金额)
   */
  private static BigDecimal calculateTrialFee(BigDecimal refundTransAmount, BigDecimal payTransAmount,
      BigDecimal payFee) {
    // 试算手续费：向下取整（退款金额/原交易金额*原支付交易手续费金额）
    return MoneyUtil.divide(refundTransAmount.multiply(payFee), payTransAmount, 2,
        RoundingMode.FLOOR);
  }

  public static void main(String[] args) {
    BigDecimal payTransAmount = new BigDecimal("100");
    BigDecimal payActualAmount = new BigDecimal("99.43");
    BigDecimal payFee = new BigDecimal("0.57");
    BigDecimal refundAmount1 = new BigDecimal("33");
    BigDecimal refundAmount2 = new BigDecimal("33");
    BigDecimal refundAmount3 = new BigDecimal("33");
    BigDecimal refundAmount4 = new BigDecimal("1");

    BigDecimal fee1 = test33(payTransAmount, payActualAmount, payFee, refundAmount1,
        new BigDecimal("0"));
    BigDecimal refundActualAmount1 = MoneyUtil.subtract(refundAmount1, fee1);
    BigDecimal historyRefundActualAmount1 = MoneyUtil.subtract(refundAmount1, fee1);
    BigDecimal leftPayActualAmount1 = MoneyUtil.subtract(payActualAmount, historyRefundActualAmount1);
    System.out.println(
        ">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>> 收款金额：\t" + payTransAmount + "，实收金额：\t" + payActualAmount + "，收款手续费：\t"
            + payFee + "，退款申请金额：\t" + refundAmount1 + "，手续费：\t" + fee1 + "，实退金额：\t" + refundActualAmount1 + "，剩余实收金额：\t" + leftPayActualAmount1);

    BigDecimal fee2 = test33(payTransAmount, payActualAmount, payFee, refundAmount2,
        historyRefundActualAmount1);
    BigDecimal refundActualAmount2 = MoneyUtil.subtract(refundAmount2, fee2);
    BigDecimal historyRefundActualAmount2 = MoneyUtil.add(refundActualAmount2, historyRefundActualAmount1);
    BigDecimal leftPayActualAmount2 = MoneyUtil.subtract(payActualAmount, historyRefundActualAmount2);
    System.out.println(
        ">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>> 收款金额：\t" + payTransAmount + "，实收金额：\t" + payActualAmount + "，收款手续费：\t"
            + payFee + "，退款申请金额：\t" + refundAmount2 + "，手续费：\t" + fee2 + "，实退金额：\t" + refundActualAmount2 + "，剩余实收金额：\t" + leftPayActualAmount2);

    BigDecimal fee3 = test33(payTransAmount, payActualAmount, payFee, refundAmount3,
        historyRefundActualAmount2);
    BigDecimal refundActualAmount3 = MoneyUtil.subtract(refundAmount3, fee3);
    BigDecimal historyRefundActualAmount3 = MoneyUtil.add(refundActualAmount3, historyRefundActualAmount2);
    BigDecimal leftPayActualAmount3 = MoneyUtil.subtract(payActualAmount, historyRefundActualAmount3);
    System.out.println(
        ">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>> 收款金额：\t" + payTransAmount + "，实收金额：\t" + payActualAmount + "，收款手续费：\t"
            + payFee + "，退款申请金额：\t" + refundAmount3 + "，手续费：\t" + fee3 + "，实退金额：\t" + refundActualAmount3 + "，剩余实收金额：\t" + leftPayActualAmount3);

    BigDecimal fee4 = test33(payTransAmount, payActualAmount, payFee, refundAmount4,
        historyRefundActualAmount3);
    BigDecimal refundActualAmount4 = MoneyUtil.subtract(refundAmount4, fee4);
    BigDecimal historyRefundActualAmount4 = MoneyUtil.add(refundActualAmount4, historyRefundActualAmount3);
    BigDecimal leftPayActualAmount4 = MoneyUtil.subtract(payActualAmount, historyRefundActualAmount4);
    System.out.println(
        ">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>> 收款金额：\t" + payTransAmount + "，实收金额：\t" + payActualAmount + "，收款手续费：\t"
            + payFee + "，退款申请金额：\t" + refundAmount4 + "，手续费：\t" + fee4 + "，实退金额：\t" + refundActualAmount4 + "，剩余实收金额：\t" + leftPayActualAmount4);

    BigDecimal fee5 = test33(payTransAmount, payActualAmount, payFee, refundAmount4,
        historyRefundActualAmount4);
    BigDecimal refundActualAmount5 = MoneyUtil.subtract(refundAmount4, fee5);
    BigDecimal historyRefundActualAmount5 = MoneyUtil.add(refundActualAmount5, historyRefundActualAmount4);
    BigDecimal leftPayActualAmount5 = MoneyUtil.subtract(payActualAmount, historyRefundActualAmount5);
    System.out.println(
        ">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>> 收款金额：\t" + payTransAmount + "，实收金额：\t" + payActualAmount + "，收款手续费：\t"
            + payFee + "，退款申请金额：\t" + refundAmount4 + "，手续费：\t" + fee5 + "，实退金额：\t" + refundActualAmount3 + "，剩余实收金额：\t" + leftPayActualAmount5);
  }

  private static BigDecimal test33(BigDecimal payTransAmount, BigDecimal payActualAmount, BigDecimal payFee,
      BigDecimal refundTransAmount, BigDecimal historyRefundActualAmount) {
    return FeeUtil.calculateRefundFee(refundTransAmount, payTransAmount, payActualAmount,
        payFee,
        historyRefundActualAmount);
  }

}
