package com.jiejing.fitness.finance.service.pay.impl;

import com.jiejing.common.exception.BizException;
import com.jiejing.fitness.finance.repository.entity.StudioCashierRecord;
import com.jiejing.fitness.finance.repository.entity.PartyToMerchant;
import com.jiejing.fitness.finance.repository.service.StudioCashierRecordRpService;
import com.jiejing.fitness.finance.repository.service.PartyToMerchantRpService;
import com.jiejing.fitness.finance.service.enums.FinanceErrorEnums;
import com.jiejing.fitness.finance.service.pay.PayService;
import com.jiejing.fitness.finance.service.pay.convert.PayConvert;
import com.jiejing.fitness.finance.service.pay.params.BrandMerchantRefundParams;
import com.jiejing.fitness.finance.service.pay.params.StudioMerchantPayParams;
import com.jiejing.fitness.finance.service.rpc.MerchantRpcService;
import com.jiejing.fitness.finance.service.rpc.PayRpcService;
import com.jiejing.fitness.finance.service.rpc.StudioRpcService;
import com.jiejing.paycenter.common.model.vo.MerchantVO;
import com.jiejing.paycenter.api.pay.request.PayRequest;
import com.jiejing.paycenter.api.pay.request.RefundPayRequest;
import com.jiejing.paycenter.common.model.vo.PayVO;
import com.jiejing.paycenter.api.pay.vo.RefundVO;
import com.jiejing.paycenter.common.event.PayEvent;
import com.jiejing.paycenter.common.event.RefundEvent;
import com.jiejing.studio.api.studio.vo.StudioVO;
import java.math.BigDecimal;
import javax.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

/**
 * @author chengyubing
 * @since 2024/2/27 11:06
 */
@Slf4j
@Service
public class PayServiceImpl implements PayService {

  @Value("${finance.brand.merchant.channel}")
  private String channel;

  @Resource
  private PartyToMerchantRpService partyToMerchantRpService;

  @Resource
  private PayRpcService payRpcService;

  @Resource
  private StudioRpcService studioRpcService;

  @Resource
  private StudioCashierRecordRpService studioCashierRecordRpService;

  @Resource
  private MerchantRpcService merchantRpcService;

  @Override
  public PayVO merchantPay(StudioMerchantPayParams params) {
    StudioVO studio = studioRpcService.getStudio(params.getStudioId());
    PartyToMerchant relation = partyToMerchantRpService.getByStudioId(studio.getId(), channel);
    if (null == relation) {
      throw new BizException(FinanceErrorEnums.MERCHANT_NOT_OPEN);
    }

    MerchantVO merchant = merchantRpcService.getByMerchantId(relation.getMerchantId());

    StudioCashierRecord record = PayConvert.convertPayInit(params, studio, merchant);
    studioCashierRecordRpService.insert(record);

    PayRequest request = PayConvert.convert(params, record);
    return payRpcService.pay(request);
  }

  @Override
  public void payCallback(PayEvent event) {
    StudioCashierRecord record = studioCashierRecordRpService.getById(Long.parseLong(event.getTransNo()))
        .orElse(null);
    if (null == record) {
      return;
    }

    StudioCashierRecord toModify = PayConvert.convertPay(record, event);
    studioCashierRecordRpService.updateById(toModify);
  }

  @Override
  public RefundVO merchantRefund(BrandMerchantRefundParams params) {
    StudioCashierRecord pay = studioCashierRecordRpService.getById(Long.parseLong(params.getPayTransNo()))
        .orElseThrow(() -> new BizException(FinanceErrorEnums.NOT_EXIST));

    BigDecimal historyRefundActualAmount = studioCashierRecordRpService.sumRefundActualAmountByPayTransNo(
        params.getPayTransNo());

    StudioCashierRecord record = PayConvert.convertRefundInit(params, pay, historyRefundActualAmount);
    studioCashierRecordRpService.insert(record);

    RefundPayRequest request = PayConvert.convert(params, record);
    RefundVO vo = payRpcService.refund(request);

    StudioCashierRecord toModify = PayConvert.convertRefund(record, vo);
    studioCashierRecordRpService.updateById(toModify);
    return vo;
  }

  @Override
  public void refundCallback(RefundEvent event) {
    StudioCashierRecord record = studioCashierRecordRpService.getById(Long.parseLong(event.getTransNo()))
        .orElse(null);
    if (null == record) {
      return;
    }

    StudioCashierRecord toModify = PayConvert.convertRefund(record, event);
    studioCashierRecordRpService.updateById(toModify);
  }

}
