package com.jiejing.fitness.finance.service.rpc;

import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.google.common.collect.Maps;
import com.jiejing.common.model.JsonResult;
import com.jiejing.paycenter.api.merchant.MerchantApi;
import com.jiejing.paycenter.api.merchant.MerchantQueryApi;
import com.jiejing.paycenter.api.merchant.request.ApplyMerchantRequest;
import com.jiejing.paycenter.api.merchant.request.AuthSubChannelRequest;
import com.jiejing.paycenter.api.merchant.request.ConfigSubChannelRequest;
import com.jiejing.paycenter.api.merchant.request.GetMerchantByIdRequest;
import com.jiejing.paycenter.api.merchant.request.ListMerchantByIdsRequest;
import com.jiejing.paycenter.api.merchant.request.SyncSubChannelAuthRequest;
import com.jiejing.paycenter.api.merchant.request.UploadRequest;
import com.jiejing.paycenter.common.model.request.Address;
import com.jiejing.paycenter.common.model.vo.ApplyMerchantResultVO;
import com.jiejing.paycenter.common.model.vo.AuthSubChannelVO;
import com.jiejing.paycenter.common.model.vo.ConfigSubChannelVO;
import com.jiejing.paycenter.common.model.vo.MerchantVO;
import com.jiejing.paycenter.common.model.vo.UploadVO;
import com.jiejing.paycenter.common.enums.common.OpenStateEnums;
import com.jiejing.paycenter.common.enums.merchant.AuthPhaseEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelAuthTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelConfigTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelEnums;
import java.util.List;
import java.util.Map;
import javax.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

/**
 * @author chengyubing
 * @since 2024/2/20 16:39
 */
@Slf4j
@Service
public class MerchantRpcService {

  @Resource
  private MerchantApi merchantApi;

  @Resource
  private MerchantQueryApi merchantQueryApi;

  public String upload(UploadRequest request) {
    JsonResult<UploadVO> result = merchantApi.upload(request);
    result.assertSuccess();
    return result.getResult().getThirdId();
  }

  public ApplyMerchantResultVO apply(ApplyMerchantRequest request) {
    // 重新映射下地址信息：系统行政区划存在错误，需要订正
    this.resetAddress(request);
    JsonResult<ApplyMerchantResultVO> result = merchantApi.apply(request);
    result.assertSuccess();
    return result.getResult();
  }

  private void resetAddress(ApplyMerchantRequest request) {
    // 重新映射下地址信息：系统行政区划存在错误，需要订正
    this.resetAddress(request.getLegal().getLegalAddress());
    this.resetAddress(request.getBankCard().getCardAddress());
    this.resetAddress(request.getLicense().getLicenseAddress());
    this.resetAddress(request.getBusiness().getBusinessAddress());
  }

  private static final Map<String, String> DISTRICT_CODE_MAP = Maps.newHashMap();
  static {
    // 东莞市
    DISTRICT_CODE_MAP.put("441901", "441900");
    // 中山市
    DISTRICT_CODE_MAP.put("442001", "442000");
  }

  private void resetAddress(Address address) {
    address.setDistrict(DISTRICT_CODE_MAP.getOrDefault(address.getDistrict(), address.getDistrict()));
  }

  public MerchantVO getByMerchantId(Long merchantId) {
    GetMerchantByIdRequest request = GetMerchantByIdRequest.builder().merchantId(merchantId).build();
    JsonResult<MerchantVO> result = merchantQueryApi.getById(request);
    result.assertSuccess();
    return result.getResult();
  }

  public ConfigSubChannelVO bindAppIdWxXcxOffline(String channelNo, Long merchantId, String appId) {
    JsonResult<ConfigSubChannelVO> result = merchantApi.configSubChannel(ConfigSubChannelRequest.builder()
        .applyNo(IdWorker.getIdStr())
        .appId(appId)
        .merchantId(merchantId)
        .channelNo(channelNo)
        .subChannel(SubChannelEnums.WX)
        .configType(SubChannelConfigTypeEnums.WX_XCX_OFFLINE)
        .build());
    if (!result.getSuccess()) {
      return ConfigSubChannelVO.builder().state(OpenStateEnums.FAIL.name()).failMessage(result.getMessage())
          .build();
    }
    return result.getResult();
  }

  public AuthSubChannelVO authSubChannel(String channel, Long merchantId, SubChannelEnums subChannel,
      SubChannelAuthTypeEnums authType) {
    JsonResult<AuthSubChannelVO> result = merchantApi.authSubChannel(AuthSubChannelRequest.builder()
        .applyNo(IdWorker.getIdStr())
        .channelNo(channel)
        .merchantId(merchantId)
        .subChannel(subChannel)
        .authType(authType)
        .build());
    if (!result.getSuccess()) {
      return AuthSubChannelVO.builder()
          .state(OpenStateEnums.FAIL.name())
          .phase(AuthPhaseEnums.APPLY_COMMIT_FAIL.name())
          .failMessage(result.getMessage()).build();
    }
    return result.getResult();
  }


  public AuthSubChannelVO syncAuthSubChannel(Long merchantId, SubChannelEnums subChannel,
      SubChannelAuthTypeEnums authType) {
    JsonResult<AuthSubChannelVO> result = merchantApi.syncAuthSubChannel(SyncSubChannelAuthRequest.builder()
        .merchantId(merchantId)
        .subChannel(subChannel)
        .authType(authType)
        .build());
    return result.getResult();
  }

  public List<MerchantVO> listByIds(List<Long> merchantIds) {
    ListMerchantByIdsRequest request = new ListMerchantByIdsRequest();
    request.setMerchantIds(merchantIds);
    JsonResult<List<MerchantVO>> result = merchantQueryApi.listByIds(request);
    result.assertSuccess();
    return result.getResult();
  }

}
