package com.jiejing.fitness.finance.service.merchant.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.google.common.collect.Lists;
import com.jiejing.common.exception.BizException;
import com.jiejing.common.model.PageVO;
import com.jiejing.common.utils.collection.CollectionUtil;
import com.jiejing.common.utils.convert.BeanUtil;
import com.jiejing.common.utils.text.StringUtil;
import com.jiejing.common.utils.time.TimeUtil;
import com.jiejing.filecenter.api.resource.vo.ResourceInfoVO;
import com.jiejing.fitness.enums.finance.EmbededXcxEnum;
import com.jiejing.fitness.enums.finance.PartyTypeEnum;
import com.jiejing.fitness.finance.api.merchant.request.model.BrandResourceInfo;
import com.jiejing.fitness.finance.api.merchant.vo.StudioEmbeddedXcxVO;
import com.jiejing.fitness.finance.api.merchant.vo.StudioMerchantApplyVO;
import com.jiejing.fitness.finance.api.merchant.vo.StudioMerchantAuthSubChannelVO;
import com.jiejing.fitness.finance.api.merchant.vo.StudioMerchantBindXcxAppIdVO;
import com.jiejing.fitness.finance.api.merchant.vo.StudioMerchantVO;
import com.jiejing.fitness.finance.repository.entity.PartyToMerchant;
import com.jiejing.fitness.finance.repository.entity.StudioEmbeddedXcxApply;
import com.jiejing.fitness.finance.repository.entity.StudioMerchantApply;
import com.jiejing.fitness.finance.repository.query.PageStudioMerchantApplyQuery;
import com.jiejing.fitness.finance.repository.service.PartyToMerchantRpService;
import com.jiejing.fitness.finance.repository.service.StudioEmbeddedXcxApplyRpService;
import com.jiejing.fitness.finance.repository.service.StudioMerchantApplyRpService;
import com.jiejing.fitness.finance.service.config.PayChannelProperties;
import com.jiejing.fitness.finance.service.enums.FinanceErrorEnums;
import com.jiejing.fitness.finance.service.global.ConfigService;
import com.jiejing.fitness.finance.service.global.dto.SubChannelInfoDTO;
import com.jiejing.fitness.finance.service.merchant.BrandMerchantService;
import com.jiejing.fitness.finance.service.merchant.StudioMerchantService;
import com.jiejing.fitness.finance.service.merchant.convert.MerchantConvert;
import com.jiejing.fitness.finance.service.merchant.convert.StudioEmbeddedXcxConvert;
import com.jiejing.fitness.finance.service.merchant.params.ApplyStudioMerchantParams;
import com.jiejing.fitness.finance.service.merchant.params.PageStudioMerchantApplyParams;
import com.jiejing.fitness.finance.service.rpc.MerchantRpcService;
import com.jiejing.fitness.finance.service.rpc.MessageRpcService;
import com.jiejing.fitness.finance.service.rpc.ResourceRpcService;
import com.jiejing.fitness.finance.service.rpc.StudioRpcService;
import com.jiejing.message.enums.MsgChannelEnum;
import com.jiejing.message.event.SendCommonMsgEvent;
import com.jiejing.paycenter.api.merchant.request.ApplyMerchantRequest;
import com.jiejing.paycenter.common.enums.merchant.SubChannelOpenTypeEnums;
import com.jiejing.paycenter.common.event.MerchantSubChannelEvent;
import com.jiejing.paycenter.common.model.request.SubChannelConfigInfo;
import com.jiejing.paycenter.common.model.vo.AuthSubChannelVO;
import com.jiejing.paycenter.common.model.vo.ConfigSubChannelVO;
import com.jiejing.paycenter.common.model.vo.MerchantVO;
import com.jiejing.paycenter.common.model.vo.SubChannelAuthVO;
import com.jiejing.paycenter.common.enums.common.OpenStateEnums;
import com.jiejing.paycenter.common.enums.merchant.ResourceTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelAuthTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelConfigTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelEnums;
import com.jiejing.paycenter.common.event.MerchantEvent;
import com.jiejing.paycenter.common.model.request.ResourceInfo;
import com.jiejing.paycenter.common.model.vo.SubChannelConfigVO;
import com.jiejing.paycenter.common.model.vo.SubChannelVO;
import com.jiejing.studio.api.studio.vo.StudioVO;
import com.jiejing.wechat.WeXcxService;
import com.jiejing.wechat.vo.xcx.HalfScreenXcxAuthVO;
import com.jiejing.wechat.vo.xcx.HalfScreenXcxAuthVO.AuthXcxInfo;
import com.xiaomai.event.EventAgent;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

/**
 * @author chengyubing
 * @since 2024/3/5 09:58
 */
@Slf4j
@Service
public class StudioMerchantServiceImpl implements StudioMerchantService {

  @Resource
  private PayChannelProperties config;

  @Resource
  private StudioRpcService studioRpcService;

  @Resource
  private MerchantRpcService merchantRpcService;

  @Resource
  private ResourceRpcService resourceRpcService;

  @Resource
  private StudioMerchantApplyRpService studioMerchantApplyRpService;

  @Resource
  private PartyToMerchantRpService partyToMerchantRpService;

  @Resource
  private BrandMerchantService brandMerchantService;

  @Resource
  private ConfigService configService;

  @Resource
  private MessageRpcService messageRpcService;

  @Resource
  private StudioEmbeddedXcxApplyRpService studioEmbeddedXcxApplyRpService;

  @Resource
  private WeXcxService weXcxService;

  @Resource(name = "financeThreadPool")
  private Executor executor;

  @Value("${wx.component.appId}")
  private String wxComponentAppId;

  private static final Integer WX_SUCCESS_CODE = 0;

  private final static List<String> SPECIAL_FAIL_LIST = Lists.newArrayList(
      "该商户号类型暂不支持与该AppID关联"
  );

  @Override
  public Long apply(ApplyStudioMerchantParams params) {

    this.checkBeforeApply(params);

    StudioMerchantApply apply = this.saveApply(params);

    executor.execute(() -> {
      try {

        Map<ResourceTypeEnums, ResourceInfo> resourceMap = this.upload(params, apply);

        SubChannelInfoDTO subChannel = configService.getDefaultBrandSubChannelInfo();

        ApplyMerchantRequest request = MerchantConvert.convertRequest(params, apply, resourceMap, subChannel);
        merchantRpcService.apply(request);

      } catch (Exception e) {
        log.error("apply brand merchant fail {}, studio id = {}", apply.getId(), params.getStudioId(), e);
        String failMessage =
            e instanceof BizException ? ((BizException) e).getDefaultErrorMessage() : e.getMessage();
        this.doOpenMerchantFail(apply.getId(), failMessage);
        return;
      }

      this.doOpenMerchantProcess(apply.getId());
    });

    return apply.getId();
  }

  @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
  @Override
  public void bind(Long studioId, Long merchantId, Boolean flag) {
    PartyToMerchant relation = this.getRelation(studioId);
    StudioMerchantApply apply = studioMerchantApplyRpService.getLatestOneSuccessByMerchantId(merchantId);
    if (null == relation) {
      partyToMerchantRpService.insert(
          MerchantConvert.convertPartyToMerchant(apply.getChannelNo(), studioId, PartyTypeEnum.STUDIO,
              apply.getMerchantId(), apply.getMerchantNo()));
    } else {
      partyToMerchantRpService.updateById(
          MerchantConvert.convertPartyToMerchant(relation.getId(), merchantId, apply.getMerchantNo()));
    }
    // 新增绑定记录
    if (flag) {
      StudioVO studio = studioRpcService.getStudio(studioId);
      StudioMerchantApply bindApply = MerchantConvert.convertBindApply(studio, apply);
      studioMerchantApplyRpService.insert(bindApply);
    }
  }

  @Override
  public void unbind(Long studioId, Long merchantId) {
    partyToMerchantRpService.deleteByPartyAndMerchantId(studioId, PartyTypeEnum.STUDIO, merchantId);
  }

  @Override
  public void unbindAll(Long merchantId) {
    log.info("unbind all studio of merchant {}", merchantId);
    partyToMerchantRpService.deleteByMerchantId(merchantId, PartyTypeEnum.STUDIO);
  }

  @Override
  public void unbindAll(Set<Long> merchantIds) {
    log.info("unbind all studio of merchantIds {}", JSON.toJSONString(merchantIds));
    partyToMerchantRpService.deleteByMerchantIds(merchantIds, PartyTypeEnum.STUDIO);
  }

  @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
  @Override
  public void callback(MerchantEvent event) {
    switch (OpenStateEnums.getByCode(event.getState())) {
      case PROCESS:
        this.doOpenMerchantProcess(Long.parseLong(event.getApplyNo()));
        break;
      case FAIL:
        this.doOpenMerchantFail(Long.parseLong(event.getApplyNo()), event.getFailMessage());
        break;
      case SUCCESS:
        this.doOpenMerchantSuccess(event);
        break;
      default:
        break;
    }
  }

  @Override
  public void merchantSubChannelEventCallback(MerchantSubChannelEvent event) {
    // 如果是特定的失败，则绑定半屏支付小程序
    if (!OpenStateEnums.FAIL.getCode().equals(event.getState())) {
      return;
    }
    if (!SubChannelConfigTypeEnums.WX_XCX_OFFLINE.getCode().equals(event.getConfigType())) {
      return;
    }
    if (!SPECIAL_FAIL_LIST.contains(event.getFailMessage())) {
      return;
    }
    StudioMerchantApply apply = studioMerchantApplyRpService.getLatestOneSuccessByMerchantId(
        event.getMerchantId());
    this.bindEmbeddedXcx(apply.getStudioId(), Lists.newArrayList(event.getAppId()));
  }

  @Override
  public StudioMerchantVO getMerchant(Long studioId) {
    PartyToMerchant relation = getRelation(studioId);
    if (null == relation) {
      return null;
    }
    MerchantVO merchant = merchantRpcService.getByMerchantId(relation.getMerchantId());
    List<Long> resourceIds = this.getResourceIds(merchant);
    StudioMerchantApply apply = studioMerchantApplyRpService.getLatestOneSuccessByMerchantId(
        relation.getMerchantId());
    Map<Long, String> urlMap = resourceRpcService.getResourceUrlMap(apply.getBrandId(), resourceIds);
    return MerchantConvert.convertStudioMerchant(relation, merchant,
        configService.getDefaultBrandSubChannelInfo(), urlMap, apply);
  }

  @Override
  public StudioEmbeddedXcxVO getEmbeddedXcx(Long studioId, String appId) {
    String systemAppId = this.listSystemXcxAppIds().stream().findFirst().orElse(null);
    StudioEmbeddedXcxApply embeddedXcx = studioEmbeddedXcxApplyRpService.getByStudioIdAndAppId(
        studioId, wxComponentAppId, systemAppId, appId);
    if (null == embeddedXcx) {
      return null;
    }
    return BeanUtil.map(embeddedXcx, StudioEmbeddedXcxVO.class);
  }

  @Async("financeThreadPool")
  @Override
  public void bindEmbeddedXcx(Long studioId, List<String> appIds) {
    if (CollectionUtil.isEmpty(appIds)) {
      return;
    }
    String systemXcxAppId = this.listSystemXcxAppIds().stream().findFirst().orElse(null);
    if (StringUtil.isBlank(systemXcxAppId)) {
      return;
    }

    Map<String, StudioEmbeddedXcxApply> existMap = studioEmbeddedXcxApplyRpService.mapByStudioIdAndAppId(
        studioId, wxComponentAppId, systemXcxAppId, appIds);
    appIds.stream()
        .filter(appId -> null == existMap.get(appId) || EmbededXcxEnum.isFail(existMap.get(appId).getState()))
        .forEach(appId -> {
          log.info("bind embedded xcx {}, {}", studioId, appId);

          StudioEmbeddedXcxApply entity = existMap.get(appId);
          if (null == entity) {
            entity = StudioEmbeddedXcxConvert.toEntity(studioId, wxComponentAppId, systemXcxAppId, appId);
            studioEmbeddedXcxApplyRpService.insert(entity);
          } else {
            studioEmbeddedXcxApplyRpService.updateById(StudioEmbeddedXcxConvert.toInit(entity));
          }
          try {
            JSONObject result = weXcxService.addEmbeddedXcx(wxComponentAppId, appId, systemXcxAppId,
                "申请半屏小程序");
            Integer code = result.getInteger("errcode");
            String message = result.getString("errmsg");
            if (WX_SUCCESS_CODE.equals(code)) {
              studioEmbeddedXcxApplyRpService.updateById(StudioEmbeddedXcxConvert.toSuccess(entity.getId()));
            } else {
              studioEmbeddedXcxApplyRpService.updateById(
                  StudioEmbeddedXcxConvert.toRefuse(entity.getId(), message));
            }
          } catch (Exception e) {
            log.error("bind embedded xcx fail {}, {}", studioId, appId, e);
            String fail =
                e instanceof BizException ? ((BizException) e).getDefaultErrorMessage() : "网络异常";
            studioEmbeddedXcxApplyRpService.updateById(
                StudioEmbeddedXcxConvert.toRefuse(entity.getId(), fail));
          }
        });
  }

  @Override
  public void syncEmbeddedXcx() {
    String systemAppId = this.listSystemXcxAppIds().stream().findFirst().orElse(null);
    List<AuthXcxInfo> list = this.listBoundAppIds(systemAppId);
    if (CollectionUtil.isEmpty(list)) {
      return;
    }

    List<StudioEmbeddedXcxApply> applies = studioEmbeddedXcxApplyRpService.listByAppIds(
        wxComponentAppId, systemAppId, list.stream().map(AuthXcxInfo::getAppId).collect(Collectors.toList()));
    if (CollectionUtil.isEmpty(applies)) {
      return;
    }

    Map<EmbededXcxEnum, List<AuthXcxInfo>> map = list.stream()
        .collect(Collectors.groupingBy(e -> EmbededXcxEnum.getByWxCode(Integer.parseInt(e.getStatus()))));

    map.keySet().forEach(status -> {
      List<AuthXcxInfo> authXcxInfos = map.getOrDefault(status, Lists.newArrayList());
      if (CollectionUtil.isEmpty(authXcxInfos)) {
        return;
      }
      List<String> appIds = authXcxInfos.stream().map(AuthXcxInfo::getAppId)
          .collect(Collectors.toList());
      switch (status) {
        case SUCCESS:
          List<Long> toSuccessList = StudioEmbeddedXcxConvert.filterAppIds(applies, appIds);
          if (CollectionUtil.isEmpty(toSuccessList)) {
            break;
          }
          studioEmbeddedXcxApplyRpService.updateByIds(StudioEmbeddedXcxConvert.toSuccess(null),
              toSuccessList);
          break;
        case CANCEL:
        case REFUSE:
        case TIMEOUT:
        case REVOKE:
          List<StudioEmbeddedXcxApply> toFailList = StudioEmbeddedXcxConvert.toFailList(status, applies,
              authXcxInfos);
          if (CollectionUtil.isEmpty(toFailList)) {
            break;
          }
          studioEmbeddedXcxApplyRpService.updateBatchById(toFailList);
        default:
          break;
      }
    });
  }

  private List<AuthXcxInfo> listBoundAppIds(String systemAppId) {
    List<AuthXcxInfo> result = new ArrayList<>();

    int num = 100;

    for (int start = 0; start < 10; start++) {
      HalfScreenXcxAuthVO vo = weXcxService.listEmbeddedXcx(wxComponentAppId, systemAppId, start, num);
      if (CollectionUtil.isEmpty(vo.getWxaEmbeddedList())) {
        break;
      }
      result.addAll(vo.getWxaEmbeddedList());
    }
    return result;
  }

  @Override
  public StudioMerchantApplyVO getApply(Long id) {
    StudioMerchantApply apply = studioMerchantApplyRpService.getById(id)
        .orElseThrow(() -> new BizException(FinanceErrorEnums.NOT_EXIST));
    Map<Long, String> urlMap = this.getResouceseUrlMap(apply);
    return MerchantConvert.convertApply(apply, urlMap);
  }

  @Override
  public PageVO<StudioMerchantApplyVO> pageApply(PageStudioMerchantApplyParams params) {
    Page<StudioMerchantApply> page = studioMerchantApplyRpService.page(BeanUtil.map(params,
        PageStudioMerchantApplyQuery.class));
    return PageVO.convert(page, MerchantConvert.convertApplyList(page.getContent()));
  }

  @Override
  public StudioMerchantApplyVO getLatestApply(Long studioId) {
    StudioMerchantApply apply = studioMerchantApplyRpService.getLatestApply(studioId);
    if (null == apply) {
      return null;
    }
    Map<Long, String> urlMap = this.getResouceseUrlMap(apply);
    return MerchantConvert.convertApply(apply, urlMap);
  }

  @Override
  public StudioMerchantBindXcxAppIdVO bindXcxAppId(Long studioId, String appId) {
    PartyToMerchant relation = getRelation(studioId);
    if (null == relation) {
      throw new BizException(FinanceErrorEnums.MERCHANT_NOT_OPEN);
    }
    ConfigSubChannelVO vo = merchantRpcService.bindAppIdWxXcxOffline(config.getCashier(),
        relation.getMerchantId(), appId);

    return StudioMerchantBindXcxAppIdVO.builder().state(vo.getState()).failMessage(vo.getFailMessage())
        .build();
  }

  @Override
  public PartyToMerchant getRelation(Long studioId) {
    List<PartyToMerchant> relations = partyToMerchantRpService.listByParty(studioId, PartyTypeEnum.STUDIO,
        config.getCashier());
    if (CollectionUtil.isEmpty(relations)) {
      return null;
    }
    return relations.get(0);
  }

  @Override
  public StudioMerchantAuthSubChannelVO authSubChannel(Long studioId, SubChannelEnums subChannel,
      SubChannelAuthTypeEnums authType) {
    PartyToMerchant relation = getRelation(studioId);
    if (null == relation) {
      throw new BizException(FinanceErrorEnums.MERCHANT_NOT_OPEN);
    }

    AuthSubChannelVO vo = merchantRpcService.authSubChannel(config.getCashier(), relation.getMerchantId(),
        subChannel, authType);
    return BeanUtil.map(vo, StudioMerchantAuthSubChannelVO.class);
  }

  @Override
  public StudioMerchantAuthSubChannelVO syncAuthSubChannel(Long studioId, SubChannelEnums subChannel,
      SubChannelAuthTypeEnums authType) {
    PartyToMerchant relation = getRelation(studioId);
    if (null == relation) {
      throw new BizException(FinanceErrorEnums.MERCHANT_NOT_OPEN);
    }
    AuthSubChannelVO vo = merchantRpcService.syncAuthSubChannel(relation.getMerchantId(), subChannel,
        authType);
    return BeanUtil.map(vo, StudioMerchantAuthSubChannelVO.class);
  }

  @Override
  public List<StudioMerchantAuthSubChannelVO> listAuthSubChannel(Long studioId) {
    PartyToMerchant relation = getRelation(studioId);
    if (null == relation) {
      throw new BizException(FinanceErrorEnums.MERCHANT_NOT_OPEN);
    }

    MerchantVO merchant = merchantRpcService.getByMerchantId(relation.getMerchantId());
    List<SubChannelAuthVO> auths = merchant.getSubChannelAuths();
    List<SubChannelVO> subChannels = Optional.ofNullable(merchant.getSubChannels()).orElse(new ArrayList<>());
    return Optional.ofNullable(auths).orElse(new ArrayList<>(1))
        .stream()
        .map(e -> {
          StudioMerchantAuthSubChannelVO vo = BeanUtil.map(e, StudioMerchantAuthSubChannelVO.class);
          switch (SubChannelAuthTypeEnums.getByCode(vo.getAuthType())) {
            case ALI_OFFLINE:
              vo.setMerchantNos(subChannels.stream().filter(
                  ele -> SubChannelOpenTypeEnums.ALI_OFFLINE == SubChannelOpenTypeEnums.getByCode(
                      ele.getOpenType())).findFirst().orElse(new SubChannelVO()).getMerchantNos());
              break;
            case WX_OFFLINE:
              vo.setMerchantNos(subChannels.stream().filter(
                  ele -> SubChannelOpenTypeEnums.WX_GZH_OFFLINE == SubChannelOpenTypeEnums.getByCode(
                      ele.getOpenType())).findFirst().orElse(new SubChannelVO()).getMerchantNos());
              break;
            default:
              break;
          }
          return vo;
        })
        .collect(Collectors.toList());
  }

  @Override
  public List<String> checkUnbind(Long studioId) {
    List<PartyToMerchant> relation = partyToMerchantRpService.listByParty(studioId, PartyTypeEnum.STUDIO,
        config.getCashier());
    if (CollectionUtil.isEmpty(relation)) {
      return Lists.newArrayList();
    }
    Long merchantId = relation.stream().findFirst().orElse(new PartyToMerchant()).getMerchantId();
    StudioMerchantApply apply = studioMerchantApplyRpService.getLatestOneSuccessByMerchantId(merchantId);
    if (!Objects.equals(studioId,
        Optional.ofNullable(apply).map(StudioMerchantApply::getStudioId).orElse(null))) {
      return Lists.newArrayList();
    }
    List<PartyToMerchant> merchants = partyToMerchantRpService.listByMerchantIdAndPartyType(merchantId,
        PartyTypeEnum.STUDIO);
    if (CollectionUtil.isEmpty(merchants)) {
      return Lists.newArrayList();
    }
    return studioRpcService.listStudio(
            merchants.stream().map(PartyToMerchant::getPartyId).collect(Collectors.toList()))
        .stream()
        .filter(e -> !e.getId().equals(studioId))
        .map(StudioVO::getName)
        .collect(Collectors.toList());
  }

  @Override
  public StudioMerchantApplyVO getLatestSuccessApply(Long studioId) {
    StudioMerchantApply apply = studioMerchantApplyRpService.getLatestSuccessApply(studioId);
    if (null == apply) {
      return null;
    }
    Map<Long, String> urlMap = this.getResouceseUrlMap(apply);
    return MerchantConvert.convertApply(apply, urlMap);
  }


  private void doOpenMerchantSuccess(MerchantEvent event) {
    StudioMerchantApply apply = studioMerchantApplyRpService.getByIdForUpdate(
        Long.parseLong(event.getApplyNo())).orElseThrow(() -> new BizException(FinanceErrorEnums.NOT_EXIST));
    StudioMerchantApply toModify = MerchantConvert.convertApply(apply, event,
        configService.getDefaultBrandSubChannelInfo().getSubChannels());
    studioMerchantApplyRpService.updateById(toModify);
    if (OpenStateEnums.SUCCESS != OpenStateEnums.getByCode(toModify.getOpenState())) {
      return;
    }
    if (OpenStateEnums.SUCCESS == OpenStateEnums.getByCode(apply.getOpenState())) {
      // 绑定逻辑幂等处理
      return;
    }

    // 绑定商户
    this.bindWhenOpenSuccess(event, apply);

  }

  private void unbindHistoryMerchant(Long studioId) {
    List<StudioMerchantApply> histories = studioMerchantApplyRpService.listSuccessApplyByStudioId(studioId);
    if (CollectionUtil.isEmpty(histories)) {
      return;
    }

    Set<Long> historyMerchantIds = histories.stream().map(StudioMerchantApply::getMerchantId)
        .collect(Collectors.toSet());
    // 将绑定了当前场馆商户号的所有其他场馆/品牌进行解绑
    partyToMerchantRpService.deleteByMerchantIds(historyMerchantIds);

  }

  private void bindWhenOpenSuccess(MerchantEvent event, StudioMerchantApply apply) {

    Long oldMerchantId = Optional.ofNullable(this.getRelation(apply.getStudioId()))
        .map(PartyToMerchant::getMerchantId).orElse(null);

    // 1. 解绑该场馆进件的历史商户所关联的所有其他场馆和品牌
    this.unbindHistoryMerchant(apply.getStudioId());

    // 2. 绑定场馆
    this.bind(apply.getStudioId(), event.getMerchantId(), false);
    // 3. 绑定品牌
    brandMerchantService.bind(apply.getBrandId(), event.getMerchantId());

    // 4. 其他操作
    executor.execute(() -> this.doAfterMerchantSuccess(event, oldMerchantId));

  }


  private void doAfterMerchantSuccess(MerchantEvent event, Long oldMerchantId) {
    StudioMerchantApplyVO apply = this.getApply(Long.parseLong(event.getApplyNo()));

    // 1. 绑定旧appId
    this.bindAppIds(event, apply, oldMerchantId);

    // 2. 短信
    this.sendOpenSuccessMessage(apply);
  }

  private void bindAppIds(MerchantEvent event, StudioMerchantApplyVO apply, Long oldMerchantId) {
    MerchantVO oldMerchant = merchantRpcService.getByMerchantId(oldMerchantId);
    if (null == oldMerchant) {
      return;
    }

    // 场馆的非系统小程序AppId
    List<SubChannelConfigVO> offlineXcxAppIds = this.getOwnOfflineXcxAppIds(oldMerchant);
    if (CollectionUtil.isEmpty(offlineXcxAppIds)) {
      return;
    }

    // 1. 将场馆旧商户已绑定的非系统小程序appId到新商户
    this.rebindMiniAppId(oldMerchant, event.getMerchantId(), offlineXcxAppIds);

    // 2. 将场馆旧商户已绑定的非系统小程序appId绑定系统半屏小程序
    this.bindEmbeddedXcx(offlineXcxAppIds, apply.getStudioId());
  }

  private void bindEmbeddedXcx(List<SubChannelConfigVO> offlineXcxAppIds, Long studioId) {
    // 当小程序为特定失败原因，则需要绑定半屏小程序
    List<String> appIds = offlineXcxAppIds.stream()
        .filter(e -> OpenStateEnums.FAIL.getCode().equals(e.getState()))
        .filter(e -> SPECIAL_FAIL_LIST.contains(e.getFailMessage()))
        .map(SubChannelConfigVO::getAppId).collect(Collectors.toList());
    this.bindEmbeddedXcx(studioId, appIds);
  }

  private List<SubChannelConfigVO> getOwnOfflineXcxAppIds(MerchantVO oldMerchant) {
    List<String> systemXcxAppIds = this.listSystemXcxAppIds();
    return oldMerchant.getSubChannelConfigs().stream()
        .filter(e -> SubChannelConfigTypeEnums.WX_XCX_OFFLINE.name().equals(e.getConfigType()))
        .filter(e -> !systemXcxAppIds.contains(e.getAppId()))
        .collect(Collectors.toList());
  }

  private List<String> listSystemXcxAppIds() {
    return configService.getDefaultBrandSubChannelInfo().getSubChannelConfigs()
        .stream().filter(e -> SubChannelConfigTypeEnums.WX_XCX_OFFLINE == e.getConfigType())
        .map(SubChannelConfigInfo::getAppId).collect(Collectors.toList());
  }

  private void rebindMiniAppId(MerchantVO oldMerchant, Long newMerchantId,
      List<SubChannelConfigVO> offlineXcxAppIds) {
    offlineXcxAppIds.forEach(
        config -> merchantRpcService.bindAppIdWxXcxOffline(oldMerchant.getChannelNo(), newMerchantId,
            config.getAppId()));
  }

  private Map<ResourceTypeEnums, ResourceInfo> upload(ApplyStudioMerchantParams params,
      StudioMerchantApply apply) {

    Map<ResourceTypeEnums, ResourceInfo> resourceMap = MerchantConvert.convertResourceMap(
        params.getResourceMap());

    Map<Long, ResourceInfoVO> resourceIdUrlMap = resourceRpcService.getResourceMap(apply.getBrandId(),
        getResourceIds(resourceMap));

    resourceMap.keySet().forEach(type -> {
      ResourceInfo info = resourceMap.get(type);
      ResourceInfoVO vo = resourceIdUrlMap.get(info.getResourceId());
      String thirdId = merchantRpcService.upload(
          MerchantConvert.convertUploadRequest(config.getCashier(), type, vo));
      info.setThirdId(thirdId);
    });
    return resourceMap;
  }

  private List<Long> getResourceIds(Map<ResourceTypeEnums, ResourceInfo> resourceMap) {
    return resourceMap.values().stream()
        .map(ResourceInfo::getResourceId)
        .filter(Objects::nonNull)
        .distinct()
        .collect(Collectors.toList());
  }

  private List<Long> getResourceIds(MerchantVO merchant) {
    return merchant.getResourceMap().values().stream()
        .map(com.jiejing.paycenter.common.model.vo.ResourceInfoVO::getResourceId)
        .filter(Objects::nonNull)
        .distinct()
        .collect(Collectors.toList());
  }

  private void doOpenMerchantFail(Long id, String failMessage) {
    StudioMerchantApplyVO apply = this.getApply(id);

    if (null == apply) {
      return;
    }
    if (OpenStateEnums.FAIL.getCode().equals(apply.getOpenState())) {
      return;
    }
    // 1. 更新
    studioMerchantApplyRpService.updateById(
        StudioMerchantApply.builder().id(id).openState(OpenStateEnums.FAIL.getCode())
            .openFailMessage(failMessage).build());

    // 2. 短信
    this.sendOpenFailMessage(apply);
  }

  private void sendOpenFailMessage(StudioMerchantApplyVO apply) {
    List<Map<String, Object>> paramList = new ArrayList<>(1);
    Map<String, Object> paramMap = new HashMap<>(1);
    paramMap.put("targetId", apply.getOperatorId());
    paramMap.put("studioId", apply.getStudioId());
    paramList.add(paramMap);
    SendCommonMsgEvent event = new SendCommonMsgEvent();
    event.setChannelEnums(Lists.newArrayList(MsgChannelEnum.SMS));
    event.setCovertTarget(true);
    event.setEventId(IdWorker.getId());
    event.setSourceId(apply.getStudioId());
    event.setBizType("OPEN_MERCHANT_FAIL");
    event.setParams(paramList);
    EventAgent.of(SendCommonMsgEvent.class).triggerEvent(event);
  }

  private void sendOpenSuccessMessage(StudioMerchantApplyVO apply) {
    List<Map<String, Object>> paramList = new ArrayList<>(1);
    Map<String, Object> paramMap = new HashMap<>(1);
    paramMap.put("targetId", apply.getOperatorId());
    paramMap.put("studioId", apply.getStudioId());
    paramList.add(paramMap);
    SendCommonMsgEvent event = new SendCommonMsgEvent();
    event.setChannelEnums(Lists.newArrayList(MsgChannelEnum.SMS));
    event.setCovertTarget(true);
    event.setEventId(IdWorker.getId());
    event.setSourceId(apply.getStudioId());
    event.setBizType("OPEN_MERCHANT_SUCCESS");
    event.setParams(paramList);
    EventAgent.of(SendCommonMsgEvent.class).triggerEvent(event);
  }

  private void doOpenMerchantProcess(Long id) {
    studioMerchantApplyRpService.updateById(
        StudioMerchantApply.builder().id(id).openState(OpenStateEnums.PROCESS.getCode()).build());
  }

  private void validSmsCode(String phone, String smsCode) {
    messageRpcService.validSmsCode(phone, smsCode, "FINANCE_MERCHANT");
  }

  private StudioMerchantApply saveApply(ApplyStudioMerchantParams params) {
    Long id = IdWorker.getId();
    StudioVO studio = studioRpcService.getStudio(params.getStudioId());
    StudioMerchantApply exist = studioMerchantApplyRpService.getLatestOneSuccessByStudioId(studio.getId());
    StudioMerchantApply apply = MerchantConvert.convertApply(id, params, studio, exist, config.getCashier());

    studioMerchantApplyRpService.insert(apply);
    return apply;
  }

  private void checkBeforeApply(ApplyStudioMerchantParams params) {
    // 校验验证码
    if (null == params.getNotCheckSmsCode() || !params.getNotCheckSmsCode()) {
      this.validSmsCode(params.getBankCard().getPhone(), params.getSmsCode());
    }

    // 重复提交
    Integer count = studioMerchantApplyRpService.countProcessByStudioId(params.getStudioId());
    if (count.compareTo(0) > 0) {
      throw new BizException(FinanceErrorEnums.EXIST_PROCESS_APPLY);
    }
  }


  private Map<Long, String> getResouceseUrlMap(StudioMerchantApply apply) {
    List<Long> resourceIds = this.getResourceIds(apply);
    return resourceRpcService.getResourceUrlMap(apply.getBrandId(), resourceIds);
  }

  private List<Long> getResourceIds(StudioMerchantApply apply) {
    ApplyStudioMerchantParams info = JSON.parseObject(apply.getMerchantInfo(),
        ApplyStudioMerchantParams.class);
    return info.getResourceMap().values().stream().map(BrandResourceInfo::getResourceId)
        .collect(Collectors.toList());
  }


}
