package com.jiejing.fitness.finance.service.cashier.impl;

import com.alibaba.fastjson.JSON;
import com.jiejing.common.exception.BizException;
import com.jiejing.common.model.PageVO;
import com.jiejing.common.utils.collection.CollectionUtil;
import com.jiejing.common.utils.convert.BeanUtil;
import com.jiejing.common.utils.text.StringUtil;
import com.jiejing.fitness.finance.api.cashier.vo.StudioCashierRecordVO;
import com.jiejing.fitness.finance.api.cashier.vo.StudioCashierStatisticVO;
import com.jiejing.fitness.finance.api.cashier.vo.StudioSettleVO;
import com.jiejing.fitness.finance.repository.entity.StudioCashierRecord;
import com.jiejing.fitness.finance.repository.entity.StudioSettleRecord;
import com.jiejing.fitness.finance.repository.query.PageBrandCashierRecordQuery;
import com.jiejing.fitness.finance.repository.query.PageStudioSettleQuery;
import com.jiejing.fitness.finance.repository.service.StudioCashierRecordRpService;
import com.jiejing.fitness.finance.repository.service.StudioSettleRecordRpService;
import com.jiejing.fitness.finance.service.cashier.StudioCashierService;
import com.jiejing.fitness.finance.service.cashier.convert.CashierConvert;
import com.jiejing.fitness.finance.service.cashier.params.PageStudioCashierParams;
import com.jiejing.fitness.finance.service.cashier.params.PageStudioSettleParams;
import com.jiejing.fitness.finance.service.enums.FinanceErrorEnums;
import java.util.Date;
import java.util.List;
import javax.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

/**
 * @author chengyubing
 * @since 2024/2/29 11:08
 */
@Slf4j
@Service
public class StudioCashierServiceImpl implements StudioCashierService {

  @Resource
  private StudioCashierRecordRpService studioCashierRecordRpService;

  @Resource
  private StudioSettleRecordRpService studioSettleRecordRpService;

  @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
  @Override
  public void bind(List<Long> ids, String subOrderNo) {
    List<StudioCashierRecord> records = studioCashierRecordRpService.getListByIds(ids);
    if (records.stream().anyMatch(e -> StringUtil.isNotBlank(e.getSubOrderNo()))) {
      throw new BizException(FinanceErrorEnums.EXIST_BIND_RECORD);
    }
    studioCashierRecordRpService.updateByIds(
        StudioCashierRecord.builder().subOrderNo(subOrderNo).updateTime(new Date()).build(), ids);
  }

  @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
  @Override
  public void unbind(List<Long> ids) {
    studioCashierRecordRpService.updateByIds(
        StudioCashierRecord.builder().subOrderNo("").updateTime(new Date()).build(), ids);
  }

  @Override
  public StudioCashierRecordVO getById(Long id) {
    StudioCashierRecord record = studioCashierRecordRpService.getById(id).orElse(null);
    if (null == record) {
      return null;
    }
    return CashierConvert.convertVO(record);
  }

  @Override
  public List<StudioCashierRecordVO> listByOrderNo(String orderNo) {
    List<StudioCashierRecord> list = studioCashierRecordRpService.listByOrderNo(orderNo);
    return CashierConvert.convertList(list);
  }

  @Override
  public List<StudioCashierRecordVO> listBySubOrderNo(String subOrderNo) {
    List<StudioCashierRecord> list = studioCashierRecordRpService.listBySubOrderNo(subOrderNo);
    return CashierConvert.convertList(list);
  }

  @Override
  public PageVO<StudioCashierRecordVO> page(PageStudioCashierParams params) {
    PageBrandCashierRecordQuery query = JSON.parseObject(JSON.toJSONString(params),
        PageBrandCashierRecordQuery.class);
    Page<StudioCashierRecord> page = studioCashierRecordRpService.page(query);
    return CashierConvert.convertPageVO(page);
  }

  @Override
  public PageVO<StudioSettleVO> page(PageStudioSettleParams params) {
    PageStudioSettleQuery query = BeanUtil.map(params, PageStudioSettleQuery.class);
    Page<StudioSettleRecord> page = studioSettleRecordRpService.page(query);
    return CashierConvert.convertSettlePageVO(page);
  }

  @Override
  public StudioCashierStatisticVO statistic(PageStudioCashierParams params) {
    PageBrandCashierRecordQuery query = JSON.parseObject(JSON.toJSONString(params),
        PageBrandCashierRecordQuery.class);
    return studioCashierRecordRpService.statistic(query);
  }

}
