package com.jiejing.fitness.finance.service.merchant.impl;

import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.jiejing.common.exception.BizException;
import com.jiejing.common.model.PageVO;
import com.jiejing.common.utils.convert.BeanUtil;
import com.jiejing.filecenter.api.resource.vo.ResourceInfoVO;
import com.jiejing.fitness.finance.api.merchant.vo.BrandMerchantApplyVO;
import com.jiejing.fitness.finance.api.merchant.vo.BrandMerchantVO;
import com.jiejing.fitness.finance.repository.entity.BrandMerchantApply;
import com.jiejing.fitness.finance.repository.entity.BrandToMerchant;
import com.jiejing.fitness.finance.repository.query.PageBrandMerchantApplyQuery;
import com.jiejing.fitness.finance.repository.service.BrandMerchantApplyRpService;
import com.jiejing.fitness.finance.repository.service.BrandToMerchantRpService;
import com.jiejing.fitness.finance.service.enums.FinanceErrorEnums;
import com.jiejing.fitness.finance.service.merchant.BrandMerchantService;
import com.jiejing.fitness.finance.service.merchant.convert.MerchantConvert;
import com.jiejing.fitness.finance.service.merchant.params.ApplyBrandMerchantParams;
import com.jiejing.fitness.finance.service.merchant.params.PageBrandMerchantApplyParams;
import com.jiejing.fitness.finance.service.rpc.MerchantRpcService;
import com.jiejing.fitness.finance.service.rpc.ResourceRpcService;
import com.jiejing.fitness.finance.service.rpc.StudioRpcService;
import com.jiejing.paycenter.api.merchant.request.ApplyMerchantRequest;
import com.jiejing.paycenter.api.merchant.vo.MerchantVO;
import com.jiejing.paycenter.common.enums.common.OpenStateEnums;
import com.jiejing.paycenter.common.enums.merchant.ResourceTypeEnums;
import com.jiejing.paycenter.common.event.MerchantEvent;
import com.jiejing.paycenter.common.model.ResourceInfo;
import com.jiejing.studio.api.studio.vo.StudioVO;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

/**
 * 品牌商户服务
 *
 * @author chengyubing
 * @since 2024/2/20 14:01
 */
@Slf4j
@Service
public class BrandMerchantServiceImpl implements BrandMerchantService {

  @Value("${finance.brand.merchant.channel}")
  private String channel;

  @Resource
  private StudioRpcService studioRpcService;

  @Resource
  private MerchantRpcService merchantRpcService;

  @Resource
  private ResourceRpcService resourceRpcService;

  @Resource
  private BrandMerchantApplyRpService brandMerchantApplyRpService;

  @Resource
  private BrandToMerchantRpService brandToMerchantRpService;


  @Async(value = "financeThreadPool")
  @Override
  public void apply(ApplyBrandMerchantParams params) {

    Long id = IdWorker.getId();
    StudioVO studio = studioRpcService.getStudio(params.getStudioId());
    BrandMerchantApply exist = brandMerchantApplyRpService.getLatestOneSuccessByBrandId(
        studio.getBrandId());
    BrandMerchantApply apply = MerchantConvert.convertApply(id, params, studio, exist, channel);

    brandMerchantApplyRpService.insert(apply);

    try {

      Map<ResourceTypeEnums, ResourceInfo> resourceMap = this.upload(params);
      ApplyMerchantRequest request = MerchantConvert.convertRequest(channel, apply.getApplyNo(), params,
          resourceMap);
      merchantRpcService.apply(request);

    } catch (Exception e) {
      log.error("apply brand merchant fail {}, brand id = {}, studio id = {}", id, studio.getBrandId(),
          params.getStudioId(), e);
      this.doOpenMerchantFail(id, e.getMessage());
      return;
    }

    this.doOpenMerchantProcess(id);

  }

  @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
  @Override
  public void callback(MerchantEvent event) {
    switch (event.getState()) {
      case PROCESS:
        this.doOpenMerchantProcess(Long.parseLong(event.getApplyNo()));
        break;
      case FAIL:
        this.doOpenMerchantFail(Long.parseLong(event.getApplyNo()), event.getFailMessage());
        break;
      case SUCCESS:
        this.doOpenMerchantSuccess(event);
        break;
      default:
        break;
    }
  }

  @Override
  public BrandMerchantVO getMerchant(Long brandId) {
    BrandToMerchant relation = brandToMerchantRpService.getByBrandId(brandId);
    MerchantVO merchant = merchantRpcService.getByMerchantId(relation.getMerchantId());
    return MerchantConvert.convertBrandMerchant(relation, merchant);
  }

  @Override
  public BrandMerchantApplyVO getApply(Long id) {
    BrandMerchantApply apply = brandMerchantApplyRpService.getById(id)
        .orElseThrow(() -> new BizException(FinanceErrorEnums.NOT_EXIST));
    return MerchantConvert.convertApply(apply);
  }

  @Override
  public PageVO<BrandMerchantApplyVO> pageApply(PageBrandMerchantApplyParams params) {
    Page<BrandMerchantApply> page = brandMerchantApplyRpService.page(BeanUtil.map(params,
        PageBrandMerchantApplyQuery.class));
    return PageVO.convert(page, MerchantConvert.convertApplyList(page.getContent()));
  }

  private void doOpenMerchantSuccess(MerchantEvent event) {
    BrandMerchantApply apply = brandMerchantApplyRpService.getByIdForUpdate(
        Long.parseLong(event.getApplyNo())).orElseThrow(() -> new BizException(FinanceErrorEnums.NOT_EXIST));

    BrandMerchantApply applyToModify = MerchantConvert.convertApply(apply, event);
    if (OpenStateEnums.SUCCESS == applyToModify.getOpenState()) {
      // 所有子通道全部开通成功
      BrandToMerchant exist = brandToMerchantRpService.getByBrandId(apply.getBrandId());
      if (null == exist) {
        brandToMerchantRpService.insert(MerchantConvert.convertMerchant(apply));
      } else {
        brandToMerchantRpService.updateById(MerchantConvert.convertMerchant(apply));
      }
    }
    brandMerchantApplyRpService.updateById(applyToModify);
  }

  private Map<ResourceTypeEnums, ResourceInfo> upload(ApplyBrandMerchantParams params) {

    Map<ResourceTypeEnums, ResourceInfo> resourceMap = MerchantConvert.convertResourceMap(
        params.getResource());

    Map<Long, ResourceInfoVO> resourceIdUrlMap = resourceRpcService.getResourceMap(params.getStudioId(),
        getResourceIds(resourceMap));

    resourceMap.keySet().forEach(type -> {
      ResourceInfo info = resourceMap.get(type);
      ResourceInfoVO vo = resourceIdUrlMap.get(info.getResourceId());
      String thirdId = merchantRpcService.upload(MerchantConvert.convertUploadRequest(channel, type, vo));
      info.setThirdId(thirdId);
    });
    return resourceMap;
  }

  private List<Long> getResourceIds(Map<ResourceTypeEnums, ResourceInfo> resourceMap) {
    return resourceMap.values().stream()
        .map(ResourceInfo::getResourceId)
        .filter(Objects::nonNull)
        .distinct()
        .collect(Collectors.toList());
  }

  private void doOpenMerchantFail(Long id, String failMessage) {
    brandMerchantApplyRpService.updateById(
        BrandMerchantApply.builder().id(id).openState(OpenStateEnums.FAIL)
            .openFailMessage(failMessage).build());
  }

  private void doOpenMerchantProcess(Long id) {
    brandMerchantApplyRpService.updateById(
        BrandMerchantApply.builder().id(id).openState(OpenStateEnums.PROCESS).build());
  }

}
