package com.jiejing.fitness.finance.service.pay.convert;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alipay.api.request.AlipaySystemOauthTokenRequest;
import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.google.common.collect.Lists;
import com.jiejing.common.exception.BizException;
import com.jiejing.common.utils.convert.BeanUtil;
import com.jiejing.common.utils.crypt.AesUtil;
import com.jiejing.common.utils.text.StringUtil;
import com.jiejing.fitness.enums.finance.BrandCashierTransStateEnum;
import com.jiejing.fitness.enums.finance.BrandCashierTransTypeEnum;
import com.jiejing.fitness.finance.repository.entity.MerchantSettleRecord;
import com.jiejing.fitness.finance.repository.entity.StudioCashierRecord;
import com.jiejing.fitness.finance.repository.entity.StudioCheckSettleRecord;
import com.jiejing.fitness.finance.repository.entity.StudioMerchantApply;
import com.jiejing.fitness.finance.repository.entity.StudioSettleRecord;
import com.jiejing.fitness.finance.service.config.PayChannelProperties;
import com.jiejing.fitness.finance.service.enums.FinanceErrorEnums;
import com.jiejing.fitness.finance.service.pay.enums.PayFailMessageReplaceEnums;
import com.jiejing.fitness.finance.service.pay.params.AppPayParams;
import com.jiejing.fitness.finance.service.pay.params.NativePayParams;
import com.jiejing.fitness.finance.service.pay.params.StudioMerchantRefundParams;
import com.jiejing.fitness.finance.service.pay.params.StudioMerchantPayParams;
import com.jiejing.fitness.finance.service.utils.FeeUtil;
import com.jiejing.fitness.finance.service.utils.MoneyUtil;
import com.jiejing.paycenter.common.enums.common.PayChannelEnums;
import com.jiejing.paycenter.common.model.vo.MerchantVO;
import com.jiejing.paycenter.common.model.vo.PayVO;
import com.jiejing.paycenter.common.model.vo.RefundVO;
import com.jiejing.paycenter.common.model.vo.SettleVO;
import com.jiejing.paycenter.common.model.vo.SubChannelVO;
import com.jiejing.paycenter.api.pay.request.PayRequest;
import com.jiejing.paycenter.api.pay.request.RefundPayRequest;
import com.jiejing.paycenter.common.enums.common.TransStateEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelOpenTypeEnums;
import com.jiejing.paycenter.common.enums.pay.PayStateEnums;
import com.jiejing.paycenter.common.enums.pay.PayTypeEnums;
import com.jiejing.paycenter.common.event.PayEvent;
import com.jiejing.paycenter.common.event.RefundEvent;
import com.jiejing.studio.api.studio.vo.StudioVO;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @author chengyubing
 * @since 2024/2/27 11:08
 */
public class PayConvert {

  private static final List<String> ALI_AUTH_CODE_PREFIX = Lists.newArrayList("25", "26", "27", "28", "29",
      "30");

  private static final List<String> WX_AUTH_CODE_PREFIX = Lists.newArrayList("10", "11", "12", "13", "14",
      "15");

  public static PayRequest convert(StudioMerchantPayParams params, StudioCashierRecord record) {
    PayRequest request = JSON.parseObject(JSON.toJSONString(params), PayRequest.class);
    request.setAmount(params.getTransAmount());
    request.setTransNo(record.getTransNo());
    request.setOrderNo(StringUtil.isBlank(record.getOrderNo()) ? record.getTransNo() : record.getOrderNo());
    request.setMerchantId(record.getMerchantId());
    request.setChannelNo(record.getChannelNo());
    request.setExtra(Optional.ofNullable(params.getExtra()).orElse(new JSONObject())
        .fluentPut("studioId", record.getStudioId())
        .fluentPut("buyerName", record.getBuyerName())
        .fluentPut("buyerPhone", record.getBuyerPhone())
        .fluentPut("remark", record.getRemark())
        .fluentPut("subOrderNo", record.getSubOrderNo())
    );
    request.setTradingTime(record.getTradingTime());
    return request;
  }

  public static StudioCashierRecord convertPayInit(StudioMerchantPayParams params, StudioVO studio,
      MerchantVO merchant) {
    setSubChannel(params);
    Long id = IdWorker.getId();
    return StudioCashierRecord.builder()
        .id(id)
        .transNo(id.toString())
        .orderNo(StringUtil.isBlank(params.getOrderNo()) ? "" : params.getOrderNo())
        .businessNo(params.getBusinessNo())
        .subOrderNo(StringUtil.isBlank(params.getSubOrderNo()) ? "" : params.getSubOrderNo())
        .orderType(params.getOrderType().getCode())
        .transType(BrandCashierTransTypeEnum.PAY.getCode())
        .brandId(studio.getBrandId())
        .studioId(params.getStudioId())
        .studioName(studio.getName())
        .merchantId(merchant.getId())
        .merchantNo(merchant.getMerchantNo())
        .channelNo(merchant.getChannelNo())
        .subChannel(params.getSubChannel().getCode())
        .transAmount(params.getTransAmount())
        .transState(BrandCashierTransStateEnum.PAY_INIT.getCode())
        .buyerName(params.getBuyerName())
        .buyerPhone(params.getBuyerPhone())
        .goods(params.getGoods())
        .remark(params.getRemark())
        .existRelatedTrans(false)
        .extra(JSON.toJSONString(Optional.ofNullable(params.getExtra()).orElse(new JSONObject())))
        .tradingTime(new Date())
        .createTime(new Date())
        .updateTime(new Date())
        .build();
  }

  private static void setSubChannel(StudioMerchantPayParams params) {
    if (PayTypeEnums.BARCODE != params.getPayType() || StringUtil.isBlank(params.getAuthCode())) {
      // 未设置子渠道，不是付款码支付或付款码为空，直接返回
      return;
    }

    // https://juejin.cn/post/7099730102203727902
    // 支付宝支付码生成规则：25 - 30开头的长度为16~24位的数字，实际字符串长度以开发者获取的付款码长度为准
    if (ALI_AUTH_CODE_PREFIX.stream().anyMatch(prefix -> params.getAuthCode().startsWith(prefix))) {
      params.setSubChannel(SubChannelEnums.ALI);
    }
    // 微信支付码生成规则：18位纯数字，以10、11、12、13、14、15开头
    if (WX_AUTH_CODE_PREFIX.stream().anyMatch(prefix -> params.getAuthCode().startsWith(prefix))) {
      params.setSubChannel(SubChannelEnums.WX);
    }
  }

  private static BigDecimal getFeeRate(StudioMerchantPayParams params, MerchantVO merchant) {
    SubChannelOpenTypeEnums scene;
    switch (params.getPayType()) {
      case MINI:
        scene = SubChannelEnums.WX == params.getSubChannel() ? SubChannelOpenTypeEnums.WX_XCX_OFFLINE
            : SubChannelOpenTypeEnums.ALI_OFFLINE;
        break;
      case JS:
        scene = SubChannelEnums.WX == params.getSubChannel() ? SubChannelOpenTypeEnums.WX_GZH_OFFLINE
            : SubChannelOpenTypeEnums.ALI_OFFLINE;
        break;
      case BARCODE:
        scene = SubChannelEnums.WX == params.getSubChannel() ? SubChannelOpenTypeEnums.WX_BARCODE
            : SubChannelOpenTypeEnums.ALI_OFFLINE;
        break;
      default:
        throw new BizException(FinanceErrorEnums.NOT_SUPPORT_TYPE);
    }
    SubChannelVO subChannelInfo = merchant.getSubChannels().stream()
        .filter(e -> scene.name().equals(e.getOpenType()))
        .findFirst()
        .orElseThrow(() -> new BizException(FinanceErrorEnums.CHANNEL_NOT_OPEN));
    return subChannelInfo.getFeeRate();
  }

  public static void main(String[] args) {
    BigDecimal feeRate = new BigDecimal("0.38");
    BigDecimal fee = MoneyUtil.defaultRound(MoneyUtil.multiply(
        MoneyUtil.divide(feeRate, new BigDecimal("100"), 4, RoundingMode.HALF_UP.ordinal()),
        new BigDecimal("168")));
    System.out.println(fee);
  }

  public static StudioCashierRecord convertPay(StudioCashierRecord record, PayEvent event) {
    return StudioCashierRecord.builder()
        .id(record.getId())
        .successTime(event.getFinishTime())
        .channelNo(event.getChannelNo())
        .transState(convertTransState(PayStateEnums.getByCode(event.getPayState())).getCode())
        .failMessage(event.getFailMsg())
        .thirdTransNo(event.getThirdTransNo())
        .updateTime(new Date())
        .build();
  }

  public static BrandCashierTransStateEnum convertTransState(PayStateEnums state) {
    switch (state) {
      case SUCCESS:
        return BrandCashierTransStateEnum.PAY_SUCCESS;
      case CLOSED:
      case FAILED:
        return BrandCashierTransStateEnum.PAY_FAIL;
      default:
        return BrandCashierTransStateEnum.PAYING;
    }
  }

  public static BrandCashierTransStateEnum convertTransState(String state) {
    return convertTransState(PayStateEnums.getByCode(state));
  }

  public static BrandCashierTransStateEnum convertTransState(TransStateEnums state) {
    switch (state) {
      case SUCCESS:
        return BrandCashierTransStateEnum.REFUND_SUCCESS;
      case FAIL:
        return BrandCashierTransStateEnum.REFUND_FAIL;
      case PROCESS:
      case INIT:
      default:
        return BrandCashierTransStateEnum.REFUNDING;
    }
  }


  public static PayEvent convertEvent(PayRequest request, PayVO vo) {
    return PayEvent.builder()
        .id(Long.parseLong(request.getTransNo()))
        .amount(request.getAmount())
        .transNo(request.getTransNo())
        .channelNo(request.getChannelNo())
        .merchantId(request.getMerchantId())
        .payState(vo.getPayState())
        .thirdTransNo(vo.getThirdTransNo())
        .failMsg(vo.getFailMsg())
        .finishTime(vo.getFinishTime())
        .extra(request.getExtra())
        .build();
  }

  public static PayRequest convertNativePay(NativePayParams params, PayChannelProperties config) {
    PayRequest req = new PayRequest();
    req.setChannelNo(getChannelNo(params.getChannel(), config));
    req.setTransNo(IdWorker.getIdStr());
    req.setAmount(params.getAmount());
    req.setPayType(PayTypeEnums.NATIVE);
    req.setGoods(params.getGoods());
    req.setOrderNo(params.getOrderNo());
    req.setOrderType(params.getOrderType());
    req.setTimeExpire(params.getTimeExpire());
    req.setTradingTime(new Date());
    req.setExtra(params.getExtra());
    return req;
  }

  public static PayRequest convertAppPay(AppPayParams params, PayChannelProperties config) {
    PayRequest req = new PayRequest();
    req.setChannelNo(getChannelNo(params.getChannel(), config));
    req.setTransNo(IdWorker.getIdStr());
    req.setAmount(params.getAmount());
    req.setPayType(PayTypeEnums.APP);
    req.setGoods(params.getGoods());
    req.setOrderNo(params.getOrderNo());
    req.setOrderType(params.getOrderType());
    req.setTimeExpire(params.getTimeExpire());
    req.setTradingTime(new Date());
    req.setExtra(params.getExtra());
    return req;
  }

  private static String getChannelNo(PayChannelEnums channel, PayChannelProperties config) {
    switch (channel) {
      case WX:
        return config.getWxApp();
      case ALI:
        return config.getAliApp();
      default:
        throw new BizException(FinanceErrorEnums.NOT_SUPPORT_TYPE);
    }
  }

  public static AlipaySystemOauthTokenRequest convertAlipaySystemOauthTokenRequest(String authCode) {
    AlipaySystemOauthTokenRequest aliRequest = new AlipaySystemOauthTokenRequest();
    aliRequest.setCode(authCode);
    aliRequest.setGrantType("authorization_code");
    return aliRequest;
  }

  public static FinanceErrorEnums replaceFailMessage(String failMsg) {
    return PayFailMessageReplaceEnums.convertBySource(failMsg);
  }

}
