package com.jiejing.fitness.finance.service.axf.convert;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.google.common.collect.Lists;
import com.jiejing.common.enums.ConfirmEnum;
import com.jiejing.common.model.JsonResult;
import com.jiejing.common.utils.convert.BeanUtil;
import com.jiejing.common.utils.crypt.AesUtil;
import com.jiejing.fitness.enums.trade.VoucherStateEnum;
import com.jiejing.fitness.finance.api.axf.enums.AxfOrderSignStateEnums;
import com.jiejing.fitness.finance.api.axf.enums.AxfStateEnums;
import com.jiejing.fitness.finance.api.axf.request.*;
import com.jiejing.fitness.finance.api.axf.vo.StudioAuthTokenVO;
import com.jiejing.fitness.finance.api.axf.vo.StudioAxfApplyVO;
import com.jiejing.fitness.finance.api.axf.vo.StudioAxfCommodityVO;
import com.jiejing.fitness.finance.api.axf.vo.StudioAxfOrderDeductionVO;
import com.jiejing.fitness.finance.api.axf.vo.StudioAxfOrderRecoveryVO;
import com.jiejing.fitness.finance.api.axf.vo.StudioAxfOrderVO;
import com.jiejing.fitness.finance.repository.entity.*;
import com.jiejing.fitness.finance.service.axf.constans.AxfConst;
import com.jiejing.fitness.finance.service.axf.impl.AxfServiceImpl;
import com.jiejing.fitness.finance.service.axf.model.AlipayCommerceMerchantcardDeductionOrderNotifyModel;
import com.jiejing.fitness.finance.service.axf.model.AlipayCommerceMerchantcardOrderNotifyModel;
import com.jiejing.fitness.finance.service.utils.MoneyUtil;
import com.jiejing.member.api.member.vo.MemberDetailVO;
import com.jiejing.paycenter.api.merchant.request.ApplyMerchantRequest;
import com.jiejing.paycenter.api.merchant.request.CreateCommodityRequest;
import com.jiejing.paycenter.api.merchant.request.CreateCommodityTemplateRequest;
import com.jiejing.paycenter.api.merchant.request.OpenStoreRequest;
import com.jiejing.paycenter.common.enums.common.OpenStateEnums;
import com.jiejing.paycenter.common.enums.merchant.CertTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.CommodityOrderDeductionStateEnums;
import com.jiejing.paycenter.common.enums.merchant.CommodityTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.CompanyTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.CycleChargeTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.CycleTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.ExpireTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.LicenseTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.ResourceTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelOpenTypeEnums;
import com.jiejing.paycenter.common.event.CommodityOrderEvent;
import com.jiejing.paycenter.common.event.OrderDeductionEvent;
import com.jiejing.paycenter.common.event.StoreEvent;
import com.jiejing.paycenter.common.model.request.Address;
import com.jiejing.paycenter.common.model.request.BusinessInfo;
import com.jiejing.paycenter.common.model.request.Contact;
import com.jiejing.paycenter.common.model.request.CycleCommodityAttr;
import com.jiejing.paycenter.common.model.request.CycleCommodityAttr.PeriodPrice;
import com.jiejing.paycenter.common.model.request.Legal;
import com.jiejing.paycenter.common.model.request.License;
import com.jiejing.paycenter.common.model.request.ResourceInfo;
import com.jiejing.paycenter.common.model.request.SubChannelInfo;
import com.jiejing.paycenter.common.model.vo.AuthTokenVO;
import com.jiejing.paycenter.common.model.vo.CommodityVO;
import com.jiejing.paycenter.common.model.vo.CreateCommodityTemplateVO;
import com.jiejing.trade.api.voucher.vo.VoucherVO;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.commons.lang3.time.DateUtils;

/**
 * @author chengyubing
 * @since 2024/9/7 11:58
 */
public class AxfConvert {

  public static StudioAxfApply toEntity(StudioAxfApplyRequest request, Function<Long, String> studioFunc,
      Function<Long, String> adminFunc) {
    String salt = AesUtil.getSalt(8);
    StudioAxfApply apply = BeanUtil.map(request, StudioAxfApply.class);
    apply.setId(IdWorker.getId());
    apply.setLicenseType(LicenseTypeEnums.NATIONAL_LEGAL_MERGE.getCode());
    apply.setLicenseExpireType(request.getLicenseExpireType().getCode());
    apply.setCompanyType(request.getCompanyType().getCode());
    apply.setState(AxfStateEnums.INIT.getCode());
    apply.setChannelNo(AxfConst.CHANNEL_NO);
    apply.setApplyType(AxfConst.OPEN);
    apply.setOpId(request.getOperatorId());
    apply.setOpName(adminFunc.apply(request.getOperatorId()));
    apply.setStudioName(studioFunc.apply(request.getStudioId()));
    apply.setSalt(salt);
    apply.setCreateTime(new Date());
    apply.setUpdateTime(new Date());
    return encrypt(apply);
  }

  public static StudioAxfApply encrypt(StudioAxfApply apply) {
    StudioAxfApply encrypt = BeanUtil.map(apply, StudioAxfApply.class);
    encrypt.setLegalCertNo(AesUtil.encrypt(apply.getSalt(), apply.getLegalCertNo()));
    encrypt.setLicenseCode(AesUtil.encrypt(apply.getSalt(), apply.getLicenseCode()));
    encrypt.setCardNo(AesUtil.encrypt(apply.getSalt(), apply.getCardNo()));
    return encrypt;
  }

  public static void main(String[] args) {
    System.out.println(AesUtil.encrypt("J7t5sz4Z", "397176538@qq.com"));
  }

  public static StudioAxfApply decrypt(StudioAxfApply apply) {
    StudioAxfApply decrypt = BeanUtil.map(apply, StudioAxfApply.class);
    decrypt.setLegalCertNo(AesUtil.decrypt(apply.getSalt(), apply.getLegalCertNo()));
    decrypt.setLicenseCode(AesUtil.decrypt(apply.getSalt(), apply.getLicenseCode()));
    decrypt.setCardNo(AesUtil.decrypt(apply.getSalt(), apply.getCardNo()));
    return decrypt;
  }

  public static Map<ResourceTypeEnums, ResourceInfo> toResourceMap(StudioAxfApplyRequest request,
      Function<List<Long>, Map<Long, String>> fileFunc) {
    Map<ResourceTypeEnums, ResourceInfo> map = new HashMap<>(2);
    Map<Long, String> fileMap = fileFunc.apply(Lists.newArrayList(
        request.getLicenseResourceId(), request.getStoreHeadResourceId()
    ));
    map.put(ResourceTypeEnums.LICENSE,
        ResourceInfo.builder()
            .resourceId(request.getLicenseResourceId())
            .thirdId(fileMap.get(request.getLicenseResourceId()))
            .type(ResourceTypeEnums.LICENSE)
            .build());
    map.put(ResourceTypeEnums.STORE_HEADER_PIC,
        ResourceInfo.builder()
            .resourceId(request.getStoreHeadResourceId())
            .thirdId(fileMap.get(request.getStoreHeadResourceId()))
            .type(ResourceTypeEnums.STORE_HEADER_PIC)
            .build());
    return map;
  }

  public static ApplyMerchantRequest toRequest(StudioAxfApply apply,
      Map<ResourceTypeEnums, ResourceInfo> resourceMap) {
    return ApplyMerchantRequest.builder()
        .bizCode(AxfConst.BIZ_CODE)
        .merchantNo(apply.getMerchantNo())
        .applyNo(apply.getId().toString())
        .channelNo(AxfConst.CHANNEL_NO)
        .companyType(CompanyTypeEnums.getByCode(apply.getCompanyType()))
        .merchantName(apply.getMerchantName())
        .shortName(apply.getShortName())
        .legal(convertLegal(apply))
        .license(convertLicense(apply))
        .business(convertBusiness(apply))
        .contact(convertContact(apply))
        .resourceMap(resourceMap)
        .subChannels(Lists.newArrayList(
            SubChannelInfo.builder()
                .subChannel(SubChannelEnums.ALI)
                .openType(SubChannelOpenTypeEnums.AXF_MERCHANT_PERIOD_PAY)
                .feeRate(AxfConst.RATE)
                .build()
        ))
        .build();
  }

  private static Contact convertContact(StudioAxfApply apply) {
    return Contact.builder().contactPhone(apply.getPhone()).build();
  }

  private static Legal convertLegal(StudioAxfApply params) {
    return Legal.builder()
        .legalName(params.getLegalName())
        .legalCertNo(params.getLegalCertNo())
        .legalCertType(CertTypeEnums.ID_CARD)
        .build();
  }

  private static License convertLicense(StudioAxfApply params) {
    return License.builder()
        .licenseCode(params.getLicenseCode())
        .licenseType(LicenseTypeEnums.valueOf(params.getLicenseType()))
        .licenseExpireType(ExpireTypeEnums.valueOf(params.getLicenseExpireType()))
        .licenseStartDate(params.getLicenseStartTime())
        .licenseEndDate(params.getLicenseEndTime())
        .build();
  }

  private static BusinessInfo convertBusiness(StudioAxfApply params) {
    return BusinessInfo.builder()
        .businessAddress(Address.builder()
            .province(params.getBusinessProvinceCode())
            .city(params.getBusinessCityCode())
            .district(params.getBusinessDistrictCode())
            .address(params.getBusinessAddress())
            .build())
        .build();
  }

  public static OpenStoreRequest toStoreRequest(Long merchantId, SubChannelOpenTypeEnums openType) {
    return OpenStoreRequest.builder().merchantId(merchantId).openType(openType).build();
  }

  public static CreateCommodityTemplateRequest toCommodityRequest(StudioAxfApply apply, StoreEvent event,
      String commodityName, Boolean chargeNow, CycleTypeEnums cycleType, Integer cycleValue) {
    int totalPeriod = 10;
    BigDecimal eachOriginalPrice = new BigDecimal("600");
    BigDecimal eachSalePrice = new BigDecimal("500");
    BigDecimal totalOriginalPrice = eachOriginalPrice.multiply(new BigDecimal(totalPeriod));
    BigDecimal totalSalePrice = eachSalePrice.multiply(new BigDecimal(totalPeriod));
    List<PeriodPrice> periodPriceList = new ArrayList<>(totalPeriod);
    for (int i = 0; i < totalPeriod; i++) {
      periodPriceList.add(
          PeriodPrice.builder().period(i + 1).originalPrice(eachOriginalPrice).salePrice(eachSalePrice)
              .build());
    }
    return CreateCommodityTemplateRequest.builder()
        .merchantId(event.getMerchantId())
        .storeId(event.getId())
        .commodityType(CommodityTypeEnums.AXF_MERCHANT_PERIOD_PAY)
        .commodityName(commodityName)
        .commodityNote(commodityName)
        .originalPrice(totalOriginalPrice)
        .salePrice(totalSalePrice)
        .saleStartTime(apply.getCreateTime())
        .saleEndTime(DateUtils.addDays(apply.getCreateTime(), 365))
        .cycleCommodityAttr(CycleCommodityAttr.builder()
            .totalPeriod(totalPeriod)
            .cycleType(cycleType)
            .cycleChargeType(CycleChargeTypeEnums.FIXED_DATE)
            .cycleValue(cycleValue)
            .chargeNow(chargeNow)
            .periodPriceList(periodPriceList)
            .build())
        .build();
  }

  public static StudioAxfCommodityTemplate toTemplate(StudioAxfApply apply,
      JsonResult<CreateCommodityTemplateVO> result, CreateCommodityTemplateRequest request) {
    OpenStateEnums state = result.getSuccess() ? result.getResult().getState() : OpenStateEnums.FAIL;
    String failMessage = result.getSuccess() ? result.getResult().getFailMessage() : "网络异常";
    return StudioAxfCommodityTemplate.builder()
        .id(IdWorker.getId())
        .studioId(apply.getStudioId())
        .merchantId(apply.getMerchantId())
        .templateName(request.getCommodityName())
        .state(state.getCode())
        .templateId(state == OpenStateEnums.SUCCESS ? result.getResult().getTemplateId() : null)
        .thirdTemplateId(state == OpenStateEnums.SUCCESS ? result.getResult().getThirdTemplateId() : null)
        .totalPeriod(request.getCycleCommodityAttr().getTotalPeriod())
        .chargeNow(request.getCycleCommodityAttr().getChargeNow())
        .cycleType(request.getCycleCommodityAttr().getCycleType().getCode())
        .cycleValue(request.getCycleCommodityAttr().getCycleValue())
        .saleStartTime(request.getSaleStartTime())
        .saleEndTime(request.getSaleEndTime())
        .cycleChargeType(request.getCycleCommodityAttr().getCycleChargeType().getCode())
        .attr(JSON.toJSONString(request))
        .failMessage(failMessage)
        .createTime(new Date())
        .updateTime(new Date())
        .build();
  }

  public static StudioAxfCommodityTemplateApply toTemplateApply(StudioAxfApply apply,
      JsonResult<CreateCommodityTemplateVO> result, CreateCommodityTemplateRequest request) {
    OpenStateEnums state = result.getSuccess() ? result.getResult().getState() : OpenStateEnums.FAIL;
    String failMessage = result.getSuccess() ? result.getResult().getFailMessage() : "网络异常";
    return StudioAxfCommodityTemplateApply.builder()
        .id(IdWorker.getId())
        .studioId(apply.getStudioId())
        .merchantId(apply.getMerchantId())
        .templateName(request.getCommodityName())
        .state(state.getCode())
        .templateId(state == OpenStateEnums.SUCCESS ? result.getResult().getTemplateId() : null)
        .thirdTemplateId(state == OpenStateEnums.SUCCESS ? result.getResult().getThirdTemplateId() : null)
        .totalPeriod(request.getCycleCommodityAttr().getTotalPeriod())
        .chargeNow(request.getCycleCommodityAttr().getChargeNow())
        .cycleType(request.getCycleCommodityAttr().getCycleType().getCode())
        .cycleValue(request.getCycleCommodityAttr().getCycleValue())
        .saleStartTime(request.getSaleStartTime())
        .saleEndTime(request.getSaleEndTime())
        .cycleChargeType(request.getCycleCommodityAttr().getCycleChargeType().getCode())
        .attr(JSON.toJSONString(request))
        .failMessage(failMessage)
        .createTime(new Date())
        .updateTime(new Date())
        .build();
  }

  public static List<StudioAxfApplyVO> toList(List<StudioAxfApply> records) {
    if (null == records) {
      return Lists.newArrayList();
    }
    return records.stream().map(e -> toVO(e, vo -> {
    })).collect(Collectors.toList());
  }

  public static StudioAxfApplyVO toVO(StudioAxfApply apply, Consumer<StudioAxfApplyVO> consumer) {
    if (null == apply) {
      return null;
    }
    StudioAxfApply decrypt = AxfConvert.decrypt(apply);
    StudioAxfApplyVO vo = BeanUtil.map(decrypt, StudioAxfApplyVO.class);
    vo.setFailMessage(AxfServiceImpl.replaceFailMessage(vo.getFailMessage()));
    consumer.accept(vo);
    return vo;
  }

  public static CreateCommodityRequest toRequest(CreateStudioAxfCommodityRequest request,
      StudioAxfCommodityTemplate template) {
    CreateCommodityTemplateRequest attr = JSON.parseObject(template.getAttr(),
        CreateCommodityTemplateRequest.class);
    return CreateCommodityRequest.builder()
        .merchantId(template.getMerchantId())
        .bizId(request.getVoucherId())
        .templateId(template.getTemplateId())
        .commodityName(template.getTemplateName())
        .originalPrice(request.getOriginalPrice())
        .salePrice(request.getSalePrice())
        .saleStartTime(attr.getSaleStartTime())
        .saleEndTime(attr.getSaleEndTime())
        .cycleCommodityAttr(request.getCycleCommodityAttr())
        .build();
  }

  public static StudioAxfOrder toEntity(CommodityOrderEvent event, String merchantNo, CommodityVO commodity,
      VoucherVO voucher, MemberDetailVO member) {
    AlipayCommerceMerchantcardOrderNotifyModel model = JSON.parseObject(event.getAttr(),
        AlipayCommerceMerchantcardOrderNotifyModel.class);
    return StudioAxfOrder.builder()
        .id(IdWorker.getId())
        .channelNo(event.getChannelNo())
        .studioId(voucher.getSupplierId())
        .merchantId(event.getMerchantId())
        .merchantNo(merchantNo)
        .commodityId(commodity.getId())
        .memberId(member.getId())
        .memberName(member.getName())
        .originalPrice(MoneyUtil.divide(new BigDecimal(model.getOriginPriceTotal()), new BigDecimal("100")))
        .salePrice(MoneyUtil.divide(new BigDecimal(model.getSalePriceTotal()), new BigDecimal("100")))
        .totalPeriod(Integer.parseInt(model.getTotalCount()))
        .signState(AxfOrderSignStateEnums.SIGN.getCode())
        .receivedAmount(BigDecimal.ZERO)
        .receivedFee(BigDecimal.ZERO)
        .receivedActualAmount(BigDecimal.ZERO)
        .receivedPeriod(0)
        .rate(AxfConst.RATE)
        .voucherId(commodity.getBizId())
        .voucherBusinessNo(
            VoucherStateEnum.CANCEL.name().equals(voucher.getState()) ? null : voucher.getBusinessNo())
        .thirdOrderNo(event.getThirdOrderNo())
        .attr(JSON.toJSONString(commodity))
        .signTime(event.getCreateTime())
        .createTime(new Date())
        .updateTime(new Date())
        .build();
  }

  public static StudioAxfOrderVO toVO(StudioAxfOrder order) {
    StudioAxfOrderVO vo = BeanUtil.map(order, StudioAxfOrderVO.class);
    if (null != order.getAttr()) {
      StudioAxfCommodityVO commodity = JSON.parseObject(order.getAttr(), StudioAxfCommodityVO.class);
      vo.setCommodity(commodity);
    }
    return vo;
  }

  public static StudioAxfOrderDeduction toDeduction(OrderDeductionEvent event, StudioAxfOrderVO order,
      AlipayCommerceMerchantcardDeductionOrderNotifyModel model, Integer recoveryTimes) {
    CycleCommodityAttr commodityAttr = order.getCommodity().getCycleCommodityAttr();
    Integer period = Integer.parseInt(model.getPeriod());
    PeriodPrice periodPrice = commodityAttr.getPeriodPriceList().stream()
        .filter(e -> e.getPeriod().equals(period))
        .findFirst().orElse(new PeriodPrice());
    BigDecimal fee = MoneyUtil.divide(MoneyUtil.multiply(periodPrice.getSalePrice(), AxfConst.RATE),
        new BigDecimal("100"));
    return StudioAxfOrderDeduction.builder()
        .id(event.getId())
        .orderId(order.getId())
        .period(period)
        .amount(periodPrice.getSalePrice())
        .fee(fee)
        .actualAmount(MoneyUtil.subtract(periodPrice.getSalePrice(), fee))
        .rate(AxfConst.RATE)
        .state(event.getState().getCode())
        .failMessage(event.getFailMessage())
        .transNo(event.getId().toString())
        .thirdOrderNo(event.getThirdOrderNo())
        .thirdTransNo(model.getTradeNo())
        .recoveryTimes(recoveryTimes)
        .deductionTime(CommodityOrderDeductionStateEnums.SUCCESS == event.getState() ? new Date() : null)
        .createTime(new Date())
        .updateTime(new Date())
        .build();
  }

  public static StudioAxfOrderRecovery toRecovery(OrderDeductionEvent event, StudioAxfOrderVO order,
      StudioAxfOrderDeduction deduction, AlipayCommerceMerchantcardDeductionOrderNotifyModel model) {
    return StudioAxfOrderRecovery.builder()
        .id(IdWorker.getId())
        .deductionId(deduction.getId())
        .orderId(order.getId())
        .period(Integer.parseInt(model.getPeriod()))
        .amount(deduction.getAmount())
        .actualAmount(deduction.getActualAmount())
        .fee(deduction.getFee())
        .rate(deduction.getRate())
        .state(event.getState().getCode())
        .failMessage(event.getFailMessage())
        .transNo(event.getId().toString())
        .thirdOrderNo(event.getThirdOrderNo())
        .thirdTransNo(model.getTradeNo())
        .deductionTime(event.getState() == CommodityOrderDeductionStateEnums.SUCCESS ? new Date() : null)
        .createTime(new Date())
        .updateTime(new Date())
        .build();
  }

  public static List<StudioAxfOrderDeductionVO> toDeductionVO(List<StudioAxfOrderDeduction> deductions) {
    return deductions.stream().map(AxfConvert::toDeductionVO).collect(Collectors.toList());
  }

  public static StudioAxfOrderDeductionVO toDeductionVO(StudioAxfOrderDeduction deduction) {
    StudioAxfOrderDeductionVO vo = BeanUtil.map(deduction, StudioAxfOrderDeductionVO.class);
    vo.setActualAmount(
        Objects.equals(CommodityOrderDeductionStateEnums.SUCCESS.getCode(), vo.getState()) ? vo.getActualAmount() : null);
    return vo;
  }

  public static List<StudioAxfOrderRecoveryVO> toRecoveryVO(List<StudioAxfOrderRecovery> recoveries) {
    return recoveries.stream().map(AxfConvert::toDeductionVO).collect(Collectors.toList());
  }

  public static StudioAxfOrderRecoveryVO toDeductionVO(StudioAxfOrderRecovery recovery) {
    return BeanUtil.map(recovery, StudioAxfOrderRecoveryVO.class);
  }

  public static StudioAuthTokenVO toAuth(Long id, AuthTokenVO result, ConfirmEnum used) {
    StudioAuthTokenVO vo = BeanUtil.map(result, StudioAuthTokenVO.class);
    vo.setUsed(used);
    vo.setId(id);
    return vo;
  }

  public static StudioAuthTokenVO toAuth(StudioAxfAuth exist) {
    StudioAuthTokenVO vo = BeanUtil.map(exist, StudioAuthTokenVO.class);
    vo.setUsed(ConfirmEnum.valueOf(exist.getUsed()));
    return vo;
  }

}
