package com.jiejing.fitness.finance.service.pay.impl;

import static java.math.BigDecimal.ZERO;
import static java.util.stream.Collectors.toList;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.jiejing.common.utils.collection.CollectionUtil;
import com.jiejing.common.utils.time.TimeUtil;
import com.jiejing.fitness.enums.finance.BrandCashierTransStateEnum;
import com.jiejing.fitness.finance.repository.entity.MerchantSettleRecord;
import com.jiejing.fitness.finance.repository.entity.StudioCashierRecord;
import com.jiejing.fitness.finance.repository.entity.StudioMerchantApply;
import com.jiejing.fitness.finance.repository.service.MerchantSettleRecordRpService;
import com.jiejing.fitness.finance.repository.service.StudioCashierRecordRpService;
import com.jiejing.fitness.finance.repository.service.StudioCheckSettleRecordRpService;
import com.jiejing.fitness.finance.repository.service.StudioMerchantApplyRpService;
import com.jiejing.fitness.finance.repository.service.StudioSettleRecordRpService;
import com.jiejing.fitness.finance.service.pay.SettleService;
import com.jiejing.fitness.finance.service.pay.convert.SettleConvert;
import com.jiejing.fitness.finance.service.rpc.PayRpcService;
import com.jiejing.fitness.finance.service.rpc.StudioRpcService;
import com.jiejing.fitness.finance.service.utils.DingUtil;
import com.jiejing.fitness.finance.service.utils.MoneyUtil;
import com.jiejing.paycenter.common.enums.common.TransStateEnums;
import com.jiejing.paycenter.common.model.vo.SettleVO;
import com.jiejing.studio.api.studio.vo.StudioVO;
import java.math.BigDecimal;
import java.time.temporal.ChronoUnit;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import javax.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.support.TransactionTemplate;

/**
 * @author chengyubing
 * @since 2024/5/7 16:33
 */
@Slf4j
@Service
public class SettleServiceImpl implements SettleService {


  @Value("${spring.profiles.active}")
  private String env;

  @Resource
  private PayRpcService payRpcService;

  @Resource
  private StudioRpcService studioRpcService;

  @Resource
  private StudioCashierRecordRpService studioCashierRecordRpService;

  @Resource
  private StudioSettleRecordRpService studioSettleRecordRpService;

  @Resource
  private StudioMerchantApplyRpService studioMerchantApplyRpService;

  @Resource
  private StudioCheckSettleRecordRpService studioCheckSettleRecordRpService;

  @Resource
  private TransactionTemplate transactionTemplate;

  @Resource
  private MerchantSettleRecordRpService merchantSettleRecordRpService;

  @Override
  public void checkSettle(Long merchantId, Date settleDate) {
    Date endDate = null == settleDate ? TimeUtil.local().startOfDay(new Date()) : settleDate;
    AtomicInteger failCount = new AtomicInteger(0);
    this.pageAndConsumer(merchantId, 200, apply -> {
      // 同步历史处理中的结算记录
      this.syncHistorySettle(apply.getMerchantId(), endDate);

      // 重复处理，直接return
      if (this.repeat(apply, endDate)) {
        return;
      }

      // 获取开始日期
      Date startDate = this.getStartDate(merchantId, settleDate);

      // 获取三方结算记录
      SettleVO vo = payRpcService.syncSettle(apply.getMerchantId(), endDate);
      // 获取本地待结算金额
      BigDecimal totalAmount = this.getWaitSettleAmount(apply, startDate, endDate);
      if (ZERO.compareTo(vo.getTransAmount()) == 0 && ZERO.compareTo(totalAmount) == 0) {
        return;
      }

      if (vo.getTransAmount().compareTo(totalAmount) == 0) {
        // 对账成功
        this.doCheckSettleSuccess(vo, apply, startDate, endDate);
      } else {
        // 对账失败
        this.doCheckSettleFail(failCount, vo, apply, totalAmount);
      }
    });

    // 钉钉消息
    this.doAfterCheckSettle(failCount);
  }

  @Override
  public void syncSettle(Long merchantId, Date settleDate) {
    Date endDate = null == settleDate ? TimeUtil.local().startOfDay(new Date()) : settleDate;
    this.pageAndConsumer(merchantId, 200, apply -> this.syncHistorySettle(apply.getMerchantId(), endDate));
  }

  private void syncHistorySettle(Long merchantId, Date endDate) {
    List<MerchantSettleRecord> list = merchantSettleRecordRpService.listInitAndProcessByMerchantIdAndBeforeOrEqualEndDate(
        merchantId, endDate);
    if (CollectionUtil.isEmpty(list)) {
      return;
    }

    for (int i = 0; i < list.size(); i++) {
      MerchantSettleRecord history = list.get(i);
      SettleVO vo = payRpcService.syncSettle(merchantId, history.getSettleDate());
      if (vo == null) {
        return;
      }

      TransStateEnums state = TransStateEnums.getByCode(vo.getTransState());
      if (TransStateEnums.INIT == state || TransStateEnums.PROCESS == state) {
        return;
      }

      transactionTemplate.executeWithoutResult(action -> {
        merchantSettleRecordRpService.updateById(SettleConvert.convertMerchantSettle(history, vo));
        studioSettleRecordRpService.updateByParentId(SettleConvert.convertStudioSettle(history, vo));
      });
      if (TransStateEnums.SUCCESS == state) {
        this.updatePayIn(merchantId, null, history.getSettleDate());
      }
    }

  }

  private void updatePayIn(Long merchantId, Date startTime, Date endTime) {
    Integer current = 0;
    Integer size = 500;

    do {
      Page<StudioCashierRecord> page = studioCashierRecordRpService.pageMerchantPaySuccess(merchantId,
          startTime, endTime, current, size);
      if (CollectionUtil.isEmpty(page.getContent())) {
        break;
      }

      try {
        List<Long> ids = page.getContent().stream().map(StudioCashierRecord::getId).collect(toList());
        studioCashierRecordRpService.updateByIds(
            StudioCashierRecord.builder().transState(BrandCashierTransStateEnum.PAY_IN.getCode())
                .inTime(endTime).updateTime(new Date()).build(), ids);
      } finally {
        current++;
      }

      if (!page.hasNext()) {
        break;
      }
    } while (true);
  }

  public void pageAndConsumer(Long merchantId, Integer batch, Consumer<StudioMerchantApply> consumer) {
    Long minId = 0L;
    log.info("consumer merchant apply start");
    do {
      List<StudioMerchantApply> list = studioMerchantApplyRpService.listByMinId(minId, merchantId, batch);
      if (CollectionUtil.isEmpty(list)) {
        break;
      }
      try {

        list.forEach(apply -> {
          try {
            log.info("start consumer merchant apply {}, {}", apply.getMerchantId(), apply.getMerchantNo());
            consumer.accept(apply);
            log.info("end consumer merchant apply {}, {}", apply.getMerchantId(), apply.getMerchantNo());
          } catch (Exception e) {
            log.error("consumer apply fail apply {}, {}", apply.getMerchantId(), apply.getMerchantNo(), e);
          }
        });

      } finally {
        minId = list.get(list.size() - 1).getId();
      }
    } while (true);
    log.info("consumer merchant apply finished");
  }

  /**
   * 计算待结算金额。对于给定的商户申请，在指定日期范围内，计算其成功支付金额减去成功退款金额的差额作为待结算金额。
   *
   * @param apply     商户申请信息，用于获取商户ID。
   * @param startDate 计算范围的开始日期。
   * @param endDate   计算范围的结束日期。
   * @return BigDecimal类型的待结算金额。
   */
  private BigDecimal getWaitSettleAmount(StudioMerchantApply apply, Date startDate, Date endDate) {
    // 计算指定日期范围内商户的成功支付总额
    BigDecimal totalPayAmount = studioCashierRecordRpService.sumMerchantPaySuccess(apply.getMerchantId(),
        startDate, endDate);
    // 计算指定日期范围内商户的成功退款总额
    BigDecimal totalRefundAmount = studioCashierRecordRpService.sumMerchantRefundSuccess(
        apply.getMerchantId(), startDate, endDate);
    // 没有钱包，退款金额只能从入账中扣，因此收款的钱必须大于等于退款的钱
    BigDecimal totalAmount = MoneyUtil.subtract(totalPayAmount, totalRefundAmount);
    log.info("merchant {} local total amount is {}, total pay amount is {}, total refund amount is {}",
        apply.getMerchantId(), totalAmount, totalPayAmount, totalRefundAmount);
    return totalAmount;
  }


  private void doCheckSettleSuccess(SettleVO vo, StudioMerchantApply apply, Date startDate, Date endDate) {
    // 对账成功：存在AB两个场馆同时绑定一个商户，A场馆没有收款但发起退款的场景。
    Map<Long, BigDecimal> studioPayAmountMap = studioCashierRecordRpService.sumMerchantPaySuccessGroupByStudioId(
        apply.getMerchantId(), startDate, endDate);
    Map<Long, BigDecimal> studioRefundAmountMap = studioCashierRecordRpService.sumMerchantRefundSuccessGroupByStudioId(
        apply.getMerchantId(), startDate, endDate);

    Set<Long> studioIds = Sets.newHashSet();
    studioIds.addAll(studioPayAmountMap.keySet());
    studioIds.addAll(studioRefundAmountMap.keySet());

    Map<Long, StudioVO> studioMap = studioRpcService.mapStudio(Lists.newArrayList(studioIds));
    transactionTemplate.executeWithoutResult(action -> {
      MerchantSettleRecord record = SettleConvert.convertMerchantSettle(apply, vo);
      merchantSettleRecordRpService.insert(record);
      studioSettleRecordRpService.insertAll(
          SettleConvert.convertStudioSettle(record, studioPayAmountMap, studioRefundAmountMap, studioMap));
    });
    if (TransStateEnums.SUCCESS == TransStateEnums.getByCode(vo.getTransState())) {
      // 结算成功，则更新收银流水状态为记录为入账成功
      this.updatePayIn(apply.getMerchantId(), startDate, endDate);
    }
  }

  private boolean repeat(StudioMerchantApply apply, Date endDate) {
    MerchantSettleRecord exist = merchantSettleRecordRpService.getByMerchantIdAndSettleDate(
        apply.getMerchantId(), endDate);
    if (null != exist) {
      return true;
    }
    return false;
  }

  /**
   * 对账失败处理
   */
  private void doCheckSettleFail(AtomicInteger failCount, SettleVO vo, StudioMerchantApply apply, BigDecimal totalAmount) {
    failCount.incrementAndGet();
    studioCheckSettleRecordRpService.insert(SettleConvert.convertCheckSettle(apply, vo, totalAmount));
  }

  private void doAfterCheckSettle(AtomicInteger failCount) {
    if (failCount.get() > 0) {
      DingUtil.sendCheckSettleFail(env, failCount.get());
    } else {
      DingUtil.sendCheckSettleSuccess(env);
    }
  }

  private Date getStartDate(Long merchantId, Date settleDate) {
    // 获取开始日期
    MerchantSettleRecord history = merchantSettleRecordRpService.getLatestRecordBeforeSettleDate(merchantId,
        settleDate);
    if (null != history) {
      return history.getSettleDate();
    }
    return TimeUtil.local().plus(settleDate, -15, ChronoUnit.DAYS);
  }

}
