package com.jiejing.fitness.finance.service.pay.impl;

import static com.jiejing.fitness.enums.finance.BrandCashierTransStateEnum.PAY_FAIL;
import static com.jiejing.fitness.enums.finance.BrandCashierTransStateEnum.PAY_IN;
import static com.jiejing.fitness.enums.finance.BrandCashierTransStateEnum.PAY_SUCCESS;
import static com.jiejing.fitness.enums.finance.BrandCashierTransStateEnum.REFUND_FAIL;
import static com.jiejing.fitness.enums.finance.BrandCashierTransStateEnum.REFUND_SUCCESS;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.google.common.collect.Lists;
import com.jiejing.common.exception.BizException;
import com.jiejing.common.utils.collection.CollectionUtil;
import com.jiejing.common.utils.convert.BeanUtil;
import com.jiejing.common.utils.text.StringUtil;
import com.jiejing.common.utils.time.TimeUtil;
import com.jiejing.fitness.enums.auth.AuthDomainEnum;
import com.jiejing.fitness.enums.config.ObjectTypeEnum;
import com.jiejing.fitness.enums.finance.BrandCashierTransStateEnum;
import com.jiejing.fitness.event.finance.CashierEvent;
import com.jiejing.fitness.finance.api.merchant.vo.StudioMerchantCheckRefundVO;
import com.jiejing.fitness.finance.api.merchant.vo.StudioMerchantCheckRefundVO.CheckRefundCodeEnum;
import com.jiejing.fitness.finance.api.merchant.vo.StudioMerchantRefundVO;
import com.jiejing.fitness.finance.repository.entity.PartyToMerchant;
import com.jiejing.fitness.finance.repository.entity.StudioCashierRecord;
import com.jiejing.fitness.finance.repository.service.StudioCashierRecordRpService;
import com.jiejing.fitness.finance.service.config.AppUrlProperties;
import com.jiejing.fitness.finance.service.enums.FinanceErrorEnums;
import com.jiejing.fitness.finance.service.global.ConfigService;
import com.jiejing.fitness.finance.service.global.dto.RefundConfigDTO;
import com.jiejing.fitness.finance.service.global.dto.RefundConfigDTO.RefuseTime;
import com.jiejing.fitness.finance.service.merchant.StudioMerchantService;
import com.jiejing.fitness.finance.service.pay.RefundService;
import com.jiejing.fitness.finance.service.pay.convert.RefundConvert;
import com.jiejing.fitness.finance.service.pay.params.StudioMerchantRefundParams;
import com.jiejing.fitness.finance.service.rpc.ConfigRpcService;
import com.jiejing.fitness.finance.service.rpc.PayRpcService;
import com.jiejing.fitness.finance.service.rpc.PermissionRpcService;
import com.jiejing.fitness.finance.service.rpc.StudioRpcService;
import com.jiejing.fitness.finance.service.utils.MoneyUtil;
import com.jiejing.message.enums.MsgChannelEnum;
import com.jiejing.message.event.SendCommonMsgEvent;
import com.jiejing.paycenter.api.pay.request.RefundPayRequest;
import com.jiejing.paycenter.common.enums.common.TransStateEnums;
import com.jiejing.paycenter.common.event.RefundEvent;
import com.jiejing.paycenter.common.model.vo.RefundVO;
import com.xiaomai.event.EventAgent;
import java.math.BigDecimal;
import java.time.temporal.ChronoUnit;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.support.TransactionTemplate;

/**
 * @author chengyubing
 * @since 2024/5/7 16:37
 */
@Slf4j
@Service
public class RefundServiceImpl implements RefundService {

  @Resource
  private PayRpcService payRpcService;

  @Resource
  private StudioMerchantService studioMerchantService;

  @Resource
  private StudioCashierRecordRpService studioCashierRecordRpService;

  @Resource
  private TransactionTemplate transactionTemplate;

  @Resource
  private PermissionRpcService permissionRpcService;

  @Resource
  private StudioRpcService studioRpcService;

  @Resource
  private AppUrlProperties appUrlProperties;

  @Resource
  private ConfigService configService;

  @Resource
  private ConfigRpcService configRpcService;

  @Resource(name = "financeThreadPool")
  private Executor executor;

  @Override
  public StudioMerchantCheckRefundVO checkBeforeMerchantRefund(StudioMerchantRefundParams params) {
    Long payId = Long.parseLong(params.getPayTransNo());
    StudioCashierRecord pay = studioCashierRecordRpService.getById(payId)
        .orElseThrow(() -> new BizException(FinanceErrorEnums.NOT_EXIST));
    RefundConfigDTO config = configService.getRefundConfig();
    // 不是当前商户号收款，金额无法线上原路退回。如仍需退款请私下处理，系统会产生新的收支
    if (!this.isCurrentMerchant(params.getStudioId(), pay.getMerchantId())) {
      return RefundConvert.convertRefundVO(CheckRefundCodeEnum.NOT_CURRENT_MERCHANT);
    }
    // 仅支持350天内交易进行退款，金额无法线上原路退回。如仍需退款请私下处理，系统会产生新的收支
    if (this.isDateLimit(pay.getSuccessTime(), config.getMaxDate())) {
      return RefundConvert.convertRefundVO(CheckRefundCodeEnum.DATE_LIMIT);
    }
    // 当前时间段不支持退款，金额无法线上原路退回。如仍需退款请私下处理，系统会产生新的收支
    if (this.isTimeLimit(new Date(), config.getRefuseTimeList())) {
      return RefundConvert.convertRefundVO(CheckRefundCodeEnum.TIME_LIMIT);
    }
    // 乐动收银可退金额不足
    if (this.isAmountLimit(params.getTransAmount(), pay)) {
      return RefundConvert.convertRefundVO(CheckRefundCodeEnum.AMOUNT_LIMIT);
    }
    return RefundConvert.convertRefundVO(CheckRefundCodeEnum.SUCCESS);
  }

  @Override
  public StudioMerchantRefundVO merchantRefund(StudioMerchantRefundParams params) {

    StudioMerchantCheckRefundVO checkResult = this.checkBeforeMerchantRefund(params);

    StudioCashierRecord refund = this.initRefundRecord(params);
    if (CheckRefundCodeEnum.isFail(checkResult.getCode())) {
      this.refundCallback(RefundConvert.convertRefundEvent(refund, checkResult));
      return StudioMerchantRefundVO.builder()
          .transNo(refund.getTransNo())
          .failCode(checkResult.getCode())
          .refundState(TransStateEnums.FAIL.getCode())
          .failMessage(checkResult.getFailMessage())
          .build();
    }

    RefundPayRequest request = RefundConvert.convert(params, refund);
    RefundVO vo = payRpcService.refund(request);

    this.refundCallback(RefundConvert.convertRefundEvent(refund, vo));

    return RefundConvert.convertRefundVO(refund, vo);
  }

  @Override
  public void refundCallback(RefundEvent event) {
    StudioCashierRecord record = studioCashierRecordRpService.getById(Long.parseLong(event.getTransNo()))
        .orElse(null);
    if (null == record) {
      return;
    }

    BrandCashierTransStateEnum originalState = BrandCashierTransStateEnum.valueOf(record.getTransState());
    BrandCashierTransStateEnum targetState = RefundConvert.convertTransState(event.getRefundState());
    if (targetState == originalState) {
      return;
    }

    StudioCashierRecord toModify = RefundConvert.convertRefund(record, event);
    studioCashierRecordRpService.updateById(toModify);

    // 发布收银事件
    this.triggerCashierEvent(record.getId());

    // 执行后续处理
    this.doAfterRefund(event, record);
  }

  private void triggerCashierEvent(Long id) {
    // 发布收银事件
    StudioCashierRecord record = studioCashierRecordRpService.getById(id).orElse(null);
    if (null == record) {
      return;
    }
    if (Lists.newArrayList(REFUND_FAIL.getCode(), REFUND_SUCCESS.getCode())
        .contains(record.getTransState())) {
      EventAgent.of(CashierEvent.class).triggerEvent(BeanUtil.map(record, CashierEvent.class));
    }
  }

  private void doAfterRefund(RefundEvent event, StudioCashierRecord record) {
    TransStateEnums state = TransStateEnums.getByCode(event.getRefundState());
    switch (state) {
      case SUCCESS:
        executor.execute(() -> {
          this.sendRefundMessage(event, record, "CASHIER_REFUND_SUCCESS");
          this.sendRefundMessageToMember(event, record, "MEMBER_REFUND_SUCCESS");
        });
        break;
      case FAIL:
        executor.execute(() -> this.sendRefundMessage(event, record, "CASHIER_REFUND_FAIL"));
        break;
      default:
        break;
    }

  }

  private void sendRefundMessageToMember(RefundEvent e, StudioCashierRecord record, String bizType) {
    if (StringUtil.isBlank(e.getExtra())) {
      return;
    }
    JSONObject extra = JSON.parseObject(e.getExtra());
    Boolean notice = extra.getBoolean("notice_member");
    if (null == notice || !notice) {
      return;
    }

    Boolean wx = configRpcService.getStudioBoolConfig(record.getStudioId(), "WECHAT_MEMBER_REFUND_SUCCESS");
    Boolean sms = configRpcService.getStudioBoolConfig(record.getStudioId(), "SMS_MEMBER_REFUND_SUCCESS");

    log.info(
        "refund {} send member message config value WECHAT_MEMBER_REFUND_SUCCESS : {}, SMS_MEMBER_REFUND_SUCCESS : {}",
        e.getId(), wx, sms);

    String targetId;
    Long memberId = extra.getLong("notice_member_id");
    List<MsgChannelEnum> channels = sms ? Lists.newArrayList(MsgChannelEnum.SMS) : Lists.newArrayList();
    if (null != memberId) {
      targetId = memberId.toString();
      if (wx) {
        channels.add(MsgChannelEnum.WE_CHAT);
      }
    } else {
      targetId = record.getBuyerPhone();
    }
    if (CollectionUtil.isEmpty(channels)) {
      log.info("all config is false {}, {}", wx, sms);
      return;
    }

    Map<String, Object> paramMap = new HashMap<>(1);
    paramMap.put("targetId", targetId);
    paramMap.put("studioId", record.getStudioId());
    paramMap.put("name", record.getBuyerName());
    paramMap.put("refundAmount", record.getTransAmount());
    paramMap.put("refundCommodity", record.getRemark());
    paramMap.put("refundReason", record.getRemark());
    paramMap.put("time", DateFormatUtils.format(record.getSuccessTime(), "yyyy-MM-dd HH:mm:ss"));
    List<Map<String, Object>> paramList = Lists.newArrayList(paramMap);

    SendCommonMsgEvent event = new SendCommonMsgEvent();
    event.setChannelEnums(channels);
    event.setCovertTarget(true);
    event.setEventId(IdWorker.getId());
    event.setSourceId(record.getStudioId());
    event.setBizType(bizType);
    event.setParams(paramList);
    log.info("send message MEMBER_REFUND_SUCCESS {}", JSON.toJSONString(event));
    EventAgent.of(SendCommonMsgEvent.class).triggerEvent(event);
  }

  private void sendRefundMessage(RefundEvent e, StudioCashierRecord record, String bizType) {
    List<Long> targetIds = this.getAdminIds(record.getStudioId());
    if (CollectionUtil.isEmpty(targetIds)) {
      return;
    }
    List<Map<String, Object>> paramList = targetIds.stream().map(targetId -> {
      Map<String, Object> paramMap = new HashMap<>(1);
      paramMap.put("targetId", targetId);
      paramMap.put("studioId", record.getStudioId());
      paramMap.put("refunderName", record.getBuyerName());
      paramMap.put("amount", record.getTransAmount());
      paramMap.put("appUrl", appUrlProperties.getTransDetail() + e.getTransNo());
      return paramMap;
    }).collect(Collectors.toList());

    SendCommonMsgEvent event = new SendCommonMsgEvent();
    event.setChannelEnums(Lists.newArrayList(MsgChannelEnum.APP_PUSH));
    event.setCovertTarget(true);
    event.setEventId(IdWorker.getId());
    event.setSourceId(record.getStudioId());
    event.setBizType(bizType);
    event.setParams(paramList);
    EventAgent.of(SendCommonMsgEvent.class).triggerEvent(event);
  }

  private List<Long> getAdminIds(Long studioId) {
    List<Long> userIds = this.getUserIds(studioId);
    if (CollectionUtil.isEmpty(userIds)) {
      return Lists.newArrayList();
    }
    return studioRpcService.listAdminIdsByUserIds(studioId, Lists.newArrayList(userIds));
  }

  private List<Long> getUserIds(Long studioId) {
    // 有乐动收银查看和操作权限的账户
    return permissionRpcService.getUserIdsByCodeList(AuthDomainEnum.FITNESS_ADMIN.getCode(), studioId,
        Lists.newArrayList("FitSeeXmPay", "FitManageXmPay"), false);
  }

  private StudioCashierRecord initRefundRecord(StudioMerchantRefundParams params) {
    return transactionTemplate.execute(action -> {
      Long payId = Long.parseLong(params.getPayTransNo());
      StudioCashierRecord pay = studioCashierRecordRpService.getById(payId)
          .orElseThrow(() -> new BizException(FinanceErrorEnums.NOT_EXIST));

      BigDecimal historyRefundActualAmount = studioCashierRecordRpService.sumRefundActualAmountByPayTransNo(
          params.getPayTransNo());

      StudioCashierRecord record = RefundConvert.convertRefundInit(params, pay, historyRefundActualAmount);
      studioCashierRecordRpService.insert(record);
      studioCashierRecordRpService.updateById(
          StudioCashierRecord.builder().id(payId).existRelatedTrans(true).updateTime(new Date()).build());
      return record;
    });
  }

  private boolean isCurrentMerchant(Long studioId, Long merchantId) {
    PartyToMerchant merchant = studioMerchantService.getRelation(studioId);
    if (null == merchant) {
      return false;
    }
    return merchant.getMerchantId().equals(merchantId);
  }

  private boolean isDateLimit(Date successTime, Long maxDate) {
    return TimeUtil.local().diff(new Date(), successTime, ChronoUnit.DAYS) > maxDate;
  }

  private boolean isTimeLimit(Date now, List<RefuseTime> refuseTimeList) {
    int minute = TimeUtil.local().getMinuteOfDay(now);
    return refuseTimeList.stream().anyMatch(refuseTime ->
        minute >= refuseTime.getStart() && minute <= refuseTime.getEnd());
  }

  private boolean isAmountLimit(BigDecimal refundAmount, StudioCashierRecord pay) {
    BigDecimal historyRefundAmount = studioCashierRecordRpService.sumRefundTransAmountByPayTransNo(
        pay.getTransNo());
    BigDecimal leftTransAmount = MoneyUtil.subtract(pay.getTransAmount(), historyRefundAmount);
    return refundAmount.compareTo(leftTransAmount) > 0;
  }

}
