package com.jiejing.fitness.finance.service.pay.impl;

import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.google.common.collect.Lists;
import com.jiejing.common.exception.BizException;
import com.jiejing.common.utils.collection.CollectionUtil;
import com.jiejing.fitness.enums.auth.AuthDomainEnum;
import com.jiejing.fitness.enums.finance.BrandCashierTransStateEnum;
import com.jiejing.fitness.finance.repository.entity.StudioCashierRecord;
import com.jiejing.fitness.finance.repository.service.StudioCashierRecordRpService;
import com.jiejing.fitness.finance.service.config.AppUrlProperties;
import com.jiejing.fitness.finance.service.enums.FinanceErrorEnums;
import com.jiejing.fitness.finance.service.pay.RefundService;
import com.jiejing.fitness.finance.service.pay.convert.RefundConvert;
import com.jiejing.fitness.finance.service.pay.params.StudioMerchantRefundParams;
import com.jiejing.fitness.finance.service.rpc.PayRpcService;
import com.jiejing.fitness.finance.service.rpc.PermissionRpcService;
import com.jiejing.fitness.finance.service.rpc.StudioRpcService;
import com.jiejing.message.enums.MsgChannelEnum;
import com.jiejing.message.event.SendCommonMsgEvent;
import com.jiejing.paycenter.api.pay.request.RefundPayRequest;
import com.jiejing.paycenter.common.enums.common.TransStateEnums;
import com.jiejing.paycenter.common.event.RefundEvent;
import com.jiejing.paycenter.common.model.vo.RefundVO;
import com.xiaomai.event.EventAgent;
import java.math.BigDecimal;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.support.TransactionTemplate;

/**
 * @author chengyubing
 * @since 2024/5/7 16:37
 */
@Slf4j
@Service
public class RefundServiceImpl implements RefundService {

  @Resource
  private PayRpcService payRpcService;

  @Resource
  private StudioCashierRecordRpService studioCashierRecordRpService;

  @Resource
  private TransactionTemplate transactionTemplate;

  @Resource
  private PermissionRpcService permissionRpcService;

  @Resource
  private StudioRpcService studioRpcService;

  @Resource
  private AppUrlProperties appUrlProperties;

  @Resource(name = "financeThreadPool")
  private Executor executor;

  @Override
  public RefundVO merchantRefund(StudioMerchantRefundParams params) {
    StudioCashierRecord refund = transactionTemplate.execute(action -> {
      Long payId = Long.parseLong(params.getPayTransNo());
      StudioCashierRecord pay = studioCashierRecordRpService.getById(payId)
          .orElseThrow(() -> new BizException(FinanceErrorEnums.NOT_EXIST));

      BigDecimal historyRefundActualAmount = studioCashierRecordRpService.sumRefundActualAmountByPayTransNo(
          params.getPayTransNo());

      StudioCashierRecord record = RefundConvert.convertRefundInit(params, pay, historyRefundActualAmount);
      studioCashierRecordRpService.insert(record);
      return record;
    });

    RefundPayRequest request = RefundConvert.convert(params, refund);
    RefundVO vo = payRpcService.refund(request);

    StudioCashierRecord toModify = RefundConvert.convertRefund(refund, vo);
    studioCashierRecordRpService.updateById(toModify);
    return vo;
  }

  @Override
  public void refundCallback(RefundEvent event) {
    StudioCashierRecord record = studioCashierRecordRpService.getById(Long.parseLong(event.getTransNo()))
        .orElse(null);
    if (null == record) {
      return;
    }

    BrandCashierTransStateEnum originalState = BrandCashierTransStateEnum.valueOf(record.getTransState());
    BrandCashierTransStateEnum targetState = RefundConvert.convertTransState(event.getRefundState());
    if (targetState == originalState) {
      return;
    }

    StudioCashierRecord toModify = RefundConvert.convertRefund(record, event);
    studioCashierRecordRpService.updateById(toModify);

    // 执行后续处理
    this.doAfterRefund(event, record);
  }

  private void doAfterRefund(RefundEvent event, StudioCashierRecord record) {
    if (!TransStateEnums.SUCCESS.getCode().equals(event.getRefundState())) {
      return;
    }
    executor.execute(() -> this.sendRefundSuccessMessage(event, record));
  }

  private void sendRefundSuccessMessage(RefundEvent e, StudioCashierRecord record) {
    List<Long> targetIds = this.getAdminIds(record.getStudioId());
    if (CollectionUtil.isEmpty(targetIds)) {
      return;
    }

    List<Map<String, Object>> paramList = targetIds.stream().map(targetId -> {
      Map<String, Object> paramMap = new HashMap<>(1);
      paramMap.put("targetId", targetId);
      paramMap.put("studioId", record.getStudioId());
      paramMap.put("buyerName", record.getBuyerName());
      paramMap.put("amount", record.getTransAmount());
      paramMap.put("appUrl", appUrlProperties.getTransDetail() + e.getTransNo());
      return paramMap;
    }).collect(Collectors.toList());

    SendCommonMsgEvent event = new SendCommonMsgEvent();
    event.setChannelEnums(Lists.newArrayList(MsgChannelEnum.APP_PUSH));
    event.setCovertTarget(true);
    event.setEventId(IdWorker.getId());
    event.setSourceId(record.getStudioId());
    event.setBizType("CASHIER_REFUND_SUCCESS");
    event.setParams(paramList);
    EventAgent.of(SendCommonMsgEvent.class).triggerEvent(event);
  }

  private List<Long> getAdminIds(Long studioId) {
    List<Long> userIds = this.getUserIds(studioId);
    if (CollectionUtil.isEmpty(userIds)) {
      return Lists.newArrayList();
    }
    return studioRpcService.listAdminIdsByUserIds(studioId, Lists.newArrayList(userIds));
  }

  private List<Long> getUserIds(Long studioId) {
    // 有乐动收银查看和操作权限的账户
    return permissionRpcService.getUserIdsByCodeList(AuthDomainEnum.FITNESS_ADMIN.getCode(), studioId,
        Lists.newArrayList("FitSeeXmPay", "FitManageXmPay"), false);
  }

}
