package com.jiejing.fitness.finance.service.pay.convert;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.jiejing.common.exception.BizException;
import com.jiejing.common.utils.convert.BeanUtil;
import com.jiejing.fitness.enums.finance.BrandCashierTransStateEnum;
import com.jiejing.fitness.enums.finance.BrandCashierTransTypeEnum;
import com.jiejing.fitness.finance.api.merchant.vo.StudioMerchantCheckRefundVO;
import com.jiejing.fitness.finance.api.merchant.vo.StudioMerchantRefundVO;
import com.jiejing.fitness.finance.api.merchant.vo.StudioMerchantCheckRefundVO.CheckRefundCodeEnum;
import com.jiejing.fitness.finance.repository.entity.StudioCashierRecord;
import com.jiejing.fitness.finance.service.enums.FinanceErrorEnums;
import com.jiejing.fitness.finance.service.pay.params.StudioMerchantRefundParams;
import com.jiejing.fitness.finance.service.utils.FeeUtil;
import com.jiejing.fitness.finance.service.utils.MoneyUtil;
import com.jiejing.paycenter.api.pay.request.RefundPayRequest;
import com.jiejing.paycenter.common.enums.common.TransStateEnums;
import com.jiejing.paycenter.common.event.RefundEvent;
import com.jiejing.paycenter.common.model.vo.RefundVO;
import java.math.BigDecimal;
import java.util.Date;
import java.util.Optional;

/**
 * @author chengyubing
 * @since 2024/5/7 16:48
 */
public class RefundConvert {

  public static StudioCashierRecord convertRefundInit(StudioMerchantRefundParams params,
      StudioCashierRecord pay, BigDecimal historyRefundActualAmount) {
    Long id = IdWorker.getId();
    BigDecimal fee = FeeUtil.calculateRefundFee(params.getTransAmount(), pay.getTransAmount(),
        pay.getActualAmount(), pay.getFee(), historyRefundActualAmount);
    BigDecimal actualAmount = MoneyUtil.subtract(params.getTransAmount(), fee);
    Date now = new Date();
    return StudioCashierRecord.builder()
        .id(id)
        .transNo(id.toString())
        .orderType(params.getOrderType().getCode())
        .orderNo(params.getOrderNo())
        .businessNo(params.getBusinessNo())
        .subOrderNo(params.getSubOrderNo())
        .transType(BrandCashierTransTypeEnum.REFUND.getCode())
        .brandId(pay.getBrandId())
        .studioId(pay.getStudioId())
        .studioName(pay.getStudioName())
        .merchantId(pay.getMerchantId())
        .merchantNo(pay.getMerchantNo())
        .channelNo(pay.getChannelNo())
        .subChannel(pay.getSubChannel())
        .transAmount(params.getTransAmount())
        .feeRate(pay.getFeeRate())
        .fee(fee)
        .actualAmount(actualAmount)
        .transState(BrandCashierTransStateEnum.REFUND_INIT.getCode())
        .buyerName(pay.getBuyerName())
        .buyerPhone(pay.getBuyerPhone())
        .goods(pay.getGoods())
        .remark(params.getRefundReason())
        .existRelatedTrans(true)
        .relatedTransNo(params.getPayTransNo())
        .extra(Optional.ofNullable(params.getExtra()).map(e -> JSON.toJSONString(e)).orElse("{}"))
        .tradingTime(now)
        .createTime(now)
        .updateTime(now)
        .build();
  }

  public static RefundPayRequest convert(StudioMerchantRefundParams params, StudioCashierRecord record) {
    RefundPayRequest request = BeanUtil.map(params, RefundPayRequest.class);
    request.setTransNo(record.getTransNo());
    return request;
  }

  public static StudioCashierRecord convertRefund(StudioCashierRecord record, RefundVO vo) {
    return StudioCashierRecord.builder()
        .id(record.getId())
        .transState(PayConvert.convertTransState(TransStateEnums.getByCode(vo.getRefundState())).getCode())
        .failMessage(vo.getFailMessage())
        .successTime(vo.getSuccessTime())
        .updateTime(new Date())
        .build();
  }

  public static StudioCashierRecord convertRefund(StudioCashierRecord record, RefundEvent event) {
    return StudioCashierRecord.builder()
        .id(record.getId())
        .transState(PayConvert.convertTransState(TransStateEnums.getByCode(event.getRefundState())).getCode())
        .failMessage(event.getFailMessage())
        .successTime(event.getSuccessTime())
        .updateTime(new Date())
        .build();
  }

  public static BrandCashierTransStateEnum convertTransState(String refundState) {
    TransStateEnums state = TransStateEnums.getByCode(refundState);
    switch (state) {
      case SUCCESS:
        return BrandCashierTransStateEnum.REFUND_SUCCESS;
      case FAIL:
        return BrandCashierTransStateEnum.REFUND_FAIL;
      case INIT:
      case PROCESS:
        return BrandCashierTransStateEnum.REFUNDING;
      default:
        throw new BizException(FinanceErrorEnums.NOT_SUPPORT_TYPE);
    }
  }

  public static StudioCashierRecord convertRefundFail(StudioCashierRecord record,
      StudioMerchantRefundVO checkResult) {
    return StudioCashierRecord.builder()
        .id(record.getId())
        .transState(BrandCashierTransStateEnum.REFUND_FAIL.getCode())
        .failMessage(checkResult.getFailMessage())
        .updateTime(new Date())
        .build();
  }

  public static StudioMerchantRefundVO convertRefundVO(StudioCashierRecord refund, RefundVO vo) {
    return StudioMerchantRefundVO.builder()
        .thirdTransNo(vo.getThirdTransNo())
        .transNo(refund.getTransNo())
        .failMessage(vo.getFailMessage())
        .refundState(vo.getRefundState())
        .successTime(vo.getSuccessTime())
        .build();
  }

  public static StudioMerchantCheckRefundVO convertRefundVO(CheckRefundCodeEnum code) {
    return StudioMerchantCheckRefundVO.builder().code(code.getCode()).failMessage(code.getMessage()).build();
  }

  public static RefundEvent convertRefundEvent(StudioCashierRecord refund, StudioMerchantCheckRefundVO vo) {
    return RefundEvent.builder()
        .id(refund.getId())
        .transNo(refund.getTransNo())
        .refundState(BrandCashierTransStateEnum.REFUND_FAIL.getCode())
        .failMessage(vo.getFailMessage())
        .updateTime(new Date())
        .build();
  }

  public static RefundEvent convertRefundEvent(StudioCashierRecord refund, RefundVO vo) {
    return RefundEvent.builder()
        .id(refund.getId())
        .transNo(refund.getTransNo())
        .thirdTransNo(vo.getThirdTransNo())
        .refundState(convertTransState(vo.getRefundState()).getCode())
        .failMessage(vo.getFailMessage())
        .successTime(vo.getSuccessTime())
        .updateTime(new Date())
        .build();
  }

}
