package com.jiejing.fitness.finance.service.merchant.impl;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.jiejing.common.exception.BizException;
import com.jiejing.common.model.PageVO;
import com.jiejing.common.utils.collection.CollectionUtil;
import com.jiejing.common.utils.convert.BeanUtil;
import com.jiejing.filecenter.api.resource.vo.ResourceInfoVO;
import com.jiejing.fitness.enums.finance.PartyTypeEnum;
import com.jiejing.fitness.finance.api.merchant.request.model.BrandResourceInfo;
import com.jiejing.fitness.finance.api.merchant.vo.StudioMerchantApplyVO;
import com.jiejing.fitness.finance.api.merchant.vo.StudioMerchantAuthSubChannelVO;
import com.jiejing.fitness.finance.api.merchant.vo.StudioMerchantBindXcxAppIdVO;
import com.jiejing.fitness.finance.api.merchant.vo.StudioMerchantVO;
import com.jiejing.fitness.finance.repository.entity.PartyToMerchant;
import com.jiejing.fitness.finance.repository.entity.StudioMerchantApply;
import com.jiejing.fitness.finance.repository.query.PageStudioMerchantApplyQuery;
import com.jiejing.fitness.finance.repository.service.PartyToMerchantRpService;
import com.jiejing.fitness.finance.repository.service.StudioMerchantApplyRpService;
import com.jiejing.fitness.finance.service.enums.FinanceErrorEnums;
import com.jiejing.fitness.finance.service.global.ConfigService;
import com.jiejing.fitness.finance.service.global.dto.SubChannelInfoDTO;
import com.jiejing.fitness.finance.service.merchant.BrandMerchantService;
import com.jiejing.fitness.finance.service.merchant.StudioMerchantService;
import com.jiejing.fitness.finance.service.merchant.convert.MerchantConvert;
import com.jiejing.fitness.finance.service.merchant.params.ApplyStudioMerchantParams;
import com.jiejing.fitness.finance.service.merchant.params.PageStudioMerchantApplyParams;
import com.jiejing.fitness.finance.service.rpc.MerchantRpcService;
import com.jiejing.fitness.finance.service.rpc.MessageRpcService;
import com.jiejing.fitness.finance.service.rpc.ResourceRpcService;
import com.jiejing.fitness.finance.service.rpc.StudioRpcService;
import com.jiejing.paycenter.api.merchant.request.ApplyMerchantRequest;
import com.jiejing.paycenter.common.model.vo.AuthSubChannelVO;
import com.jiejing.paycenter.common.model.vo.ConfigSubChannelVO;
import com.jiejing.paycenter.common.model.vo.MerchantVO;
import com.jiejing.paycenter.common.model.vo.SubChannelAuthVO;
import com.jiejing.paycenter.common.enums.common.OpenStateEnums;
import com.jiejing.paycenter.common.enums.merchant.ResourceTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelAuthTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelConfigTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelEnums;
import com.jiejing.paycenter.common.event.MerchantEvent;
import com.jiejing.paycenter.common.model.request.ResourceInfo;
import com.jiejing.studio.api.studio.vo.StudioVO;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

/**
 * @author chengyubing
 * @since 2024/3/5 09:58
 */
@Slf4j
@Service
public class StudioMerchantServiceImpl implements StudioMerchantService {

  @Value("${finance.brand.merchant.channel}")
  private String channel;

  @Resource
  private StudioRpcService studioRpcService;

  @Resource
  private MerchantRpcService merchantRpcService;

  @Resource
  private ResourceRpcService resourceRpcService;

  @Resource
  private StudioMerchantApplyRpService studioMerchantApplyRpService;

  @Resource
  private PartyToMerchantRpService partyToMerchantRpService;

  @Resource
  private BrandMerchantService brandMerchantService;

  @Resource
  private ConfigService configService;

  @Resource
  private MessageRpcService messageRpcService;

  @Resource(name = "financeThreadPool")
  private Executor executor;

  @Override
  public Long apply(ApplyStudioMerchantParams params) {

    this.checkBeforeApply(params);

    StudioMerchantApply apply = this.saveApply(params);

    executor.execute(() -> {
      try {

        Map<ResourceTypeEnums, ResourceInfo> resourceMap = this.upload(params);

        SubChannelInfoDTO subChannel = configService.getDefaultBrandSubChannelInfo();

        ApplyMerchantRequest request = MerchantConvert.convertRequest(params, apply, resourceMap, subChannel);
        merchantRpcService.apply(request);

      } catch (Exception e) {
        log.error("apply brand merchant fail {}, studio id = {}", apply.getId(), params.getStudioId(), e);
        this.doOpenMerchantFail(apply.getId(), e.getMessage());
        return;
      }

      this.doOpenMerchantProcess(apply.getId());
    });

    return apply.getId();
  }

  @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
  @Override
  public void bind(Long studioId, Long merchantId, Boolean flag) {
    PartyToMerchant relation = this.getRelation(studioId);
    StudioMerchantApply apply = studioMerchantApplyRpService.getLatestOneSuccessByMerchantId(merchantId);
    if (null == relation) {
      partyToMerchantRpService.insert(
          MerchantConvert.convertPartyToMerchant(apply.getChannelNo(), studioId, PartyTypeEnum.STUDIO,
              apply.getMerchantId(), apply.getMerchantNo()));
    } else {
      partyToMerchantRpService.updateById(
          MerchantConvert.convertPartyToMerchant(relation.getId(), merchantId, apply.getMerchantNo()));
    }
    // 新增绑定记录
    if (flag) {
      StudioMerchantApply bindApply = MerchantConvert.convertBindApply(studioId, apply);
      studioMerchantApplyRpService.insert(bindApply);
    }
  }

  @Override
  public void unbind(Long studioId, Long merchantId) {
    partyToMerchantRpService.deleteByPartyAndMerchantId(studioId, PartyTypeEnum.STUDIO, merchantId);
  }

  @Override
  public void unbindAll(Long studioId) {
    partyToMerchantRpService.deleteByParty(studioId, PartyTypeEnum.STUDIO);
  }

  @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
  @Override
  public void callback(MerchantEvent event) {
    switch (OpenStateEnums.getByCode(event.getState())) {
      case PROCESS:
        this.doOpenMerchantProcess(Long.parseLong(event.getApplyNo()));
        break;
      case FAIL:
        this.doOpenMerchantFail(Long.parseLong(event.getApplyNo()), event.getFailMessage());
        break;
      case SUCCESS:
        this.doOpenMerchantSuccess(event);
        break;
      default:
        break;
    }
  }

  @Override
  public StudioMerchantVO getMerchant(Long studioId) {
    PartyToMerchant relation = getRelation(studioId);
    if (null == relation) {
      return null;
    }
    MerchantVO merchant = merchantRpcService.getByMerchantId(relation.getMerchantId());
    List<Long> resourceIds = this.getResourceIds(merchant);
    StudioMerchantApply apply = studioMerchantApplyRpService.getLatestOneSuccessByMerchantId(
        relation.getMerchantId());
    Map<Long, String> urlMap = resourceRpcService.getResourceUrlMap(apply.getBrandId(), resourceIds);
    return MerchantConvert.convertStudioMerchant(relation, merchant,
        configService.getDefaultBrandSubChannelInfo(), urlMap);
  }

  @Override
  public StudioMerchantApplyVO getApply(Long id) {
    StudioMerchantApply apply = studioMerchantApplyRpService.getById(id)
        .orElseThrow(() -> new BizException(FinanceErrorEnums.NOT_EXIST));
    Map<Long, String> urlMap = this.getResouceseUrlMap(apply);
    return MerchantConvert.convertApply(apply, urlMap);
  }

  private Map<Long, String> getResouceseUrlMap(StudioMerchantApply apply) {
    List<Long> resourceIds = this.getResourceIds(apply);
    return resourceRpcService.getResourceUrlMap(apply.getBrandId(), resourceIds);
  }

  private List<Long> getResourceIds(StudioMerchantApply apply) {
    ApplyStudioMerchantParams info = JSON.parseObject(apply.getMerchantInfo(),
        ApplyStudioMerchantParams.class);
    return info.getResourceMap().values().stream().map(BrandResourceInfo::getResourceId)
        .collect(Collectors.toList());
  }

  @Override
  public PageVO<StudioMerchantApplyVO> pageApply(PageStudioMerchantApplyParams params) {
    Page<StudioMerchantApply> page = studioMerchantApplyRpService.page(BeanUtil.map(params,
        PageStudioMerchantApplyQuery.class));
    return PageVO.convert(page, MerchantConvert.convertApplyList(page.getContent()));
  }

  @Override
  public StudioMerchantApplyVO getLatestApply(Long studioId) {
    StudioMerchantApply apply = studioMerchantApplyRpService.getLatestApply(studioId);
    if (null == apply) {
      return null;
    }
    return MerchantConvert.convertApply(apply);
  }

  @Override
  public StudioMerchantBindXcxAppIdVO bindXcxAppId(Long studioId, String appId) {
    PartyToMerchant relation = getRelation(studioId);
    if (null == relation) {
      throw new BizException(FinanceErrorEnums.MERCHANT_NOT_OPEN);
    }
    ConfigSubChannelVO vo = merchantRpcService.bindAppIdWxXcxOffline(channel, relation.getMerchantId(),
        appId);

    return StudioMerchantBindXcxAppIdVO.builder().state(vo.getState()).failMessage(vo.getFailMessage())
        .build();
  }

  private PartyToMerchant getRelation(Long studioId) {
    List<PartyToMerchant> relations = partyToMerchantRpService.listByParty(studioId, PartyTypeEnum.STUDIO,
        channel);
    if (CollectionUtil.isEmpty(relations)) {
      return null;
    }
    return relations.get(0);
  }

  @Override
  public StudioMerchantAuthSubChannelVO authSubChannel(Long studioId, SubChannelEnums subChannel,
      SubChannelAuthTypeEnums authType) {
    PartyToMerchant relation = getRelation(studioId);
    if (null == relation) {
      throw new BizException(FinanceErrorEnums.MERCHANT_NOT_OPEN);
    }

    AuthSubChannelVO vo = merchantRpcService.authSubChannel(channel, relation.getMerchantId(), subChannel,
        authType);
    return BeanUtil.map(vo, StudioMerchantAuthSubChannelVO.class);
  }

  @Override
  public List<StudioMerchantAuthSubChannelVO> listAuthSubChannel(Long studioId) {
    PartyToMerchant relation = getRelation(studioId);
    if (null == relation) {
      throw new BizException(FinanceErrorEnums.MERCHANT_NOT_OPEN);
    }

    MerchantVO merchant = merchantRpcService.getByMerchantId(relation.getMerchantId());
    List<SubChannelAuthVO> auths = merchant.getSubChannelAuths();
    return Optional.ofNullable(auths).orElse(new ArrayList<>(1)).stream()
        .map(e -> BeanUtil.map(e, StudioMerchantAuthSubChannelVO.class)).collect(Collectors.toList());
  }


  private void doOpenMerchantSuccess(MerchantEvent event) {
    StudioMerchantApply apply = studioMerchantApplyRpService.getByIdForUpdate(
        Long.parseLong(event.getApplyNo())).orElseThrow(() -> new BizException(FinanceErrorEnums.NOT_EXIST));
    StudioMerchantApply toModify = MerchantConvert.convertApply(apply, event,
        configService.getDefaultBrandSubChannelInfo().getSubChannels());
    studioMerchantApplyRpService.updateById(toModify);
    if (!OpenStateEnums.SUCCESS.getCode().equals(toModify.getOpenState())) {
      return;
    }
    // 所有子通道全部开通成功
    PartyToMerchant old = getRelation(apply.getStudioId());

    // 1. 绑定场馆
    this.bind(apply.getStudioId(), event.getMerchantId(), false);

    // 2. 绑定品牌
    brandMerchantService.bind(apply.getBrandId(), event.getMerchantId());

    // 3. 其他操作
    executor.execute(() -> this.doAfterMerchantSuccess(event,
        Optional.ofNullable(old).map(PartyToMerchant::getMerchantId).orElse(null)));

  }

  private void doAfterMerchantSuccess(MerchantEvent event, Long oldMerchantId) {
    // 1. 配置场馆小程序appId
    this.rebindMiniAppId(oldMerchantId, event.getMerchantId());
  }

  private void rebindMiniAppId(Long oldMerchantId, Long newMerchantId) {
    if (null == oldMerchantId) {
      return;
    }
    MerchantVO merchant = merchantRpcService.getByMerchantId(oldMerchantId);
    merchant.getSubChannelConfigs().stream()
        .filter(e -> SubChannelConfigTypeEnums.WX_XCX_OFFLINE.name().equals(e.getConfigType()))
        .forEach(
            config -> merchantRpcService.bindAppIdWxXcxOffline(merchant.getChannelNo(), newMerchantId,
                config.getAppId()));
  }

  private Map<ResourceTypeEnums, ResourceInfo> upload(ApplyStudioMerchantParams params) {

    Map<ResourceTypeEnums, ResourceInfo> resourceMap = MerchantConvert.convertResourceMap(
        params.getResourceMap());

    Map<Long, ResourceInfoVO> resourceIdUrlMap = resourceRpcService.getResourceMap(params.getStudioId(),
        getResourceIds(resourceMap));

    resourceMap.keySet().forEach(type -> {
      ResourceInfo info = resourceMap.get(type);
      ResourceInfoVO vo = resourceIdUrlMap.get(info.getResourceId());
      String thirdId = merchantRpcService.upload(MerchantConvert.convertUploadRequest(channel, type, vo));
      info.setThirdId(thirdId);
    });
    return resourceMap;
  }

  private List<Long> getResourceIds(Map<ResourceTypeEnums, ResourceInfo> resourceMap) {
    return resourceMap.values().stream()
        .map(ResourceInfo::getResourceId)
        .filter(Objects::nonNull)
        .distinct()
        .collect(Collectors.toList());
  }

  private List<Long> getResourceIds(MerchantVO merchant) {
    return merchant.getResourceMap().values().stream()
        .map(com.jiejing.paycenter.common.model.vo.ResourceInfoVO::getResourceId)
        .filter(Objects::nonNull)
        .distinct()
        .collect(Collectors.toList());
  }

  private void doOpenMerchantFail(Long id, String failMessage) {
    studioMerchantApplyRpService.updateById(
        StudioMerchantApply.builder().id(id).openState(OpenStateEnums.FAIL.getCode())
            .openFailMessage(failMessage).build());
  }

  private void doOpenMerchantProcess(Long id) {
    studioMerchantApplyRpService.updateById(
        StudioMerchantApply.builder().id(id).openState(OpenStateEnums.PROCESS.getCode()).build());
  }

  private void validSmsCode(String phone, String smsCode) {
    messageRpcService.validSmsCode(phone, smsCode, "FINANCE_MERCHANT");
  }

  private StudioMerchantApply saveApply(ApplyStudioMerchantParams params) {
    Long id = IdWorker.getId();
    StudioVO studio = studioRpcService.getStudio(params.getStudioId());
    StudioMerchantApply exist = studioMerchantApplyRpService.getLatestOneSuccessByStudioId(studio.getId());
    StudioMerchantApply apply = MerchantConvert.convertApply(id, params, studio, exist, channel);

    studioMerchantApplyRpService.insert(apply);
    return apply;
  }

  private void checkBeforeApply(ApplyStudioMerchantParams params) {
    // 校验验证码
    this.validSmsCode(params.getBankCard().getPhone(), params.getSmsCode());
    // 重复提交
    Integer count = studioMerchantApplyRpService.countProcessByStudioId(params.getStudioId());
    if (count.compareTo(0) > 0) {
      throw new BizException(FinanceErrorEnums.EXIST_PROCESS_APPLY);
    }
  }

}
