package com.jiejing.fitness.finance.service.merchant.convert;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.jiejing.common.utils.collection.CollectionUtil;
import com.jiejing.common.utils.convert.BeanUtil;
import com.jiejing.common.utils.crypt.AesUtil;
import com.jiejing.filecenter.api.resource.vo.ResourceInfoVO;
import com.jiejing.fitness.enums.finance.ApplyTypeEnum;
import com.jiejing.fitness.enums.finance.PartyTypeEnum;
import com.jiejing.fitness.finance.api.merchant.request.model.AddressInfo;
import com.jiejing.fitness.finance.api.merchant.request.model.BrandResourceInfo;
import com.jiejing.fitness.finance.api.merchant.vo.BrandBankCardVO;
import com.jiejing.fitness.finance.api.merchant.vo.BrandLegalVO;
import com.jiejing.fitness.finance.api.merchant.vo.BrandLicenseVO;
import com.jiejing.fitness.finance.api.merchant.vo.BrandMerchantVO;
import com.jiejing.fitness.finance.api.merchant.vo.StudioMerchantApplyVO;
import com.jiejing.fitness.finance.api.merchant.vo.StudioMerchantVO;
import com.jiejing.fitness.finance.repository.entity.PartyToMerchant;
import com.jiejing.fitness.finance.repository.entity.StudioMerchantApply;
import com.jiejing.fitness.finance.service.global.dto.SubChannelInfoDTO;
import com.jiejing.fitness.finance.service.merchant.params.ApplyStudioMerchantParams;
import com.jiejing.fitness.finance.service.utils.JsonUtil;
import com.jiejing.paycenter.api.merchant.request.ApplyMerchantRequest;
import com.jiejing.paycenter.api.merchant.request.UploadRequest;
import com.jiejing.paycenter.common.enums.merchant.AuthPhaseEnums;
import com.jiejing.paycenter.common.enums.merchant.CompanyTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelEnums;
import com.jiejing.paycenter.common.model.vo.ContactVO;
import com.jiejing.paycenter.common.model.vo.LegalVO;
import com.jiejing.paycenter.common.model.vo.MerchantVO;
import com.jiejing.paycenter.common.enums.common.OpenStateEnums;
import com.jiejing.paycenter.common.enums.merchant.BusinessTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.LicenseTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.MccEnums;
import com.jiejing.paycenter.common.enums.merchant.ResourceTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SceneTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelAuthTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelOpenTypeEnums;
import com.jiejing.paycenter.common.event.MerchantEvent;
import com.jiejing.paycenter.common.model.request.Address;
import com.jiejing.paycenter.common.model.request.BankCard;
import com.jiejing.paycenter.common.model.request.BusinessInfo;
import com.jiejing.paycenter.common.model.request.Contact;
import com.jiejing.paycenter.common.model.request.Legal;
import com.jiejing.paycenter.common.model.request.License;
import com.jiejing.paycenter.common.model.request.ResourceInfo;
import com.jiejing.paycenter.common.model.request.SubChannelAuthInfo;
import com.jiejing.paycenter.common.model.request.SubChannelInfo;
import com.jiejing.paycenter.common.model.vo.SubChannelAuthVO;
import com.jiejing.paycenter.common.model.vo.SubChannelVO;
import com.jiejing.studio.api.studio.vo.StudioVO;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @author chengyubing
 * @since 2024/2/20 16:46
 */
public class MerchantConvert {

  public static ApplyMerchantRequest convertRequest(ApplyStudioMerchantParams params,
      StudioMerchantApply apply, Map<ResourceTypeEnums, ResourceInfo> resourceMap,
      SubChannelInfoDTO subChannelInfo) {
    return ApplyMerchantRequest.builder()
        .bizCode(PartyTypeEnum.STUDIO.getCode())
        .applyNo(apply.getApplyNo())
        .channelNo(apply.getChannelNo())
        .companyType(CompanyTypeEnums.getByCode(apply.getCompanyType()))
        .merchantName(apply.getMerchantName())
        .shortName(apply.getShortName())
        .legal(convertLegal(params))
        .bankCard(convertBankCard(params))
        .license(convertLicense(params))
        .business(convertBusiness(params))
        .contact(convertContact(params))
        .resourceMap(resourceMap)
        .subChannels(subChannelInfo.getSubChannels())
        .subChannelConfigs(subChannelInfo.getSubChannelConfigs())
        .build();
  }


  private static Contact convertContact(ApplyStudioMerchantParams params) {
    return Contact.builder()
        .contactName(params.getLegal().getLegalName())
        .contactPhone(params.getLegal().getLegalPhone())
        .contactEmail(params.getLegal().getLegalEmail())
        .servicePhone(params.getLegal().getLegalPhone())
        .build();
  }

  private static BusinessInfo convertBusiness(ApplyStudioMerchantParams params) {
    AddressInfo address = params.getLicense().getLicenseAddress();
    return BusinessInfo.builder()
        .mcc(MccEnums.MCC_7941)
        .businessType(BusinessTypeEnums.REALITY)
        .sceneType(SceneTypeEnums.OFFLINE)
        .businessAddress(Optional.ofNullable(address).map(e -> BeanUtil.map(e, Address.class)).orElse(null))
        .build();
  }

  private static License convertLicense(ApplyStudioMerchantParams params) {
    AddressInfo address = params.getLicense().getLicenseAddress();
    return License.builder()
        .licenseType(LicenseTypeEnums.NATIONAL_LEGAL_MERGE)
        .registeredCapital("10000")
        .licenseCode(params.getLicense().getLicenseCode())
        .licenseExpireType(params.getLicense().getLicenseExpireType())
        .licenseStartDate(params.getLicense().getLicenseStartDate())
        .licenseEndDate(params.getLicense().getLicenseEndDate())
        .businessScope(params.getLicense().getBusinessScope())
        .foundDate(params.getLicense().getFoundDate())
        .licenseAddress(Optional.ofNullable(address).map(e -> BeanUtil.map(e, Address.class)).orElse(null))
        .build();
  }

  private static BankCard convertBankCard(ApplyStudioMerchantParams params) {
    AddressInfo address = params.getBankCard().getCardAddress();
    return BankCard.builder()
        .cardType(params.getBankCard().getCardType())
        .cardNo(params.getBankCard().getCardNo())
        .cardName(params.getBankCard().getCardName())
        .phone(params.getBankCard().getPhone())
        .certNo(params.getBankCard().getCertNo())
        .certType(params.getBankCard().getCertType())
        .baseDepositNo(params.getBankCard().getBaseDepositNo())
        .certExpireType(params.getBankCard().getCertExpireType())
        .certStartDate(params.getBankCard().getCertStartDate())
        .certEndDate(params.getBankCard().getCertEndDate())
        .bankCode(params.getBankCard().getBankCode())
        .bankName(params.getBankCard().getBankName())
        .branchName(params.getBankCard().getBranchName())
        .branchCode(params.getBankCard().getBranchCode())
        .cardAddress(Optional.ofNullable(address).map(e -> BeanUtil.map(e, Address.class)).orElse(null))
        .build();
  }

  private static Legal convertLegal(ApplyStudioMerchantParams params) {
    AddressInfo address = params.getLegal().getLegalAddress();
    return Legal.builder()
        .legalName(params.getLegal().getLegalName())
        .legalPhone(params.getLegal().getLegalPhone())
        .legalCertNo(params.getLegal().getLegalCertNo())
        .legalCertType(params.getLegal().getLegalCertType())
        .legalCertExpireType(params.getLegal().getLegalCertExpireType())
        .legalCertStartDate(params.getLegal().getLegalCertStartDate())
        .legalCertEndDate(params.getLegal().getLegalCertEndDate())
        .legalAddress(Optional.ofNullable(address).map(e -> BeanUtil.map(e, Address.class)).orElse(null))
        .build();
  }

  public static StudioMerchantApply convertApply(Long id, ApplyStudioMerchantParams params, StudioVO studio,
      StudioMerchantApply exist, String channel) {
    String salt = AesUtil.getSalt(8);
    ApplyStudioMerchantParams encrypt = params.encrypt(salt);
    return StudioMerchantApply.builder()
        .id(id)
        .applyNo(Long.toString(id))
        .brandId(studio.getBrandId())
        .studioId(encrypt.getStudioId())
        .applyType(exist == null ? ApplyTypeEnum.OPEN.getCode() : ApplyTypeEnum.RE_OPEN.getCode())
        .channelNo(channel)
        .merchantName(encrypt.getMerchantName())
        .shortName(studio.getName())
        .companyType(encrypt.getCompanyType().getCode())
        .openState(OpenStateEnums.INIT.getCode())
        .merchantInfo(convertMerchantInfo(encrypt))
        .salt(salt)
        .createTime(new Date())
        .updateTime(new Date())
        .build();
  }

  private static String convertMerchantInfo(ApplyStudioMerchantParams encrypt) {
    return JSON.toJSONString(ApplyStudioMerchantParams.builder()
        .legal(encrypt.getLegal())
        .license(encrypt.getLicense())
        .bankCard(encrypt.getBankCard())
        .resourceMap(encrypt.getResourceMap())
        .build());
  }

  public static UploadRequest convertUploadRequest(String channel, ResourceTypeEnums type,
      ResourceInfoVO vo) {
    return UploadRequest.builder()
        .url(vo.getUrl())
        .fileSize(vo.getFileSize())
        .suffix(vo.getSuffix())
        .channelNo(channel)
        .resourceType(type)
        .build();
  }

  private static List<SubChannelInfo> convertSubChannelInfos(List<SubChannelVO> list) {
    return list.stream()
        .map(e -> SubChannelInfo.builder()
            .subChannel(SubChannelEnums.getByCode(e.getSubChannel()))
            .openType(SubChannelOpenTypeEnums.getByCode(e.getOpenType()))
            .feeRate(e.getFeeRate())
            .state(OpenStateEnums.getByCode(e.getState()))
            .merchantNos(e.getMerchantNos())
            .failMessage(e.getFailMessage())
            .openTime(e.getOpenTime())
            .build()).collect(Collectors.toList());
  }

  public static StudioMerchantApply convertApply(StudioMerchantApply apply, MerchantEvent event,
      List<SubChannelInfo> defaultSubChannels) {
    // 品牌商户规则：微信线下公众号、微信线下小程序、支付宝线下通道全部开通成功才算成功
    List<SubChannelInfo> subChannels = convertSubChannelInfos(event.getSubChannels());
    List<SubChannelAuthInfo> auths = convertSubChannelAuths(event.getSubChannelAuths());
    Map<SubChannelOpenTypeEnums, SubChannelInfo> channelMap = convertSubChannelMap(subChannels,
        defaultSubChannels);
    OpenStateEnums state = convertOpenState(channelMap, defaultSubChannels);
    String openFailMessage = OpenStateEnums.FAIL == state ? convertFailMessage(subChannels) : null;
    SubChannelInfo ali = channelMap.get(SubChannelOpenTypeEnums.ALI_OFFLINE);
    SubChannelInfo wxGzhOffline = channelMap.get(SubChannelOpenTypeEnums.WX_GZH_OFFLINE);
    OpenStateEnums wxOfflineState = convertWxOfflineState(channelMap);
    String wxOfflineFailMessage = convertWxOfflineFailMessage(channelMap);
    return StudioMerchantApply.builder()
        .id(apply.getId())
        .brandId(apply.getBrandId())
        .studioId(apply.getStudioId())
        .applyNo(apply.getApplyNo())
        .applyType(apply.getApplyType())
        .channelNo(apply.getChannelNo())
        .merchantId(event.getMerchantId())
        .merchantNo(event.getMerchantNo())
        .merchantName(event.getMerchantName())
        .shortName(event.getShortName())
        .companyType(event.getCompanyType())
        .openState(state.getCode())
        .openFailMessage(openFailMessage)
        .aliOpenState(ali.getState().getCode())
        .aliAuthState(convertAliAuthState(auths))
        .aliMerchantNo(
            Optional.ofNullable(ali.getMerchantNos()).orElse(new ArrayList<>(1)).stream().findFirst()
                .orElse(null))
        .aliOpenFailMessage(ali.getFailMessage())
        .wxOfflineOpenState(wxOfflineState.getCode())
        .wxOfflineMerchantNo(
            Optional.ofNullable(wxGzhOffline.getMerchantNos()).orElse(new ArrayList<>(1)).stream().findFirst()
                .orElse(null))
        .wxOfflineAuthState(convertWxOfflineAuthState(auths))
        .wxOfflineOpenFailMessage(wxOfflineFailMessage)
        .finishTime(OpenStateEnums.SUCCESS == state ? new Date() : null)
        .updateTime(new Date())
        .build();
  }

  private static List<SubChannelAuthInfo> convertSubChannelAuths(List<SubChannelAuthVO> auths) {
    return auths.stream().map(e -> {
      return SubChannelAuthInfo.builder()
          .subChannel(SubChannelEnums.getByCode(e.getSubChannel()))
          .authType(SubChannelAuthTypeEnums.getByCode(e.getAuthType()))
          .state(OpenStateEnums.getByCode(e.getState()))
          .phase(AuthPhaseEnums.getByCode(e.getPhase()))
          .qrCode(e.getQrCode())
          .authTime(e.getAuthTime())
          .failMessage(e.getFailMessage())
          .build();
    }).collect(Collectors.toList());
  }

  private static BrandLegalVO convertLegal(LegalVO legal, ContactVO contact) {
    BrandLegalVO legalInfo = JsonUtil.convertObject(legal, BrandLegalVO.class);
    legalInfo.setLegalEmail(contact.getContactEmail());
    return legalInfo;
  }

  private static Map<String, BrandResourceInfo> convertResourceMap(
      Map<String, com.jiejing.paycenter.common.model.vo.ResourceInfoVO> resourceMap,
      Map<Long, String> urlMap) {
    Map<String, BrandResourceInfo> map = new HashMap<>(resourceMap.size());
    resourceMap.keySet().forEach(key -> {
      Long resourceId = resourceMap.get(key).getResourceId();
      String url = urlMap.get(resourceId);
      map.put(key, BrandResourceInfo.builder().resourceId(resourceId).url(url).build());
    });
    return map;
  }

  private static String convertWxOfflineAuthState(List<SubChannelAuthInfo> auths) {
    return auths.stream()
        .filter(e -> e.getAuthType() == SubChannelAuthTypeEnums.WX_OFFLINE)
        .findFirst()
        .map(e -> e.getState().getCode())
        .orElse(null);
  }

  private static String convertAliAuthState(List<SubChannelAuthInfo> auths) {
    return auths.stream()
        .filter(e -> e.getAuthType() == SubChannelAuthTypeEnums.ALI_OFFLINE)
        .findFirst()
        .map(e -> e.getState().getCode())
        .orElse(null);
  }

  private static String convertWxOfflineFailMessage(Map<SubChannelOpenTypeEnums, SubChannelInfo> channelMap) {
    SubChannelInfo wxGzhOffline = channelMap.get(SubChannelOpenTypeEnums.WX_GZH_OFFLINE);
    SubChannelInfo wxXcxOffline = channelMap.get(SubChannelOpenTypeEnums.WX_XCX_OFFLINE);
    if (OpenStateEnums.FAIL == wxGzhOffline.getState()) {
      return wxGzhOffline.getFailMessage();
    }
    if (OpenStateEnums.FAIL == wxXcxOffline.getState()) {
      return wxXcxOffline.getFailMessage();
    }
    return null;
  }

  private static OpenStateEnums convertWxOfflineState(
      Map<SubChannelOpenTypeEnums, SubChannelInfo> channelMap) {
    SubChannelInfo wxGzhOffline = channelMap.get(SubChannelOpenTypeEnums.WX_GZH_OFFLINE);
    SubChannelInfo wxXcxOffline = channelMap.get(SubChannelOpenTypeEnums.WX_XCX_OFFLINE);
    if (OpenStateEnums.SUCCESS == wxGzhOffline.getState()
        && OpenStateEnums.SUCCESS == wxXcxOffline.getState()) {
      return OpenStateEnums.SUCCESS;
    }
    if (OpenStateEnums.FAIL == wxGzhOffline.getState()) {
      return OpenStateEnums.FAIL;
    }
    if (OpenStateEnums.FAIL == wxXcxOffline.getState()) {
      return OpenStateEnums.FAIL;
    }
    return OpenStateEnums.PROCESS;
  }

  private static String convertFailMessage(List<SubChannelInfo> subChannels) {
    return subChannels.stream()
        .filter(e -> OpenStateEnums.FAIL == e.getState())
        .map(SubChannelInfo::getFailMessage)
        .findFirst()
        .orElse(null);
  }

  private static OpenStateEnums convertOpenState(Map<SubChannelOpenTypeEnums, SubChannelInfo> channelMap,
      List<SubChannelInfo> defaultSubChannels) {

    List<SubChannelOpenTypeEnums> sceneList = getDefaultScene(defaultSubChannels);

    OpenStateEnums state = OpenStateEnums.SUCCESS;
    for (SubChannelOpenTypeEnums scene : sceneList) {
      SubChannelInfo subChannel = channelMap.get(scene);
      if (OpenStateEnums.PROCESS == subChannel.getState()) {
        state = OpenStateEnums.PROCESS;
      }
      if (OpenStateEnums.FAIL == subChannel.getState()) {
        return OpenStateEnums.FAIL;
      }
    }
    return state;
  }

  private static Map<SubChannelOpenTypeEnums, SubChannelInfo> convertSubChannelMap(
      List<SubChannelInfo> subChannels, List<SubChannelInfo> defaultSubChannels) {
    Map<SubChannelOpenTypeEnums, SubChannelInfo> map = subChannels.stream()
        .collect(Collectors.toMap(SubChannelInfo::getOpenType, e -> e));
    for (SubChannelInfo defaultSubChannel : defaultSubChannels) {
      if (null == map.get(defaultSubChannel.getOpenType())) {
        map.put(defaultSubChannel.getOpenType(),
            SubChannelInfo.builder().openType(defaultSubChannel.getOpenType()).state(OpenStateEnums.PROCESS)
                .build());
      }
    }
    return map;
  }

  public static List<SubChannelOpenTypeEnums> getDefaultScene(List<SubChannelInfo> defaultSubChannels) {
    return defaultSubChannels.stream().map(SubChannelInfo::getOpenType)
        .collect(Collectors.toList());
  }

  private static ResourceInfo convertResourceInfo(ResourceTypeEnums type, BrandResourceInfo info) {
    return ResourceInfo.builder()
        .type(type)
        .resourceId(info.getResourceId())
        .build();
  }

  public static Map<ResourceTypeEnums, ResourceInfo> convertResourceMap(
      Map<String, BrandResourceInfo> resourceMap) {
    Map<ResourceTypeEnums, ResourceInfo> map = new HashMap<>(1);
    resourceMap.keySet().stream()
        .filter(key -> null != ResourceTypeEnums.getByName(key))
        .forEach(key -> {
          ResourceTypeEnums type = ResourceTypeEnums.getByName(key);
          map.put(type, convertResourceInfo(type, resourceMap.get(key)));
        });
    return map;
  }

  public static StudioMerchantVO convertStudioMerchant(PartyToMerchant relation, MerchantVO merchant,
      SubChannelInfoDTO dto, Map<Long, String> urlMap, StudioMerchantApply apply) {
    Map<SubChannelOpenTypeEnums, SubChannelInfo> channelMap = convertSubChannelMap(
        JsonUtil.convertList(merchant.getSubChannels(), SubChannelInfo.class), dto.getSubChannels());
    SubChannelInfo ali = channelMap.get(SubChannelOpenTypeEnums.ALI_OFFLINE);
    SubChannelInfo wxGzhOffline = channelMap.get(SubChannelOpenTypeEnums.WX_GZH_OFFLINE);
    OpenStateEnums wxOfflineState = convertWxOfflineState(channelMap);
    String wxOfflineFailMessage = convertWxOfflineFailMessage(channelMap);
    String aliAuthState = convertAliAuthState(
        JsonUtil.convertList(merchant.getSubChannelAuths(), SubChannelAuthInfo.class));
    String wxOfflineAuthState = convertWxOfflineAuthState(
        JsonUtil.convertList(merchant.getSubChannelAuths(), SubChannelAuthInfo.class));
    return StudioMerchantVO.builder()
        .studioId(relation.getPartyId())
        .channelNo(merchant.getChannelNo())
        .merchantId(merchant.getId())
        .merchantNo(merchant.getMerchantNo())
        .merchantName(merchant.getMerchantName())
        .shortName(merchant.getShortName())
        .openState(apply.getOpenState())
        .companyType(merchant.getCompanyType())
        .aliOpenState(Optional.ofNullable(ali.getState()).map(Enum::name).orElse(null))
        .aliAuthState(aliAuthState)
        .aliMerchantNo(
            Optional.ofNullable(ali.getMerchantNos()).orElse(new ArrayList<>()).stream().findFirst()
                .orElse(null))
        .aliOpenFailMessage(ali.getFailMessage())
        .wxOfflineOpenState(Optional.ofNullable(wxOfflineState).map(Enum::name).orElse(null))
        .wxOfflineMerchantNo(
            Optional.ofNullable(wxGzhOffline.getMerchantNos()).orElse(new ArrayList<>()).stream().findFirst()
                .orElse(null))
        .wxOfflineAuthState(wxOfflineAuthState)
        .wxOfflineOpenFailMessage(wxOfflineFailMessage)
        .license(JsonUtil.convertObject(merchant.getLicense(), BrandLicenseVO.class))
        .legal(convertLegal(merchant.getLegal(), merchant.getContact()))
        .bankCard(JsonUtil.convertObject(merchant.getBankCard(), BrandBankCardVO.class))
        .resourceMap(convertResourceMap(merchant.getResourceMap(), urlMap))
        .subChannelConfigs(merchant.getSubChannelConfigs())
        .build();
  }

  public static StudioMerchantApplyVO convertApply(StudioMerchantApply apply, Map<Long, String> urlMap) {
    ApplyStudioMerchantParams info = JSON.parseObject(apply.getMerchantInfo(),
        ApplyStudioMerchantParams.class).decrypt(apply.getSalt());
    if (CollectionUtil.isNotEmpty(info.getResourceMap()) && CollectionUtil.isNotEmpty(urlMap)) {
      info.getResourceMap().values().forEach(e -> e.setUrl(urlMap.get(e.getResourceId())));
    }
    return StudioMerchantApplyVO.builder()
        .id(apply.getId())
        .brandId(apply.getBrandId())
        .studioId(apply.getStudioId())
        .applyNo(apply.getApplyNo())
        .applyType(apply.getApplyType())
        .channelNo(apply.getChannelNo())
        .merchantId(apply.getMerchantId())
        .merchantNo(apply.getMerchantNo())
        .merchantName(apply.getMerchantName())
        .shortName(apply.getShortName())
        .companyType(apply.getCompanyType())
        .openState(apply.getOpenState())
        .openFailMessage(apply.getOpenFailMessage())
        .aliOpenState(apply.getAliOpenState())
        .aliAuthState(apply.getAliAuthState())
        .aliMerchantNo(apply.getAliMerchantNo())
        .aliOpenFailMessage(apply.getAliOpenFailMessage())
        .wxOfflineOpenState(apply.getWxOfflineOpenState())
        .wxOfflineAuthState(apply.getWxOfflineAuthState())
        .wxOfflineMerchantNo(apply.getWxOfflineMerchantNo())
        .wxOfflineOpenFailMessage(apply.getWxOfflineOpenFailMessage())
        .wxOnlineOpenState(apply.getWxOnlineOpenState())
        .wxOnlineAuthState(apply.getWxOnlineAuthState())
        .wxOnlineMerchantNo(apply.getWxOnlineMerchantNo())
        .wxOnlineOpenFailMessage(apply.getWxOnlineOpenFailMessage())
        .license(info.getLicense())
        .legal(info.getLegal())
        .bankCard(info.getBankCard())
        .resourceMap(info.getResourceMap())
        .finishTime(apply.getFinishTime())
        .createTime(apply.getCreateTime())
        .updateTime(apply.getUpdateTime())
        .build();
  }

  public static List<StudioMerchantApplyVO> convertApplyList(List<StudioMerchantApply> applies) {
    return Optional.ofNullable(applies)
        .orElse(new ArrayList<>())
        .stream()
        .map(e -> convertApply(e, null))
        .collect(Collectors.toList());
  }

  public static PartyToMerchant convertPartyToMerchant(String channelNo, Long partyId, PartyTypeEnum type,
      Long merchantId, String merchantNo) {
    PartyToMerchant merchant = new PartyToMerchant();
    merchant.setId(IdWorker.getId());
    merchant.setChannelNo(channelNo);
    merchant.setMerchantId(merchantId);
    merchant.setMerchantNo(merchantNo);
    merchant.setPartyId(partyId);
    merchant.setPartyType(type.getCode());
    merchant.setCreateTime(new Date());
    merchant.setUpdateTime(new Date());
    return merchant;
  }

  public static PartyToMerchant convertPartyToMerchant(Long id, Long merchantId, String merchantNo) {
    PartyToMerchant merchant = new PartyToMerchant();
    merchant.setId(id);
    merchant.setMerchantId(merchantId);
    merchant.setMerchantNo(merchantNo);
    merchant.setUpdateTime(new Date());
    return merchant;
  }

  private static BrandMerchantVO convertBrandMerchant(Long brandId, List<String> studioNames,
      MerchantVO merchant, SubChannelInfoDTO dto, StudioMerchantApply apply) {
    Map<SubChannelOpenTypeEnums, SubChannelInfo> channelMap = convertSubChannelMap(
        JsonUtil.convertList(merchant.getSubChannels(), SubChannelInfo.class), dto.getSubChannels());
    SubChannelInfo ali = channelMap.get(SubChannelOpenTypeEnums.ALI_OFFLINE);
    SubChannelInfo wxGzhOffline = channelMap.get(SubChannelOpenTypeEnums.WX_GZH_OFFLINE);
    OpenStateEnums wxOfflineState = convertWxOfflineState(channelMap);
    String wxOfflineFailMessage = convertWxOfflineFailMessage(channelMap);
    String aliAuthState = convertAliAuthState(
        JsonUtil.convertList(merchant.getSubChannelAuths(), SubChannelAuthInfo.class));
    String wxOfflineAuthState = convertWxOfflineAuthState(
        JsonUtil.convertList(merchant.getSubChannelAuths(), SubChannelAuthInfo.class));
    return BrandMerchantVO.builder()
        .brandId(brandId)
        .studioNames(studioNames)
        .channelNo(merchant.getChannelNo())
        .merchantId(merchant.getId())
        .merchantNo(merchant.getMerchantNo())
        .merchantName(merchant.getMerchantName())
        .shortName(merchant.getShortName())
        .companyType(merchant.getCompanyType())
        .aliOpenState(Optional.ofNullable(ali.getState()).map(Enum::name).orElse(null))
        .aliAuthState(aliAuthState).aliMerchantNo(
            Optional.ofNullable(ali.getMerchantNos()).orElse(new ArrayList<>()).stream().findFirst()
                .orElse(null))
        .aliOpenFailMessage(ali.getFailMessage())
        .wxOfflineOpenState(Optional.ofNullable(wxOfflineState).map(Enum::name).orElse(null))
        .wxOfflineMerchantNo(
            Optional.ofNullable(wxGzhOffline.getMerchantNos()).orElse(new ArrayList<>()).stream().findFirst()
                .orElse(null))
        .wxOfflineAuthState(wxOfflineAuthState)
        .wxOfflineOpenFailMessage(wxOfflineFailMessage)
        .license(JsonUtil.convertObject(merchant.getLicense(), BrandLicenseVO.class))
        .legal(convertLegal(merchant.getLegal(), merchant.getContact()))
        .bankCard(JsonUtil.convertObject(merchant.getBankCard(), BrandBankCardVO.class))
        .subChannelConfigs(merchant.getSubChannelConfigs())
        .finishTime(apply.getFinishTime())
        .build();
  }

  public static List<BrandMerchantVO> convertBrandMerchantList(Long brandId, List<MerchantVO> merchants,
      SubChannelInfoDTO dto, Map<Long, List<StudioVO>> studioMap, Map<Long, StudioMerchantApply> applyMap) {
    return merchants.stream().map(e -> {
      List<String> studioNames = studioMap.getOrDefault(brandId, new ArrayList<>()).stream()
          .map(StudioVO::getName).collect(Collectors.toList());
      return convertBrandMerchant(brandId, studioNames, e, dto, applyMap.get(e.getId()));
    }).collect(Collectors.toList());
  }

  public static StudioMerchantApply convertBindApply(Long studioId, StudioMerchantApply apply) {
    StudioMerchantApply toSave = BeanUtil.map(apply, StudioMerchantApply.class);
    toSave.setId(IdWorker.getId());
    toSave.setApplyNo("-1");
    toSave.setApplyType(ApplyTypeEnum.BIND.getCode());
    toSave.setStudioId(studioId);
    toSave.setCreateTime(new Date());
    toSave.setUpdateTime(new Date());
    return toSave;
  }

}
