package com.jiejing.fitness.finance.service.merchant.impl;

import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.jiejing.common.exception.BizException;
import com.jiejing.common.model.PageVO;
import com.jiejing.common.utils.collection.CollectionUtil;
import com.jiejing.common.utils.convert.BeanUtil;
import com.jiejing.filecenter.api.resource.vo.ResourceInfoVO;
import com.jiejing.fitness.enums.finance.PartyTypeEnum;
import com.jiejing.fitness.finance.api.merchant.vo.StudioMerchantApplyVO;
import com.jiejing.fitness.finance.api.merchant.vo.StudioMerchantAuthSubChannelVO;
import com.jiejing.fitness.finance.api.merchant.vo.StudioMerchantBindXcxAppIdVO;
import com.jiejing.fitness.finance.api.merchant.vo.StudioMerchantVO;
import com.jiejing.fitness.finance.repository.entity.PartyToMerchant;
import com.jiejing.fitness.finance.repository.entity.StudioMerchantApply;
import com.jiejing.fitness.finance.repository.query.PageStudioMerchantApplyQuery;
import com.jiejing.fitness.finance.repository.service.PartyToMerchantRpService;
import com.jiejing.fitness.finance.repository.service.StudioMerchantApplyRpService;
import com.jiejing.fitness.finance.service.enums.FinanceErrorEnums;
import com.jiejing.fitness.finance.service.global.ConfigService;
import com.jiejing.fitness.finance.service.merchant.BrandMerchantService;
import com.jiejing.fitness.finance.service.merchant.StudioMerchantService;
import com.jiejing.fitness.finance.service.merchant.convert.MerchantConvert;
import com.jiejing.fitness.finance.service.merchant.params.ApplyStudioMerchantParams;
import com.jiejing.fitness.finance.service.merchant.params.PageStudioMerchantApplyParams;
import com.jiejing.fitness.finance.service.rpc.MerchantRpcService;
import com.jiejing.fitness.finance.service.rpc.ResourceRpcService;
import com.jiejing.fitness.finance.service.rpc.StudioRpcService;
import com.jiejing.paycenter.api.merchant.request.ApplyMerchantRequest;
import com.jiejing.paycenter.api.merchant.vo.AuthSubChannelVO;
import com.jiejing.paycenter.api.merchant.vo.ConfigSubChannelVO;
import com.jiejing.paycenter.api.merchant.vo.MerchantVO;
import com.jiejing.paycenter.api.merchant.vo.SubChannelAuthVO;
import com.jiejing.paycenter.common.enums.common.OpenStateEnums;
import com.jiejing.paycenter.common.enums.merchant.ResourceTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelAuthTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelConfigTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelEnums;
import com.jiejing.paycenter.common.event.MerchantEvent;
import com.jiejing.paycenter.common.model.ResourceInfo;
import com.jiejing.studio.api.studio.vo.StudioVO;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

/**
 * @author chengyubing
 * @since 2024/3/5 09:58
 */
@Slf4j
@Service
public class StudioMerchantServiceImpl implements StudioMerchantService {

  @Value("${finance.brand.merchant.channel}")
  private String channel;

  @Resource
  private StudioRpcService studioRpcService;

  @Resource
  private MerchantRpcService merchantRpcService;

  @Resource
  private ResourceRpcService resourceRpcService;

  @Resource
  private StudioMerchantApplyRpService studioMerchantApplyRpService;

  @Resource
  private PartyToMerchantRpService partyToMerchantRpService;

  @Resource
  private BrandMerchantService brandMerchantService;

  @Resource
  private ConfigService configService;

  @Resource(name = "financeThreadPool")
  private Executor executor;

  @Async(value = "financeThreadPool")
  @Override
  public void apply(ApplyStudioMerchantParams params) {

    Long id = IdWorker.getId();
    StudioVO studio = studioRpcService.getStudio(params.getStudioId());
    StudioMerchantApply exist = studioMerchantApplyRpService.getLatestOneSuccessByStudioId(studio.getId());
    StudioMerchantApply apply = MerchantConvert.convertApply(id, params, studio, exist, channel);

    studioMerchantApplyRpService.insert(apply);

    try {

      Map<ResourceTypeEnums, ResourceInfo> resourceMap = this.upload(params);
      ApplyMerchantRequest request = MerchantConvert.convertRequest(channel, apply.getApplyNo(), params,
          studio, resourceMap, configService.getDefaultBrandSubChannelInfo());
      merchantRpcService.apply(request);

    } catch (Exception e) {
      log.error("apply brand merchant fail {}, brand id = {}, studio id = {}", id, studio.getBrandId(),
          params.getStudioId(), e);
      this.doOpenMerchantFail(id, e.getMessage());
      return;
    }

    this.doOpenMerchantProcess(id);

  }

  @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
  @Override
  public void bind(Long studioId, Long merchantId) {
    // 解绑后，绑定新商户
    this.unbindAll(studioId);
    PartyToMerchant relation = getRelation(studioId);
    StudioMerchantApply apply = studioMerchantApplyRpService.getLatestOneSuccessByMerchantId(merchantId);
    if (null == relation) {
      partyToMerchantRpService.insert(
          MerchantConvert.convertPartyToMerchant(apply.getChannelNo(), studioId, PartyTypeEnum.STUDIO,
              apply.getMerchantId(), apply.getMerchantNo()));
    } else {
      partyToMerchantRpService.updateById(
          MerchantConvert.convertPartyToMerchant(relation.getId(), merchantId, apply.getMerchantNo()));
    }
  }

  @Override
  public void unbind(Long studioId, Long merchantId) {
    partyToMerchantRpService.deleteByPartyAndMerchantId(studioId, PartyTypeEnum.STUDIO, merchantId);
  }

  @Override
  public void unbindAll(Long studioId) {
    partyToMerchantRpService.deleteByParty(studioId, PartyTypeEnum.STUDIO);
  }

  @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
  @Override
  public void callback(MerchantEvent event) {
    switch (event.getState()) {
      case PROCESS:
        this.doOpenMerchantProcess(Long.parseLong(event.getApplyNo()));
        break;
      case FAIL:
        this.doOpenMerchantFail(Long.parseLong(event.getApplyNo()), event.getFailMessage());
        break;
      case SUCCESS:
        this.doOpenMerchantSuccess(event);
        break;
      default:
        break;
    }
  }

  @Override
  public StudioMerchantVO getMerchant(Long studioId) {
    PartyToMerchant relation = getRelation(studioId);
    if (null == relation) {
      return null;
    }
    MerchantVO merchant = merchantRpcService.getByMerchantId(relation.getMerchantId());
    List<Long> resourceIds = this.getResourceIds(merchant);
    Map<Long, String> urlMap = resourceRpcService.getResourceUrlMap(studioId, resourceIds);
    return MerchantConvert.convertStudioMerchant(relation, merchant,
        configService.getDefaultBrandSubChannelInfo(), urlMap);
  }

  @Override
  public StudioMerchantApplyVO getApply(Long id) {
    StudioMerchantApply apply = studioMerchantApplyRpService.getById(id)
        .orElseThrow(() -> new BizException(FinanceErrorEnums.NOT_EXIST));
    return MerchantConvert.convertApply(apply);
  }

  @Override
  public PageVO<StudioMerchantApplyVO> pageApply(PageStudioMerchantApplyParams params) {
    Page<StudioMerchantApply> page = studioMerchantApplyRpService.page(BeanUtil.map(params,
        PageStudioMerchantApplyQuery.class));
    return PageVO.convert(page, MerchantConvert.convertApplyList(page.getContent()));
  }

  @Override
  public StudioMerchantBindXcxAppIdVO bindXcxAppId(Long studioId, String appId) {
    PartyToMerchant relation = getRelation(studioId);
    if (null == relation) {
      throw new BizException(FinanceErrorEnums.MERCHANT_NOT_OPEN);
    }
    ConfigSubChannelVO vo = merchantRpcService.bindAppIdWxXcxOffline(channel, relation.getMerchantId(),
        appId);

    return StudioMerchantBindXcxAppIdVO.builder().state(vo.getState()).failMessage(vo.getFailMessage())
        .build();
  }

  private PartyToMerchant getRelation(Long studioId) {
    List<PartyToMerchant> relations = partyToMerchantRpService.listByParty(studioId, PartyTypeEnum.STUDIO,
        channel);
    if (CollectionUtil.isEmpty(relations)) {
      return null;
    }
    return relations.get(0);
  }

  @Override
  public StudioMerchantAuthSubChannelVO authSubChannel(Long studioId, SubChannelEnums subChannel,
      SubChannelAuthTypeEnums authType) {
    PartyToMerchant relation = getRelation(studioId);
    if (null == relation) {
      throw new BizException(FinanceErrorEnums.MERCHANT_NOT_OPEN);
    }

    AuthSubChannelVO vo = merchantRpcService.authSubChannel(channel, relation.getMerchantId(), subChannel,
        authType);
    return BeanUtil.map(vo, StudioMerchantAuthSubChannelVO.class);
  }

  @Override
  public List<StudioMerchantAuthSubChannelVO> listAuthSubChannel(Long studioId) {
    PartyToMerchant relation = getRelation(studioId);
    if (null == relation) {
      throw new BizException(FinanceErrorEnums.MERCHANT_NOT_OPEN);
    }

    MerchantVO merchant = merchantRpcService.getByMerchantId(relation.getMerchantId());
    List<SubChannelAuthVO> auths = merchant.getSubChannelAuths();
    return Optional.ofNullable(auths).orElse(new ArrayList<>(1)).stream()
        .map(e -> BeanUtil.map(e, StudioMerchantAuthSubChannelVO.class)).collect(Collectors.toList());
  }


  private void doOpenMerchantSuccess(MerchantEvent event) {
    StudioMerchantApply apply = studioMerchantApplyRpService.getByIdForUpdate(
        Long.parseLong(event.getApplyNo())).orElseThrow(() -> new BizException(FinanceErrorEnums.NOT_EXIST));
    StudioMerchantApply toModify = MerchantConvert.convertApply(apply, event,
        configService.getDefaultBrandSubChannelInfo().getSubChannels());
    studioMerchantApplyRpService.updateById(toModify);
    if (OpenStateEnums.SUCCESS != toModify.getOpenState()) {
      return;
    }
    // 所有子通道全部开通成功
    PartyToMerchant old = getRelation(apply.getStudioId());

    // 1. 绑定场馆
    this.bind(apply.getStudioId(), event.getMerchantId());

    // 2. 绑定品牌
    brandMerchantService.bind(apply.getBrandId(), event.getMerchantId());

    // 3. 其他操作
    executor.execute(() -> this.doAfterMerchantSuccess(event,
        Optional.ofNullable(old).map(PartyToMerchant::getMerchantId).orElse(null)));

  }

  private void doAfterMerchantSuccess(MerchantEvent event, Long oldMerchantId) {
    // 1. 配置场馆小程序appId
    this.rebindMiniAppId(oldMerchantId, event.getMerchantId());
  }

  private void rebindMiniAppId(Long oldMerchantId, Long newMerchantId) {
    if (null == oldMerchantId) {
      return;
    }
    MerchantVO merchant = merchantRpcService.getByMerchantId(oldMerchantId);
    merchant.getSubChannelConfigs().stream()
        .filter(e -> SubChannelConfigTypeEnums.WX_XCX_OFFLINE.name().equals(e.getConfigType()))
        .forEach(
            config -> merchantRpcService.bindAppIdWxXcxOffline(merchant.getChannelNo(), newMerchantId,
                config.getAppId()));
  }

  private Map<ResourceTypeEnums, ResourceInfo> upload(ApplyStudioMerchantParams params) {

    Map<ResourceTypeEnums, ResourceInfo> resourceMap = MerchantConvert.convertResourceMap(
        params.getResource());

    Map<Long, ResourceInfoVO> resourceIdUrlMap = resourceRpcService.getResourceMap(params.getStudioId(),
        getResourceIds(resourceMap));

    resourceMap.keySet().forEach(type -> {
      ResourceInfo info = resourceMap.get(type);
      ResourceInfoVO vo = resourceIdUrlMap.get(info.getResourceId());
      String thirdId = merchantRpcService.upload(MerchantConvert.convertUploadRequest(channel, type, vo));
      info.setThirdId(thirdId);
    });
    return resourceMap;
  }

  private List<Long> getResourceIds(Map<ResourceTypeEnums, ResourceInfo> resourceMap) {
    return resourceMap.values().stream()
        .map(ResourceInfo::getResourceId)
        .filter(Objects::nonNull)
        .distinct()
        .collect(Collectors.toList());
  }

  private List<Long> getResourceIds(MerchantVO merchant) {
    return merchant.getResourceMap().values().stream()
        .map(e -> e.getResourceId())
        .filter(Objects::nonNull)
        .distinct()
        .collect(Collectors.toList());
  }

  private void doOpenMerchantFail(Long id, String failMessage) {
    studioMerchantApplyRpService.updateById(
        StudioMerchantApply.builder().id(id).openState(OpenStateEnums.FAIL)
            .openFailMessage(failMessage).build());
  }

  private void doOpenMerchantProcess(Long id) {
    studioMerchantApplyRpService.updateById(
        StudioMerchantApply.builder().id(id).openState(OpenStateEnums.PROCESS).build());
  }

}
