package com.jiejing.fitness.finance.service.utils;

import java.math.BigDecimal;
import java.math.RoundingMode;

/**
 * @author chengyubing
 * @since 2024/2/27 14:16
 */
public class FeeUtil {

  /**
   * 【支付】计算支付手续费，保留小数点后两位，四舍五入
   *
   * @param feeRate 费率（%）
   * @param amount  支付金额（元）
   * @return 手续费
   */
  public static BigDecimal calPayFee(BigDecimal feeRate, BigDecimal amount) {
    return MoneyUtil.defaultRound(MoneyUtil.multiply(
        MoneyUtil.divide(feeRate, new BigDecimal("100"), 4, RoundingMode.HALF_UP.ordinal()),
        amount));
  }

  /**
   * 【退款】计算退款手续费（元）
   *
   * <code>试算手续费 = 向下取整(退款金额/原交易金额*原交易手续费金额)</code>
   * <code>试算实退金额 = 退款申请金额-试算手续费</code>
   * <code>剩余实收金额 = 实收金额-已实退金额</code>
   * <code>若：剩余实收金额 >= 试算实退金额，则：应退手续费 = 试算手续费</code>
   * <code>若：剩余实收金额 < 试算实退金额，则：应退手续费 = 试算手续费 + (试算实退金额 - 剩余实收金额)</code>
   *
   * @param refundTransAmount         退款申请金额（元）
   * @param payTransAmount            原支付交易金额（元）
   * @param payActualAmount           用户实收金额（元）
   * @param payFee                    平台实收手续费（元）
   * @param historyRefundActualAmount 历史实退金额（元）
   * @return 本次退款应退手续费（元）
   */
  public static BigDecimal calculateRefundFee(BigDecimal refundTransAmount, BigDecimal payTransAmount,
      BigDecimal payActualAmount, BigDecimal payFee, BigDecimal historyRefundActualAmount) {
    // 机构剩余实收金额
    BigDecimal leftPayActualAmount = MoneyUtil.subtract(payActualAmount, historyRefundActualAmount);
    // 试算手续费：向下取整（退款金额/原交易金额*原支付交易手续费金额）
    BigDecimal trialFee = MoneyUtil.divide(refundTransAmount.multiply(payFee), payTransAmount, 2,
        RoundingMode.FLOOR);
    // 试算实退金额
    BigDecimal trialActualAmount = MoneyUtil.subtract(refundTransAmount, trialFee);
    // 实退手续费
    return leftPayActualAmount.compareTo(trialActualAmount) >= 0 ? trialFee
        : MoneyUtil.add(trialFee, MoneyUtil.subtract(trialActualAmount, leftPayActualAmount));
  }

  public static void main(String[] args) {
    BigDecimal refundTransAmount = new BigDecimal("100");
    BigDecimal payTransAmount = new BigDecimal("100");
    BigDecimal payActualAmount = new BigDecimal("99.43");
    BigDecimal payFee = new BigDecimal("0.57");
    BigDecimal historyRefundActualAmount = new BigDecimal("99.43");
    System.out.println(
        FeeUtil.calculateRefundFee(refundTransAmount, payTransAmount, payActualAmount, payFee,
            historyRefundActualAmount));
  }

}
