package com.jiejing.fitness.finance.service.merchant.convert;

import static com.jiejing.paycenter.common.enums.merchant.ResourceTypeEnums.*;

import com.alibaba.fastjson.JSON;
import com.jiejing.common.utils.convert.BeanUtil;
import com.jiejing.common.utils.crypt.AesUtil;
import com.jiejing.filecenter.api.resource.vo.ResourceInfoVO;
import com.jiejing.fitness.finance.api.enums.ApplyTypeEnums;
import com.jiejing.fitness.finance.api.enums.MerchantTypeEnums;
import com.jiejing.fitness.finance.api.merchant.model.AddressInfo;
import com.jiejing.fitness.finance.api.merchant.model.BrandBankCardInfo;
import com.jiejing.fitness.finance.api.merchant.model.BrandLegalInfo;
import com.jiejing.fitness.finance.api.merchant.model.BrandLicenseInfo;
import com.jiejing.fitness.finance.api.merchant.model.BrandResourceInfo;
import com.jiejing.fitness.finance.api.merchant.vo.BrandMerchantApplyVO;
import com.jiejing.fitness.finance.api.merchant.vo.BrandMerchantVO;
import com.jiejing.fitness.finance.repository.entity.BrandMerchantApply;
import com.jiejing.fitness.finance.repository.entity.BrandToMerchant;
import com.jiejing.fitness.finance.service.merchant.params.ApplyBrandMerchantParams;
import com.jiejing.paycenter.api.merchant.request.ApplyMerchantRequest;
import com.jiejing.paycenter.api.merchant.request.UploadRequest;
import com.jiejing.paycenter.api.merchant.vo.MerchantVO;
import com.jiejing.paycenter.common.enums.common.OpenStateEnums;
import com.jiejing.paycenter.common.enums.merchant.BusinessTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.LicenseTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.MccEnums;
import com.jiejing.paycenter.common.enums.merchant.ResourceTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SceneTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelAuthTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelOpenTypeEnums;
import com.jiejing.paycenter.common.event.MerchantEvent;
import com.jiejing.paycenter.common.model.Address;
import com.jiejing.paycenter.common.model.BankCard;
import com.jiejing.paycenter.common.model.BusinessInfo;
import com.jiejing.paycenter.common.model.Contact;
import com.jiejing.paycenter.common.model.Legal;
import com.jiejing.paycenter.common.model.License;
import com.jiejing.paycenter.common.model.ResourceInfo;
import com.jiejing.paycenter.common.model.SubChannelAuthInfo;
import com.jiejing.paycenter.common.model.SubChannelInfo;
import com.jiejing.studio.api.studio.vo.StudioVO;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @author chengyubing
 * @since 2024/2/20 16:46
 */
public class MerchantConvert {

  public static ApplyMerchantRequest convertRequest(String channelNo, String applyNo,
      ApplyBrandMerchantParams params, Map<ResourceTypeEnums, ResourceInfo> resourceMap,
      ApplyMerchantRequest subChannelInfo) {
    return ApplyMerchantRequest.builder()
        .bizCode(MerchantTypeEnums.BRAND.getCode())
        .applyNo(applyNo)
        .channelNo(channelNo)
        .companyType(params.getCompanyType())
        .merchantName(params.getMerchantName())
        .shortName(params.getShortName())
        .legal(convertLegal(params))
        .bankCard(convertBankCard(params))
        .license(convertLicense(params))
        .business(convertBusiness(params))
        .contact(convertContact(params))
        .resourceMap(resourceMap)
        .subChannels(subChannelInfo.getSubChannels())
        .subChannelConfigs(subChannelInfo.getSubChannelConfigs())
        .build();
  }


  private static Contact convertContact(ApplyBrandMerchantParams params) {
    return Contact.builder()
        .contactName(params.getLegal().getLegalName())
        .contactPhone(params.getLegal().getLegalPhone())
        .contactEmail(params.getLegal().getLegalEmail())
        .servicePhone(params.getLegal().getLegalPhone())
        .build();
  }

  private static BusinessInfo convertBusiness(ApplyBrandMerchantParams params) {
    AddressInfo address = params.getLicense().getLicenseAddress();
    return BusinessInfo.builder()
        .mcc(MccEnums.MCC_7941)
        .businessType(BusinessTypeEnums.REALITY)
        .sceneType(SceneTypeEnums.OFFLINE)
        .businessAddress(Address.builder()
            .province(null == address ? null : address.getProvince())
            .city(null == address ? null : address.getCity())
            .district(null == address ? null : address.getDistrict())
            .address(null == address ? null : address.getAddress())
            .build())
        .build();
  }

  private static License convertLicense(ApplyBrandMerchantParams params) {
    AddressInfo address = params.getLicense().getLicenseAddress();
    return License.builder()
        .licenseType(LicenseTypeEnums.NATIONAL_LEGAL_MERGE)
        .registeredCapital("10000")
        .licenseCode(params.getLicense().getLicenseCode())
        .licenseExpireType(params.getLicense().getLicenseExpireType())
        .licenseStartDate(params.getLicense().getLicenseStartDate())
        .licenseEndDate(params.getLicense().getLicenseEndDate())
        .businessScope(params.getLicense().getBusinessScope())
        .foundDate(params.getLicense().getFoundDate())
        .licenseAddress(Address.builder()
            .province(null == address ? null : address.getProvince())
            .city(null == address ? null : address.getCity())
            .district(null == address ? null : address.getDistrict())
            .address(null == address ? null : address.getAddress())
            .build())
        .build();
  }

  private static BankCard convertBankCard(ApplyBrandMerchantParams params) {
    AddressInfo address = params.getBankCard().getCardAddress();
    return BankCard.builder()
        .cardType(params.getBankCard().getCardType())
        .cardNo(params.getBankCard().getCardNo())
        .cardName(params.getBankCard().getCardName())
        .phone(params.getBankCard().getPhone())
        .certNo(params.getBankCard().getCertNo())
        .certType(params.getBankCard().getCertType())
        .baseDepositNo(params.getBankCard().getBaseDepositNo())
        .certExpireType(params.getBankCard().getCertExpireType())
        .certStartDate(params.getBankCard().getCertStartDate())
        .certEndDate(params.getBankCard().getCertEndDate())
        .bankCode(params.getBankCard().getBankCode())
        .bankName(params.getBankCard().getBankName())
        .branchName(params.getBankCard().getBranchName())
        .branchCode(params.getBankCard().getBranchCode())
        .cardAddress(Address.builder()
            .province(null == address ? null : address.getProvince())
            .city(null == address ? null : address.getCity())
            .district(null == address ? null : address.getDistrict())
            .address(null == address ? null : address.getAddress())
            .build())
        .build();
  }

  private static Legal convertLegal(ApplyBrandMerchantParams params) {
    AddressInfo address = params.getLegal().getLegalAddress();
    return Legal.builder()
        .legalName(params.getLegal().getLegalName())
        .legalPhone(params.getLegal().getLegalPhone())
        .legalCertNo(params.getLegal().getLegalCertNo())
        .legalCertType(params.getLegal().getLegalCertType())
        .legalCertExpireType(params.getLegal().getLegalCertExpireType())
        .legalCertStartDate(params.getLegal().getLegalCertStartDate())
        .legalCertEndDate(params.getLegal().getLegalCertEndDate())
        .legalAddress(Address.builder()
            .province(null == address ? null : address.getProvince())
            .city(null == address ? null : address.getCity())
            .district(null == address ? null : address.getDistrict())
            .address(null == address ? null : address.getAddress())
            .build())
        .build();
  }

  public static BrandMerchantApply convertApply(Long id, ApplyBrandMerchantParams params, StudioVO studio,
      BrandMerchantApply exist, String channel) {
    String salt = AesUtil.getSalt(8);
    ApplyBrandMerchantParams encrypt = params.encrypt(salt);
    return BrandMerchantApply.builder()
        .id(id)
        .applyNo(Long.toString(id))
        .brandId(studio.getBrandId())
        .studioId(encrypt.getStudioId())
        .applyType(exist == null ? ApplyTypeEnums.OPEN : ApplyTypeEnums.RE_OPEN)
        .channelNo(channel)
        .merchantName(encrypt.getMerchantName())
        .shortName(encrypt.getShortName())
        .companyType(encrypt.getCompanyType())
        .openState(OpenStateEnums.INIT)
        .merchantInfo(convertMerchantInfo(encrypt))
        .salt(salt)
        .createTime(new Date())
        .updateTime(new Date())
        .build();
  }

  private static String convertMerchantInfo(ApplyBrandMerchantParams encrypt) {
    return JSON.toJSONString(ApplyBrandMerchantParams.builder()
        .legal(encrypt.getLegal())
        .license(encrypt.getLicense())
        .bankCard(encrypt.getBankCard())
        .resource(encrypt.getResource())
        .build());
  }

  public static UploadRequest convertUploadRequest(String channel, ResourceTypeEnums type,
      ResourceInfoVO vo) {
    return UploadRequest.builder()
        .url(vo.getUrl())
        .fileSize(vo.getFileSize())
        .suffix(vo.getSuffix())
        .channelNo(channel)
        .resourceType(type)
        .build();
  }

  public static BrandMerchantApply convertApply(BrandMerchantApply apply, MerchantEvent event,
      List<SubChannelInfo> defaultSubChannels) {
    // 品牌商户规则：微信线下公众号、微信线下小程序、支付宝线下通道全部开通成功才算成功
    Map<SubChannelOpenTypeEnums, SubChannelInfo> channelMap = convertSubChannelMap(event.getSubChannels(),
        defaultSubChannels);
    OpenStateEnums state = convertOpenState(channelMap, defaultSubChannels);
    String openFailMessage = OpenStateEnums.FAIL == state ? convertFailMessage(event.getSubChannels()) : null;
    SubChannelInfo ali = channelMap.get(SubChannelOpenTypeEnums.ALI_OFFLINE);
    SubChannelInfo wxGzhOffline = channelMap.get(SubChannelOpenTypeEnums.WX_GZH_OFFLINE);
    OpenStateEnums wxOfflineState = convertWxOfflineState(channelMap);
    String wxOfflineFailMessage = convertWxOfflineFailMessage(channelMap);
    return BrandMerchantApply.builder()
        .id(apply.getId())
        .brandId(apply.getBrandId())
        .studioId(apply.getStudioId())
        .applyNo(apply.getApplyNo())
        .applyType(apply.getApplyType())
        .channelNo(apply.getChannelNo())
        .merchantId(event.getMerchantId())
        .merchantNo(event.getMerchantNo())
        .merchantName(event.getMerchantName())
        .shortName(event.getShortName())
        .companyType(event.getCompanyType())
        .openState(state)
        .openFailMessage(openFailMessage)
        .aliOpenState(ali.getState())
        .aliAuthState(convertAliAuthState(event.getSubChannelAuths()))
        .aliMerchantNo(JSON.toJSONString(ali.getMerchantNos()))
        .aliOpenFailMessage(ali.getFailMessage())
        .wxOfflineOpenState(wxOfflineState)
        .wxOfflineMerchantNo(JSON.toJSONString(wxGzhOffline.getMerchantNos()))
        .wxOfflineAuthState(convertWxOfflineAuthState(event.getSubChannelAuths()))
        .wxOfflineOpenFailMessage(wxOfflineFailMessage)
        .finishTime(OpenStateEnums.SUCCESS == state ? new Date() : null)
        .updateTime(new Date())
        .build();
  }

  private static BrandLicenseInfo convertLicense(License license) {
    return JSON.parseObject(JSON.toJSONString(license), BrandLicenseInfo.class);
  }

  private static BrandLegalInfo convertLegal(Legal legal, Contact contact) {
    BrandLegalInfo legalInfo = JSON.parseObject(JSON.toJSONString(legal), BrandLegalInfo.class);
    legalInfo.setLegalEmail(contact.getContactEmail());
    return legalInfo;
  }

  private static BrandBankCardInfo convertBankCard(BankCard bankCard) {
    return JSON.parseObject(JSON.toJSONString(bankCard), BrandBankCardInfo.class);
  }

  private static BrandResourceInfo convertResource(Map<ResourceTypeEnums, ResourceInfo> resourceMap) {
    BrandResourceInfo resourceInfo = new BrandResourceInfo();
    resourceInfo.setLicensePicId(resourceMap.getOrDefault(LICENSE, new ResourceInfo()).getResourceId());
    resourceInfo.setStoreHeaderPicId(
        resourceMap.getOrDefault(STORE_HEADER_PIC, new ResourceInfo()).getResourceId());
    resourceInfo.setStoreInnerPicId(
        resourceMap.getOrDefault(STORE_INNER_PIC, new ResourceInfo()).getResourceId());
    resourceInfo.setStoreCashierDescPicId(
        resourceMap.getOrDefault(STORE_CASHIER_DESK_PIC, new ResourceInfo()).getResourceId());
    resourceInfo.setLegalCertFrontPicId(
        resourceMap.getOrDefault(LEGAL_CERT_FRONT, new ResourceInfo()).getResourceId());
    resourceInfo.setLegalCertBackPicId(
        resourceMap.getOrDefault(LEGAL_CERT_BACK, new ResourceInfo()).getResourceId());
    resourceInfo.setBankCardFrontPicId(
        resourceMap.getOrDefault(BANK_CARD_FRONT, new ResourceInfo()).getResourceId());
    resourceInfo.setBankCardBackPicId(
        resourceMap.getOrDefault(BANK_CARD_BACK, new ResourceInfo()).getResourceId());
    resourceInfo.setBankCardCertFrontPicId(
        resourceMap.getOrDefault(BANK_CARD_CERT_FRONT, new ResourceInfo()).getResourceId());
    resourceInfo.setBankCardCertBackPicId(
        resourceMap.getOrDefault(BANK_CARD_CERT_BACK, new ResourceInfo()).getResourceId());
    return resourceInfo;
  }

  private static OpenStateEnums convertWxOfflineAuthState(List<SubChannelAuthInfo> auths) {
    return auths.stream()
        .filter(e -> e.getAuthType() == SubChannelAuthTypeEnums.WX_OFFLINE)
        .findFirst()
        .map(SubChannelAuthInfo::getState)
        .orElse(null);
  }

  private static OpenStateEnums convertAliAuthState(List<SubChannelAuthInfo> auths) {
    return auths.stream()
        .filter(e -> e.getAuthType() == SubChannelAuthTypeEnums.ALI_OFFLINE)
        .findFirst()
        .map(SubChannelAuthInfo::getState)
        .orElse(null);
  }

  private static String convertWxOfflineFailMessage(Map<SubChannelOpenTypeEnums, SubChannelInfo> channelMap) {
    SubChannelInfo wxGzhOffline = channelMap.get(SubChannelOpenTypeEnums.WX_GZH_OFFLINE);
    SubChannelInfo wxXcxOffline = channelMap.get(SubChannelOpenTypeEnums.WX_XCX_OFFLINE);
    if (OpenStateEnums.FAIL == wxGzhOffline.getState()) {
      return wxGzhOffline.getFailMessage();
    }
    if (OpenStateEnums.FAIL == wxXcxOffline.getState()) {
      return wxXcxOffline.getFailMessage();
    }
    return null;
  }

  private static OpenStateEnums convertWxOfflineState(
      Map<SubChannelOpenTypeEnums, SubChannelInfo> channelMap) {
    SubChannelInfo wxGzhOffline = channelMap.get(SubChannelOpenTypeEnums.WX_GZH_OFFLINE);
    SubChannelInfo wxXcxOffline = channelMap.get(SubChannelOpenTypeEnums.WX_XCX_OFFLINE);
    if (OpenStateEnums.SUCCESS == wxGzhOffline.getState()
        && OpenStateEnums.SUCCESS == wxXcxOffline.getState()) {
      return OpenStateEnums.SUCCESS;
    }
    if (OpenStateEnums.FAIL == wxGzhOffline.getState()) {
      return OpenStateEnums.FAIL;
    }
    if (OpenStateEnums.FAIL == wxXcxOffline.getState()) {
      return OpenStateEnums.FAIL;
    }
    return OpenStateEnums.PROCESS;
  }

  private static String convertFailMessage(List<SubChannelInfo> subChannels) {
    return subChannels.stream()
        .filter(e -> OpenStateEnums.FAIL == e.getState())
        .map(SubChannelInfo::getFailMessage)
        .findFirst()
        .orElse(null);
  }

  private static OpenStateEnums convertOpenState(Map<SubChannelOpenTypeEnums, SubChannelInfo> channelMap,
      List<SubChannelInfo> defaultSubChannels) {

    List<SubChannelOpenTypeEnums> sceneList = getDefaultScene(defaultSubChannels);

    OpenStateEnums state = OpenStateEnums.SUCCESS;
    for (SubChannelOpenTypeEnums scene : sceneList) {
      SubChannelInfo subChannel = channelMap.get(scene);
      if (OpenStateEnums.PROCESS == subChannel.getState()) {
        state = OpenStateEnums.PROCESS;
      }
      if (OpenStateEnums.FAIL == subChannel.getState()) {
        return OpenStateEnums.FAIL;
      }
    }
    return state;
  }

  private static Map<SubChannelOpenTypeEnums, SubChannelInfo> convertSubChannelMap(
      List<SubChannelInfo> subChannels, List<SubChannelInfo> defaultSubChannels) {
    Map<SubChannelOpenTypeEnums, SubChannelInfo> map = subChannels.stream()
        .collect(Collectors.toMap(SubChannelInfo::getOpenType, e -> e));
    for (SubChannelInfo defaultSubChannel : defaultSubChannels) {
      if (null == map.get(defaultSubChannel.getOpenType())) {
        map.put(defaultSubChannel.getOpenType(),
            SubChannelInfo.builder().openType(defaultSubChannel.getOpenType()).state(OpenStateEnums.PROCESS)
                .build());
      }
    }
    return map;
  }

  public static List<SubChannelOpenTypeEnums> getDefaultScene(List<SubChannelInfo> defaultSubChannels) {
    return defaultSubChannels.stream().map(SubChannelInfo::getOpenType)
        .collect(Collectors.toList());
  }

  public static BrandToMerchant convertMerchant(BrandMerchantApply apply) {
    BrandToMerchant merchant = BeanUtil.map(apply, BrandToMerchant.class);
    merchant.setId(apply.getBrandId());
    merchant.setCreateTime(new Date());
    merchant.setUpdateTime(new Date());
    return merchant;
  }

  public static Map<ResourceTypeEnums, ResourceInfo> convertResourceMap(BrandResourceInfo resource) {
    Map<ResourceTypeEnums, ResourceInfo> map = new HashMap<>(1);
    if (null != resource.getLicensePicId()) {
      map.put(LICENSE, convert(resource.getLicensePicId(), LICENSE));
    }
    if (null != resource.getStoreHeaderPicId()) {
      map.put(STORE_HEADER_PIC, convert(resource.getStoreHeaderPicId(), STORE_HEADER_PIC));
    }
    if (null != resource.getStoreInnerPicId()) {
      map.put(STORE_INNER_PIC, convert(resource.getStoreInnerPicId(), STORE_INNER_PIC));
    }
    if (null != resource.getStoreCashierDescPicId()) {
      map.put(STORE_CASHIER_DESK_PIC, convert(resource.getStoreCashierDescPicId(), STORE_CASHIER_DESK_PIC));
    }
    if (null != resource.getLegalCertFrontPicId()) {
      map.put(LEGAL_CERT_FRONT, convert(resource.getLegalCertFrontPicId(), LEGAL_CERT_FRONT));
    }
    if (null != resource.getLegalCertBackPicId()) {
      map.put(LEGAL_CERT_BACK, convert(resource.getLegalCertBackPicId(), LEGAL_CERT_BACK));
    }
    if (null != resource.getBankCardFrontPicId()) {
      map.put(BANK_CARD_FRONT, convert(resource.getBankCardFrontPicId(), BANK_CARD_FRONT));
    }
    if (null != resource.getBankCardBackPicId()) {
      map.put(BANK_CARD_BACK, convert(resource.getBankCardBackPicId(), BANK_CARD_BACK));
    }
    if (null != resource.getBankCardCertFrontPicId()) {
      map.put(BANK_CARD_CERT_FRONT, convert(resource.getBankCardCertFrontPicId(), BANK_CARD_CERT_FRONT));
    }
    if (null != resource.getBankCardCertBackPicId()) {
      map.put(BANK_CARD_CERT_BACK, convert(resource.getBankCardCertBackPicId(), BANK_CARD_CERT_BACK));
    }
    return map;
  }

  private static ResourceInfo convert(Long id, ResourceTypeEnums type) {
    return ResourceInfo.builder().resourceId(id).type(type).build();
  }

  public static BrandMerchantVO convertBrandMerchant(BrandToMerchant relation, MerchantVO merchant,
      List<SubChannelInfo> defaultSubChannels) {
    Map<SubChannelOpenTypeEnums, SubChannelInfo> channelMap = convertSubChannelMap(merchant.getSubChannels(),
        defaultSubChannels);
    SubChannelInfo ali = channelMap.get(SubChannelOpenTypeEnums.ALI_OFFLINE);
    SubChannelInfo wxGzhOffline = channelMap.get(SubChannelOpenTypeEnums.WX_GZH_OFFLINE);
    OpenStateEnums wxOfflineState = convertWxOfflineState(channelMap);
    String wxOfflineFailMessage = convertWxOfflineFailMessage(channelMap);
    return BrandMerchantVO.builder()
        .brandId(relation.getBrandId())
        .channelNo(merchant.getChannelNo())
        .merchantId(merchant.getId())
        .merchantNo(merchant.getMerchantNo())
        .merchantName(merchant.getMerchantName())
        .shortName(merchant.getShortName())
        .companyType(merchant.getCompanyType())
        .aliOpenState(ali.getState())
        .aliAuthState(convertAliAuthState(merchant.getSubChannelAuths()))
        .aliMerchantNo(JSON.toJSONString(ali.getMerchantNos()))
        .aliOpenFailMessage(ali.getFailMessage())
        .wxOfflineOpenState(wxOfflineState)
        .wxOfflineMerchantNo(JSON.toJSONString(wxGzhOffline.getMerchantNos()))
        .wxOfflineAuthState(convertWxOfflineAuthState(merchant.getSubChannelAuths()))
        .wxOfflineOpenFailMessage(wxOfflineFailMessage)
        .license(convertLicense(merchant.getLicense()))
        .legal(convertLegal(merchant.getLegal(), merchant.getContact()))
        .bankCard(convertBankCard(merchant.getBankCard()))
        .resource(convertResource(merchant.getResourceMap()))
        .build();
  }

  public static BrandMerchantApplyVO convertApply(BrandMerchantApply apply) {
    ApplyBrandMerchantParams info = JSON.parseObject(apply.getMerchantInfo(),
        ApplyBrandMerchantParams.class).decrypt(apply.getSalt());
    BrandMerchantApplyVO vo = BeanUtil.map(apply, BrandMerchantApplyVO.class);
    vo.setLegal(info.getLegal());
    vo.setResource(info.getResource());
    vo.setLicense(info.getLicense());
    vo.setBankCard(info.getBankCard());
    return vo;
  }

  public static List<BrandMerchantApplyVO> convertApplyList(List<BrandMerchantApply> applies) {
    return Optional.ofNullable(applies)
        .orElse(new ArrayList<>())
        .stream()
        .map(MerchantConvert::convertApply)
        .collect(Collectors.toList());
  }
}
