package com.jiejing.fitness.finance.service.pay.convert;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.google.common.collect.Lists;
import com.jiejing.common.exception.BizException;
import com.jiejing.common.utils.convert.BeanUtil;
import com.jiejing.common.utils.text.StringUtil;
import com.jiejing.fitness.finance.api.enums.BrandCashierTransStateEnums;
import com.jiejing.fitness.finance.api.enums.BrandCashierTransTypeEnums;
import com.jiejing.fitness.finance.repository.entity.BrandCashierRecord;
import com.jiejing.fitness.finance.repository.entity.BrandToMerchant;
import com.jiejing.fitness.finance.service.enums.FinanceErrorEnums;
import com.jiejing.fitness.finance.service.pay.params.BrandMerchantPayParams;
import com.jiejing.fitness.finance.service.pay.params.BrandMerchantRefundParams;
import com.jiejing.fitness.finance.service.utils.FeeUtil;
import com.jiejing.fitness.finance.service.utils.MoneyUtil;
import com.jiejing.paycenter.api.merchant.vo.MerchantVO;
import com.jiejing.paycenter.api.pay.request.PayRequest;
import com.jiejing.paycenter.common.enums.common.TransStateEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelOpenTypeEnums;
import com.jiejing.paycenter.common.enums.pay.PayStateEnums;
import com.jiejing.paycenter.common.enums.pay.PayTypeEnums;
import com.jiejing.paycenter.common.event.PayEvent;
import com.jiejing.paycenter.common.model.SubChannelInfo;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Date;
import java.util.List;
import java.util.Optional;

/**
 * @author chengyubing
 * @since 2024/2/27 11:08
 */
public class PayConvert {

  private static final List<String> ALI_AUTH_CODE_PREFIX = Lists.newArrayList("25", "26", "27", "28", "29",
      "30");

  private static final List<String> WX_AUTH_CODE_PREFIX = Lists.newArrayList("10", "11", "12", "13", "14",
      "15");

  public static PayRequest convert(BrandMerchantPayParams params, BrandCashierRecord record,
      MerchantVO merchant) {
    PayRequest request = BeanUtil.map(params, PayRequest.class);
    request.setTransNo(record.getTransNo());
    request.setMerchantId(merchant.getId());
    request.setExtra(Optional.ofNullable(params.getExtra()).orElse(new JSONObject())
        .fluentPut("studioId", params.getStudioId())
        .fluentPut("buyerName", params.getBuyerName())
        .fluentPut("buyerPhone", params.getBuyerPhone())
        .fluentPut("remark", params.getRemark())
    );
    request.setTransTime(new Date());
    return request;
  }

  public static BrandCashierRecord convertPayInit(BrandMerchantPayParams params, BrandToMerchant relation,
      MerchantVO merchant) {
    setSubChannel(params);
    Long id = IdWorker.getId();
    BigDecimal feeRate = getFeeRate(params, merchant);
    BigDecimal fee = FeeUtil.calPayFee(feeRate, params.getTransAmount());
    return BrandCashierRecord.builder()
        .id(id)
        .transNo(id.toString())
        .orderNo(params.getOrderNo())
        .orderType(params.getOrderType())
        .transType(BrandCashierTransTypeEnums.PAY)
        .brandId(relation.getBrandId())
        .studioId(params.getStudioId())
        .merchantId(merchant.getId())
        .merchantNo(merchant.getMerchantNo())
        .channelNo(merchant.getChannelNo())
        .subChannel(params.getSubChannel())
        .transAmount(params.getTransAmount())
        .feeRate(feeRate)
        .fee(fee)
        .actualAmount(MoneyUtil.subtract(params.getTransAmount(), fee))
        .transState(BrandCashierTransStateEnums.PAY_INIT)
        .buyerName(params.getBuyerName())
        .buyerPhone(params.getBuyerPhone())
        .goods(params.getGoods())
        .remark(params.getRemark())
        .existRelatedTrans(false)
        .tradingTime(new Date())
        .createTime(new Date())
        .updateTime(new Date())
        .build();
  }

  private static void setSubChannel(BrandMerchantPayParams params) {
    if (PayTypeEnums.BARCODE != params.getPayType() || StringUtil.isBlank(params.getAuthCode())) {
      // 未设置子渠道，不是付款码支付或付款码为空，直接返回
      return;
    }

    // https://juejin.cn/post/7099730102203727902
    // 支付宝支付码生成规则：25 - 30开头的长度为16~24位的数字，实际字符串长度以开发者获取的付款码长度为准
    if (ALI_AUTH_CODE_PREFIX.stream().anyMatch(prefix -> params.getAuthCode().startsWith(prefix))) {
      params.setSubChannel(SubChannelEnums.ALI);
    }
    // 微信支付码生成规则：18位纯数字，以10、11、12、13、14、15开头
    if (WX_AUTH_CODE_PREFIX.stream().anyMatch(prefix -> params.getAuthCode().startsWith(prefix))) {
      params.setSubChannel(SubChannelEnums.WX);
    }
  }

  private static BigDecimal getFeeRate(BrandMerchantPayParams params, MerchantVO merchant) {
    SubChannelOpenTypeEnums scene;
    switch (params.getPayType()) {
      case MINI:
        scene = SubChannelEnums.WX == params.getSubChannel() ? SubChannelOpenTypeEnums.WX_XCX_OFFLINE
            : SubChannelOpenTypeEnums.ALI_OFFLINE;
        break;
      case JS:
        scene = SubChannelEnums.WX == params.getSubChannel() ? SubChannelOpenTypeEnums.WX_GZH_OFFLINE
            : SubChannelOpenTypeEnums.ALI_OFFLINE;
        break;
      default:
        throw new BizException(FinanceErrorEnums.NOT_SUPPORT_TYPE);
    }
    SubChannelInfo subChannelInfo = merchant.getSubChannels().stream()
        .filter(e -> scene == e.getOpenType())
        .findFirst()
        .orElseThrow(() -> new BizException(FinanceErrorEnums.CHANNEL_NOT_OPEN));
    return subChannelInfo.getFeeRate();
  }

  public static void main(String[] args) {
    BigDecimal feeRate = new BigDecimal("0.38");
    BigDecimal fee = MoneyUtil.defaultRound(MoneyUtil.multiply(
        MoneyUtil.divide(feeRate, new BigDecimal("100"), 4, RoundingMode.HALF_UP.ordinal()),
        new BigDecimal("168")));
    System.out.println(fee);
  }

  public static BrandCashierRecord convertPay(BrandCashierRecord record, PayEvent event) {
    return BrandCashierRecord.builder()
        .id(record.getId())
        .successTime(event.getFinishTime())
        .channelNo(event.getChannelNo())
        .transState(convertTransState(event.getPayState()))
        .failMessage(event.getFailMsg())
        .thirdTransNo(event.getThirdTransNo())
        .updateTime(new Date())
        .build();
  }

  private static BrandCashierTransStateEnums convertTransState(PayStateEnums state) {
    switch (state) {
      case SUCCESS:
        return BrandCashierTransStateEnums.PAY_SUCCESS;
      case CLOSED:
      case FAILED:
        return BrandCashierTransStateEnums.PAY_FAIL;
      default:
        return null;
    }
  }

}
