package com.jiejing.fitness.finance.app.controller.cashier;

import com.jiejing.common.model.JsonResult;
import com.jiejing.common.model.PageVO;
import com.jiejing.common.utils.convert.BeanUtil;
import com.jiejing.fitness.finance.api.cashier.StudioCashierApi;
import com.jiejing.fitness.finance.api.cashier.request.BindStudioCashierRequest;
import com.jiejing.fitness.finance.api.cashier.request.GetStudioCashierByIdRequest;
import com.jiejing.fitness.finance.api.cashier.request.GetStudioSettleRequest;
import com.jiejing.fitness.finance.api.cashier.request.ListRelatedStudioCashierRecordRequest;
import com.jiejing.fitness.finance.api.cashier.request.ListStudioCashierByIdsRequest;
import com.jiejing.fitness.finance.api.cashier.request.ListStudioCashierByOrderNoRequest;
import com.jiejing.fitness.finance.api.cashier.request.ListStudioCashierBySubOrderNoRequest;
import com.jiejing.fitness.finance.api.cashier.request.ListStudioCashierByTransNoRequest;
import com.jiejing.fitness.finance.api.cashier.request.PageStudioCashierRequest;
import com.jiejing.fitness.finance.api.cashier.request.PageStudioSettleRequest;
import com.jiejing.fitness.finance.api.cashier.request.UnbindStudioCashierRequest;
import com.jiejing.fitness.finance.api.cashier.vo.StudioCashierRecordVO;
import com.jiejing.fitness.finance.api.cashier.vo.StudioCashierStatisticVO;
import com.jiejing.fitness.finance.api.cashier.vo.StudioSettleVO;
import com.jiejing.fitness.finance.service.cashier.StudioCashierService;
import com.jiejing.fitness.finance.service.cashier.params.PageStudioCashierParams;
import com.jiejing.fitness.finance.service.cashier.params.PageStudioSettleParams;
import com.jiejing.paycenter.common.enums.common.PayChannelEnums;
import io.swagger.annotations.ApiOperation;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import javax.annotation.Resource;
import javax.validation.Valid;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

/**
 * @author chengyubing
 * @since 2024/2/29 10:41
 */
@RestController
public class StudioCashierController implements StudioCashierApi {

  @Resource
  private StudioCashierService studioCashierService;

  @ApiOperation(value = "获取收银所需appId", tags = {TAG})
  @PostMapping(value = "/private/studioCashier/getAppId")
  @Override
  public JsonResult<Map<PayChannelEnums, String>> getAppId() {
    return JsonResult.success(studioCashierService.getAppId());
  }

  @ApiOperation(value = "绑定流水", tags = {TAG})
  @PostMapping(value = "/private/studioCashier/bind")
  @Override
  public JsonResult<Void> bind(@Valid @RequestBody BindStudioCashierRequest request) {
    studioCashierService.bind(request.getIds(), request.getOrderNo(), request.getBusinessNo(),
        request.getSubOrderNo());
    return JsonResult.success();
  }

  @ApiOperation(value = "解绑流水", tags = {TAG})
  @PostMapping(value = "/private/studioCashier/unbind")
  @Override
  public JsonResult<Void> unbind(@Valid @RequestBody UnbindStudioCashierRequest request) {
    studioCashierService.unbind(request.getIds());
    return JsonResult.success();
  }

  @ApiOperation(value = "分页筛选收银流水", tags = {TAG})
  @PostMapping(value = "private/studioCashier/page")
  @Override
  public JsonResult<PageVO<StudioCashierRecordVO>> page(
      @Valid @RequestBody PageStudioCashierRequest request) {
    PageStudioCashierParams params = BeanUtil.map(request, PageStudioCashierParams.class);
    params.setCurrent(Optional.ofNullable(request.getCurrent()).orElse(0));
    params.setSize(Optional.ofNullable(request.getSize()).orElse(10));
    return JsonResult.success(studioCashierService.page(params));
  }

  @ApiOperation(value = "统计收银流水金额", tags = {TAG})
  @PostMapping(value = "/private/studioCashier/statistic")
  @Override
  public JsonResult<StudioCashierStatisticVO> statistic(
      @Valid @RequestBody PageStudioCashierRequest request) {
    PageStudioCashierParams params = BeanUtil.map(request, PageStudioCashierParams.class);
    return JsonResult.success(studioCashierService.statistic(params));
  }

  @ApiOperation(value = "查询收银流水详情", tags = {TAG})
  @PostMapping(value = "/private/studioCashier/getById")
  @Override
  public JsonResult<StudioCashierRecordVO> getById(@Valid @RequestBody GetStudioCashierByIdRequest request) {
    return JsonResult.success(studioCashierService.getById(request.getId()));
  }

  @ApiOperation(value = "查询关联收银流水", tags = {TAG})
  @PostMapping(value = "/private/studioCashier/listByRelatedTransNo")
  @Override
  public JsonResult<List<StudioCashierRecordVO>> listByRelatedTransNo(
      @Valid @RequestBody ListRelatedStudioCashierRecordRequest request) {
    return JsonResult.success(studioCashierService.listByRelatedTransNo(request.getRelatedTransNo()));
  }

  @ApiOperation(value = "查询收银流水详情列表", tags = {TAG})
  @PostMapping(value = "/private/studioCashier/listByIds")
  @Override
  public JsonResult<List<StudioCashierRecordVO>> listByIds(
      @Valid @RequestBody ListStudioCashierByIdsRequest request) {
    return JsonResult.success(studioCashierService.listByIds(request.getIds()));
  }

  @ApiOperation(value = "查询收银流水详情列表", tags = {TAG})
  @PostMapping(value = "/private/studioCashier/listByTransNo")
  @Override
  public JsonResult<List<StudioCashierRecordVO>> listByTransNo(
      @Valid @RequestBody ListStudioCashierByTransNoRequest request) {
    return JsonResult.success(studioCashierService.listByTransNo(request.getTransNos()));
  }

  @ApiOperation(value = "根据订单号查询收银列表", tags = {TAG})
  @PostMapping(value = "/private/studioCashier/listByOrderNo")
  @Override
  public JsonResult<List<StudioCashierRecordVO>> listByOrderNo(
      @Valid @RequestBody ListStudioCashierByOrderNoRequest request) {
    return JsonResult.success(studioCashierService.listByOrderNo(request.getOrderNo()));
  }

  @ApiOperation(value = "根据子订单号查询收银列表", tags = {TAG})
  @PostMapping(value = "/private/studioCashier/listBySubOrderNo")
  @Override
  public JsonResult<List<StudioCashierRecordVO>> listBySubOrderNo(
      @Valid @RequestBody ListStudioCashierBySubOrderNoRequest request) {
    return JsonResult.success(studioCashierService.listBySubOrderNo(request.getSubOrderNos()));
  }

  @ApiOperation(value = "分页筛选结算记录", tags = {TAG})
  @PostMapping(value = "/private/studioCashier/pageSettle")
  @Override
  public JsonResult<PageVO<StudioSettleVO>> pageSettle(@Valid @RequestBody PageStudioSettleRequest request) {
    PageStudioSettleParams params = BeanUtil.map(request, PageStudioSettleParams.class);
    return JsonResult.success(studioCashierService.page(params));
  }

  @ApiOperation(value = "查询结算记录", tags = {TAG})
  @PostMapping(value = "/private/studioCashier/getSettle")
  @Override
  public JsonResult<StudioSettleVO> getSettle(@Valid @RequestBody GetStudioSettleRequest request) {
    return JsonResult.success(studioCashierService.getSettle(request.getId()));
  }

}
