package com.jiejing.fitness.finance.service.axf.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.dynamic.datasource.annotation.DS;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.jiejing.common.enums.ConfirmEnum;
import com.jiejing.common.exception.BizException;
import com.jiejing.common.model.JsonResult;
import com.jiejing.common.model.PageVO;
import com.jiejing.common.request.IdRequest;
import com.jiejing.common.request.IdsRequest;
import com.jiejing.common.utils.convert.BeanUtil;
import com.jiejing.common.utils.crypt.AesUtil;
import com.jiejing.fitness.enums.delay.DelayTaskCodeEnum;
import com.jiejing.fitness.enums.finance.PartyTypeEnum;
import com.jiejing.fitness.finance.api.axf.enums.AxfOrderSignStateEnums;
import com.jiejing.fitness.finance.api.axf.enums.AxfStateEnums;
import com.jiejing.fitness.finance.api.axf.request.CreateStudioAxfCommodityRequest;
import com.jiejing.fitness.finance.api.axf.request.PageStudioAxfApplyRequest;
import com.jiejing.fitness.finance.api.axf.request.PageStudioAxfOrderRequest;
import com.jiejing.fitness.finance.api.axf.request.StudioAxfApplyRequest;
import com.jiejing.fitness.finance.api.axf.vo.AggStudioAxfApplyVO;
import com.jiejing.fitness.finance.api.axf.vo.AggStudioAxfOrderVO;
import com.jiejing.fitness.finance.api.axf.vo.StudioAuthTokenVO;
import com.jiejing.fitness.finance.api.axf.vo.StudioAxfApplyVO;
import com.jiejing.fitness.finance.api.axf.vo.StudioAxfOrderDeductionVO;
import com.jiejing.fitness.finance.api.axf.vo.StudioAxfOrderRecoveryVO;
import com.jiejing.fitness.finance.api.axf.vo.StudioAxfOrderVO;
import com.jiejing.fitness.finance.repository.entity.PartyToMerchant;
import com.jiejing.fitness.finance.repository.entity.StudioAxfApply;
import com.jiejing.fitness.finance.repository.entity.StudioAxfAuth;
import com.jiejing.fitness.finance.repository.entity.StudioAxfAuthApply;
import com.jiejing.fitness.finance.repository.entity.StudioAxfCommodityTemplate;
import com.jiejing.fitness.finance.repository.entity.StudioAxfCommodityTemplateApply;
import com.jiejing.fitness.finance.repository.entity.StudioAxfOrder;
import com.jiejing.fitness.finance.repository.entity.StudioAxfOrderDeduction;
import com.jiejing.fitness.finance.repository.entity.StudioAxfOrderRecovery;
import com.jiejing.fitness.finance.repository.service.PartyToMerchantRpService;
import com.jiejing.fitness.finance.repository.service.StudioAxfApplyRpService;
import com.jiejing.fitness.finance.repository.service.StudioAxfAuthApplyRpService;
import com.jiejing.fitness.finance.repository.service.StudioAxfAuthRpService;
import com.jiejing.fitness.finance.repository.service.StudioAxfCommodityTemplateApplyRpService;
import com.jiejing.fitness.finance.repository.service.StudioAxfCommodityTemplateRpService;
import com.jiejing.fitness.finance.repository.service.StudioAxfOrderDeductionRpService;
import com.jiejing.fitness.finance.repository.service.StudioAxfOrderRecoveryRpService;
import com.jiejing.fitness.finance.repository.service.StudioAxfOrderRpService;
import com.jiejing.fitness.finance.service.axf.AxfService;
import com.jiejing.fitness.finance.service.axf.constans.AxfConst;
import com.jiejing.fitness.finance.service.axf.convert.AxfConvert;
import com.jiejing.fitness.finance.service.axf.model.AlipayCommerceMerchantcardDeductionOrderNotifyModel;
import com.jiejing.fitness.finance.service.enums.FinanceErrorEnums;
import com.jiejing.fitness.finance.service.rpc.MerchantRpcService;
import com.jiejing.fitness.finance.service.rpc.ResourceRpcService;
import com.jiejing.fitness.finance.service.rpc.StudioRpcService;
import com.jiejing.fitness.finance.service.utils.DingUtil;
import com.jiejing.fitness.finance.service.utils.MoneyUtil;
import com.jiejing.fitness.request.MemberIdRequest;
import com.jiejing.member.api.member.MemberApi;
import com.jiejing.member.api.member.request.ListStudioMemberInfoRequest;
import com.jiejing.member.api.member.vo.MemberDetailVO;
import com.jiejing.member.api.member.vo.MemberDigestVO;
import com.jiejing.member.api.member.vo.StudioMemberSimpleVO;
import com.jiejing.message.enums.MsgChannelEnum;
import com.jiejing.message.event.SendCommonMsgEvent;
import com.jiejing.paycenter.api.merchant.MerchantApi;
import com.jiejing.paycenter.api.merchant.MerchantQueryApi;
import com.jiejing.paycenter.api.merchant.StoreApi;
import com.jiejing.paycenter.api.merchant.StoreQueryApi;
import com.jiejing.paycenter.api.merchant.request.ApplyMerchantRequest;
import com.jiejing.paycenter.api.merchant.request.AuthTokenRequest;
import com.jiejing.paycenter.api.merchant.request.CancelCommodityOrderRequest;
import com.jiejing.paycenter.api.merchant.request.CreateCommodityRequest;
import com.jiejing.paycenter.api.merchant.request.CreateCommodityTemplateRequest;
import com.jiejing.paycenter.api.merchant.request.GetByMerchantIdRequest;
import com.jiejing.paycenter.api.merchant.request.GetCommodityByBizIdRequest;
import com.jiejing.paycenter.api.merchant.request.GetCommodityByThirdCommodityIdRequest;
import com.jiejing.paycenter.api.merchant.request.ListBankCardRequest;
import com.jiejing.paycenter.api.merchant.request.MerchantBindCardRequest;
import com.jiejing.paycenter.api.merchant.request.UploadRequest;
import com.jiejing.paycenter.common.enums.common.OpenStateEnums;
import com.jiejing.paycenter.common.enums.merchant.CardTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.CommodityOrderDeductionStateEnums;
import com.jiejing.paycenter.common.enums.merchant.CycleTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.GrantTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.ResourceTypeEnums;
import com.jiejing.paycenter.common.enums.merchant.SubChannelOpenTypeEnums;
import com.jiejing.paycenter.common.event.CommodityOrderEvent;
import com.jiejing.paycenter.common.event.MerchantEvent;
import com.jiejing.paycenter.common.event.OrderDeductionEvent;
import com.jiejing.paycenter.common.event.StoreEvent;
import com.jiejing.paycenter.common.model.request.BankCard;
import com.jiejing.paycenter.common.model.request.CycleCommodityAttr;
import com.jiejing.paycenter.common.model.request.CycleCommodityAttr.PeriodPrice;
import com.jiejing.paycenter.common.model.request.ResourceInfo;
import com.jiejing.paycenter.common.model.vo.ApplyMerchantResultVO;
import com.jiejing.paycenter.common.model.vo.AuthTokenVO;
import com.jiejing.paycenter.common.model.vo.BankCardVO;
import com.jiejing.paycenter.common.model.vo.BindCardVO;
import com.jiejing.paycenter.common.model.vo.CancelCommodityOrderVO;
import com.jiejing.paycenter.common.model.vo.CommodityVO;
import com.jiejing.paycenter.common.model.vo.CreateCommodityTemplateVO;
import com.jiejing.paycenter.common.model.vo.CreateCommodityVO;
import com.jiejing.paycenter.common.model.vo.MerchantVO;
import com.jiejing.paycenter.common.model.vo.OpenStoreVO;
import com.jiejing.paycenter.common.model.vo.StoreVO;
import com.jiejing.paycenter.common.model.vo.SubChannelVO;
import com.jiejing.paycenter.common.model.vo.UploadVO;
import com.jiejing.studio.api.admin.vo.AdminVO;
import com.jiejing.studio.api.admin.vo.BaseAdminVO;
import com.jiejing.studio.api.studio.vo.StudioVO;
import com.jiejing.trade.api.voucher.VoucherApi;
import com.jiejing.trade.api.voucher.vo.VoucherVO;
import com.jiejing.workflow.api.DelayQueueApi;
import com.jiejing.workflow.api.request.RemoveDelayTaskRequest;
import com.jiejing.workflow.api.request.SaveDelayTaskRequest;
import com.jiejing.workflow.common.enums.DelayTopicEnum;
import com.xiaomai.event.EventAgent;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

/**
 * @author chengyubing
 * @since 2024/11/4 16:31
 */
@Slf4j
@Service
public class AxfServiceImpl implements AxfService {

  @Resource
  private MerchantApi merchantApi;

  @Resource
  private StudioAxfApplyRpService studioAxfApplyRpService;

  @Resource
  private StudioAxfCommodityTemplateRpService studioAxfCommodityTemplateRpService;

  @Resource
  private StudioAxfCommodityTemplateApplyRpService studioAxfCommodityTemplateApplyRpService;

  @Resource(name = "financeThreadPool")
  private Executor executor;

  @Resource
  private StoreApi storeApi;

  @Resource
  private StoreQueryApi storeQueryApi;

  @Resource
  private MerchantQueryApi merchantQueryApi;

  @Resource
  private StudioAxfAuthRpService studioAxfAuthRpService;

  @Resource
  private StudioAxfAuthApplyRpService studioAxfAuthApplyRpService;

  @Resource
  private PartyToMerchantRpService partyToMerchantRpService;

  @Resource
  private StudioAxfOrderRpService studioAxfOrderRpService;

  @Resource
  private StudioAxfOrderDeductionRpService studioAxfOrderDeductionRpService;

  @Resource
  private StudioAxfOrderRecoveryRpService studioAxfOrderRecoveryRpService;

  @Resource
  private DelayQueueApi delayQueueApi;

  @Resource
  private StudioRpcService studioRpcService;

  @Resource
  private MemberApi memberApi;

  @Resource
  private MerchantRpcService merchantRpcService;

  @Value("${spring.profiles.active}")
  private String env;

  @Value("${message.axf.app}")
  private String pushUrl;

  @Override
  public StudioAxfApplyVO getMerchant(Long studioId) {
    PartyToMerchant merchant = partyToMerchantRpService.getByStudioId(studioId, AxfConst.CHANNEL_NO);
    if (null == merchant) {
      return null;
    }
    StudioAxfApply apply = studioAxfApplyRpService.getLatestOneByMerchantId(merchant.getMerchantId(),
        AxfStateEnums.SETTLE_SUCCESS);
    return this.toVO(apply);
  }

  @DS("finance")
  @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
  @Override
  public StudioAuthTokenVO authToken(Long studioId, GrantTypeEnums grantType, String code,
      String refreshToken) {
    AuthTokenRequest req = AuthTokenRequest.builder().grantType(grantType).code(code)
        .refreshToken(refreshToken).channelNo(AxfConst.CHANNEL_NO).build();
    JsonResult<AuthTokenVO> result = merchantApi.authToken(req);
    result.assertSuccess();

    // 存在，则表示曾经授权过，本次为重新授权
    StudioAxfAuthApply existApply = studioAxfAuthApplyRpService.getLatestByUserId(studioId,
        result.getResult().getUserId());

    StudioAxfAuthApply toSave = BeanUtil.map(result.getResult(), StudioAxfAuthApply.class);
    toSave.setId(IdWorker.getId());
    toSave.setStudioId(studioId);
    toSave.setCreateTime(new Date());
    toSave.setUpdateTime(new Date());
    toSave.setUsed(null == existApply ? ConfirmEnum.NO.name() : existApply.getUsed());
    studioAxfAuthApplyRpService.insert(toSave);
    StudioAxfAuth exist = studioAxfAuthRpService.getById(studioId).orElse(null);
    if (null != exist) {
      StudioAxfAuth auth = BeanUtil.map(result.getResult(), StudioAxfAuth.class);
      auth.setId(exist.getId());
      auth.setUpdateTime(new Date());
      auth.setUsed(toSave.getUsed());
      studioAxfAuthRpService.updateById(auth);
      return AxfConvert.toAuth(exist.getId(), result.getResult(), ConfirmEnum.valueOf(toSave.getUsed()));
    }
    StudioAxfAuth auth = BeanUtil.map(result.getResult(), StudioAxfAuth.class);
    auth.setId(studioId);
    auth.setStudioId(studioId);
    auth.setUsed(toSave.getUsed());
    auth.setCreateTime(new Date());
    auth.setUpdateTime(new Date());
    studioAxfAuthRpService.insert(auth);
    return AxfConvert.toAuth(auth.getId(), result.getResult(), ConfirmEnum.valueOf(toSave.getUsed()));
  }

  @Override
  public StudioAuthTokenVO getAuthToken(Long studioId) {
    StudioAxfAuth exist = studioAxfAuthRpService.getById(studioId).orElse(null);
    if (null != exist) {
      return AxfConvert.toAuth(exist);
    }
    return null;
  }

  @Override
  public List<String> listSettleAccount(Long studioId) {
    StudioAuthTokenVO auth = this.getAuthToken(studioId);
    ListBankCardRequest request = new ListBankCardRequest();
    request.setMerchantNo(auth.getUserId());
    request.setChannelNo(AxfConst.CHANNEL_NO);
    JsonResult<List<BankCardVO>> result = merchantApi.listBankCard(request);
    result.assertSuccess();
    return result.getResult().stream().map(BankCardVO::getCardNo).collect(Collectors.toList());
  }

  @Override
  public Long apply(StudioAxfApplyRequest params) {

    this.checkBeforeApply(params);
    StudioAxfApply apply = AxfConvert.toEntity(params, studioId -> {
      StudioVO studio = studioRpcService.getStudio(params.getStudioId());
      return null == studio ? "" : studio.getName();
    }, adminId -> {
      AdminVO admin = studioRpcService.getAdmin(params.getOperatorId());
      return null == admin ? "" : admin.getName();
    });

    studioAxfApplyRpService.insert(apply);

    executor.execute(() -> {
      try {
        Map<ResourceTypeEnums, ResourceInfo> resourceMap = this.upload(params, apply);

        ApplyMerchantRequest request = AxfConvert.toRequest(AxfConvert.decrypt(apply), resourceMap);
        JsonResult<ApplyMerchantResultVO> result = merchantApi.apply(request);
        result.assertSuccess();

      } catch (Exception e) {
        log.error("apply axf merchant fail {}, studio id = {}", apply.getId(), params.getStudioId(), e);
        String failMessage =
            e instanceof BizException ? ((BizException) e).getDefaultErrorMessage() : e.getMessage();
        this.doOpenAxfFail(apply.getId(), failMessage);
        return;
      }

      this.doOpenAxfProcess(apply.getId());

    });

    return apply.getId();
  }

  @Override
  public BindCardVO bindCard(Long id, String cardNo) {
    StudioAxfApply apply = studioAxfApplyRpService.getById(id)
        .orElseThrow(() -> new BizException(FinanceErrorEnums.NOT_EXIST));
    StudioAxfApply toModify = StudioAxfApply.builder().id(apply.getId())
        .cardNo(AesUtil.encrypt(apply.getSalt(), cardNo)).build();
    studioAxfApplyRpService.updateById(toModify);

    // 绑定结算账户
    BindCardVO vo = this.bindSettleAccount(id, apply.getMerchantId());
    StudioAxfApply newApply = studioAxfApplyRpService.getById(id)
        .orElseThrow(() -> new BizException(FinanceErrorEnums.NOT_EXIST));
    if (AxfStateEnums.SETTLE_SUCCESS.getCode().equals(newApply.getState())) {
      // 结算账户绑定完成，安心付可用
      this.openAxfAllSuccess(apply);
    } else if (AxfStateEnums.SETTLE_FAIL.getCode().equals(newApply.getState())) {
      executor.execute(() -> this.sendAxfFailMessage(id, replaceFailMessage(vo.getFailMessage())));
    }
    return vo;
  }

  private void openAxfAllSuccess(StudioAxfApply apply) {
    PartyToMerchant exist = partyToMerchantRpService.getByStudioId(apply.getStudioId(), AxfConst.CHANNEL_NO);
    if (null != exist) {
      exist.setMerchantId(apply.getMerchantId());
      exist.setMerchantNo(apply.getMerchantNo());
      partyToMerchantRpService.updateById(exist);
    } else {
      partyToMerchantRpService.insert(
          PartyToMerchant.builder().id(IdWorker.getId()).partyId(apply.getStudioId())
              .partyType(PartyTypeEnum.STUDIO.getCode()).merchantId(apply.getMerchantId())
              .merchantNo(apply.getMerchantNo()).channelNo(AxfConst.CHANNEL_NO).createTime(new Date())
              .updateTime(new Date()).build());
    }

    this.setAuthUsed(apply.getStudioId(), apply.getMerchantNo());

    // 开通标品
    executor.execute(() -> {
      JsonResult<StoreVO> result = storeQueryApi.getStoreByMerchantId(
          GetByMerchantIdRequest.builder().merchantId(apply.getMerchantId()).build());
      result.assertSuccess();
      StoreEvent event = StoreEvent.builder().merchantId(apply.getMerchantId()).id(result.getResult().getId())
          .build();
      this.createCommodityTemplate(apply, event, "立即扣款周期付周1", true, CycleTypeEnums.WEEK, 1);
      this.createCommodityTemplate(apply, event, "立即扣款周期付周3", true, CycleTypeEnums.WEEK, 3);
      this.createCommodityTemplate(apply, event, "立即扣款周期付月1", true, CycleTypeEnums.MONTH, 1);
      this.createCommodityTemplate(apply, event, "立即扣款周期付月15", true, CycleTypeEnums.MONTH, 15);
      this.createCommodityTemplate(apply, event, "立即扣款周期付月28", true, CycleTypeEnums.MONTH, 28);
      this.createCommodityTemplate(apply, event, "周期付周1", false, CycleTypeEnums.WEEK, 1);
      this.createCommodityTemplate(apply, event, "周期付周3", false, CycleTypeEnums.WEEK, 3);
      this.createCommodityTemplate(apply, event, "周期付月1", false, CycleTypeEnums.MONTH, 1);
      this.createCommodityTemplate(apply, event, "周期付月15", false, CycleTypeEnums.MONTH, 15);
      this.createCommodityTemplate(apply, event, "周期付月28", false, CycleTypeEnums.MONTH, 28);

      this.sendAxfSuccessMessage(apply);
    });
  }

  @Override
  public CancelCommodityOrderVO cancelOrder(Long id, Long opId) {
    StudioAxfOrder order = studioAxfOrderRpService.getById(id)
        .orElseThrow(() -> new BizException(FinanceErrorEnums.NOT_EXIST));
    StudioAxfOrderVO orderVO = AxfConvert.toVO(order);
    if (AxfOrderSignStateEnums.CANCEL.getCode().equals(order.getSignState())
        || AxfOrderSignStateEnums.CANCEL_PROCESS.getCode().equals(order.getSignState())) {
      throw new BizException("", "请勿重复解约");
    }
    if (AxfOrderSignStateEnums.SIGN.getCode().equals(order.getSignState())) {
      if (orderVO.getCommodity().getCycleCommodityAttr().getTotalPeriod().equals(order.getReceivedPeriod())) {
        // 全部扣款完成
        throw new BizException("", "已全部扣款完成，无需解约");
      }
    }
    List<StudioAxfOrderDeductionVO> deductions = this.listDeduction(id);
    if (deductions.stream().anyMatch(e -> Objects.equals(
        CommodityOrderDeductionStateEnums.FAIL.getCode(), e.getState()))) {
      throw new BizException("", "订购单存在扣款失败，请确保扣款成功后再发起解约");
    }

    AdminVO admin = studioRpcService.getAdmin(opId);
    CancelCommodityOrderVO vo;
    try {
      JsonResult<CancelCommodityOrderVO> result = storeApi.cancelCommodityOrder(
          CancelCommodityOrderRequest.builder().merchantId(order.getMerchantId())
              .thirdOrderNo(order.getThirdOrderNo()).build());
      result.assertSuccess();
      vo = result.getResult();
    } catch (Exception e) {
      log.error("cancel order fail {}", id, e);
      vo = CancelCommodityOrderVO.builder().state(OpenStateEnums.FAIL)
          .failMessage(e instanceof BizException ? ((BizException) e).getDefaultErrorMessage() : "网络异常")
          .build();
    }

    if (OpenStateEnums.FAIL == vo.getState()) {
      this.cancelFailNoCheck(id, opId, new Date());
      return vo;
    }

    StudioAxfOrder toModify = StudioAxfOrder.builder().id(id).cancelOpId(opId)
        .signState(AxfOrderSignStateEnums.CANCEL_PROCESS.getCode()).cancelOpName(admin.getName())
        .cancelFailMessage("").updateTime(new Date()).build();
    studioAxfOrderRpService.updateById(toModify);

    delayQueueApi.save(
        SaveDelayTaskRequest.builder().bizCode(DelayTaskCodeEnum.CANCEL_AXF_ORDER_FAIL.getCode()).bizId(id)
            .executionTime(DateUtils.addDays(toModify.getUpdateTime(), 1)).taskBody(
                new JSONObject().fluentPut("opId", opId).fluentPut("cancelTime", toModify.getUpdateTime()))
            .tenantId(order.getStudioId()).topic(DelayTopicEnum.COMMON_DELAY_TASK).build());
    return vo;
  }

  @Override
  public void callback(MerchantEvent event) {
    if (CollectionUtils.isEmpty(event.getSubChannels())) {
      return;
    }
    SubChannelVO subChannel = event.getSubChannels().stream()
        .filter(e -> SubChannelOpenTypeEnums.AXF_MERCHANT_PERIOD_PAY.getCode().equals(e.getOpenType()))
        .findFirst().orElse(null);
    if (null == subChannel) {
      return;
    }
    Long id = Long.parseLong(event.getApplyNo());
    switch (OpenStateEnums.getByCode(subChannel.getState())) {
      case FAIL:
        this.doOpenAxfFail(id, subChannel.getFailMessage());
        break;
      case SUCCESS:
        this.doOpenAxfSuccess(id, event.getMerchantId());
        break;
      default:
        break;
    }
  }

  @Override
  public void storeCallback(StoreEvent event) {

    OpenStateEnums state = OpenStateEnums.getByCode(event.getState());
    if (OpenStateEnums.SUCCESS != state && OpenStateEnums.FAIL != state) {
      return;
    }

    StudioAxfApply apply = studioAxfApplyRpService.getLatestOneByMerchantId(event.getMerchantId(),
        AxfStateEnums.STORE_PROCESS);
    if (null == apply) {
      return;
    }

    if (OpenStateEnums.SUCCESS == state) {
      this.doOpenAxfStoreSuccess(apply, event);
    } else {
      this.doOpenAxfStoreFail(apply.getId(), event.getFailMessage());
    }
  }

  @Override
  public void orderCallback(CommodityOrderEvent event) {
    switch (event.getState()) {
      case AVAILABLE:
        // 签约，生成订购单
        this.createOrderCallback(event);
        break;
      case CLOSED:
        // 解约，更新订购单状态
        this.cancelOrderCallback(event);
        break;
      default:
        break;
    }
  }

  @DS("finance")
  @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
  @Override
  public void orderDeductionCallback(OrderDeductionEvent event) {
    if (CommodityOrderDeductionStateEnums.INIT == event.getState()
        || CommodityOrderDeductionStateEnums.REFUND == event.getState()) {
      return;
    }
    AlipayCommerceMerchantcardDeductionOrderNotifyModel model = JSON.parseObject(event.getAttr(),
        AlipayCommerceMerchantcardDeductionOrderNotifyModel.class);
    StudioAxfOrderVO order = this.getByThirdOrderNo(event.getMerchantId(), event.getThirdOrderNo());
    List<StudioAxfOrderDeduction> deductions = studioAxfOrderDeductionRpService.listByOrderId(order.getId());
    StudioAxfOrderDeduction exist = deductions.stream()
        .filter(e -> e.getPeriod().equals(Integer.parseInt(model.getPeriod()))).findFirst().orElse(null);
    if (null == exist) {
      // 生成扣款记录
      StudioAxfOrderDeduction deduction = AxfConvert.toDeduction(event, order, model, 0);
      studioAxfOrderDeductionRpService.insert(deduction);
    } else if (!CommodityOrderDeductionStateEnums.SUCCESS.getCode().equals(exist.getState())) {
      // 生成补缴记录
      StudioAxfOrderRecovery recovery = AxfConvert.toRecovery(event, order, exist, model);
      studioAxfOrderRecoveryRpService.insert(recovery);
      if (CommodityOrderDeductionStateEnums.SUCCESS == event.getState()) {
        studioAxfOrderDeductionRpService.deleteById(exist.getId());
        StudioAxfOrderDeduction deduction = AxfConvert.toDeduction(event, order, model,
            exist.getRecoveryTimes() + 1);
        studioAxfOrderDeductionRpService.insert(deduction);
        List<StudioAxfOrderRecovery> recoveries = studioAxfOrderRecoveryRpService.listByDeductionId(
            exist.getId());
        studioAxfOrderRecoveryRpService.updateBatchById(recoveries.stream()
            .map(e -> StudioAxfOrderRecovery.builder().id(e.getId()).deductionId(deduction.getId()).build())
            .collect(Collectors.toSet()));
      } else {
        studioAxfOrderDeductionRpService.incRecoveryTimes(exist.getId());
      }
    }

    if (CommodityOrderDeductionStateEnums.SUCCESS == event.getState()) {
      this.syncOrder(order);
    }
  }

  @Override
  public StudioAxfApplyVO getById(Long id) {
    StudioAxfApply apply = studioAxfApplyRpService.getById(id).orElse(null);
    if (apply == null) {
      return null;
    }
    return toVO(apply);
  }

  @Override
  public PageVO<StudioAxfApplyVO> page(PageStudioAxfApplyRequest request) {
    org.springframework.data.domain.Page<StudioAxfApply> page = studioAxfApplyRpService.page(
        request.getStudioId(), request.getStudioName(),
        request.getShortName(), request.getMerchantNo(),
        request.getCreateTimeStart(), request.getCreateTimeEnd(),
        request.getStateList(), request.getCurrent(), request.getSize());
    return PageVO.convert(page, AxfConvert.toList(page.getContent()));
  }

  @Override
  public StudioAxfApplyVO getLatestApply(Long studioId, String merchantNo) {
    StudioAxfApply apply = studioAxfApplyRpService.getLatestOneByMerchantNo(studioId, merchantNo);
    if (null == apply) {
      return null;
    }
    return toVO(apply);
  }

  @Override
  public AggStudioAxfApplyVO agg() {
    Future<Integer> total = ((ThreadPoolTaskExecutor) executor).submit(
        () -> studioAxfApplyRpService.countDistinctStudioId());
    Future<Integer> success = ((ThreadPoolTaskExecutor) executor).submit(
        () -> (partyToMerchantRpService.countByChannelNo(AxfConst.CHANNEL_NO)));
    try {
      return AggStudioAxfApplyVO.builder().total(total.get(5, TimeUnit.SECONDS))
          .success(success.get(5, TimeUnit.SECONDS)).build();
    } catch (Exception e) {
      log.error("agg studio axf apply fail", e);
      throw new BizException(FinanceErrorEnums.AGG_TIME_OUT);
    }
  }

  @Override
  public PageVO<StudioAxfOrderVO> pageOrder(PageStudioAxfOrderRequest request) {
    Set<Long> studioIds = new HashSet<>(1);
    if (StringUtils.isNotEmpty(request.getStudioName())) {
      List<Long> queryStudioIds = studioRpcService.queryStudioIdsByNameLike(request.getStudioName());
      if (CollectionUtils.isEmpty(queryStudioIds)) {
        return PageVO.empty(request.getCurrent(), request.getCurrent());
      }
      studioIds.addAll(queryStudioIds);
    }
    if (null != request.getStudioId()) {
      if (CollectionUtils.isNotEmpty(studioIds)) {
        if (!studioIds.contains(request.getStudioId())) {
          return PageVO.empty(request.getCurrent(), request.getCurrent());
        }
      }
      studioIds.add(request.getStudioId());
    }

    Page<StudioAxfOrder> page = studioAxfOrderRpService.page(Lists.newArrayList(studioIds),
        request.getMemberId(), request.getMemberName(), request.getMerchantNo(),
        request.getVoucherBusinessNo(),
        request.getThirdOrderNo(), request.getSignTimeStart(), request.getSignTimeEnd(),
        request.getCancelTimeStart(), request.getCancelTimeEnd(), request.getState(), request.getCurrent(),
        request.getSize());

    if (CollectionUtils.isEmpty(page.getContent())) {
      return PageVO.empty(request.getCurrent(), request.getCurrent());
    }

    // 补充学员姓名+校区名称
    List<StudioAxfOrderVO> vos = toVOList(page.getContent());
    this.appendStudioAndMemberInfo(vos);
    return PageVO.convert(page, vos);
  }

  @Override
  public AggStudioAxfOrderVO aggOrder(PageStudioAxfOrderRequest request) {
    AggStudioAxfOrderVO emptyVO = AggStudioAxfOrderVO.builder().receivedActualAmount(BigDecimal.ZERO)
        .receivedAmount(BigDecimal.ZERO)
        .originalPrice(BigDecimal.ZERO)
        .salePrice(BigDecimal.ZERO)
        .build();
    Set<Long> studioIds = new HashSet<>(1);
    if (StringUtils.isNotEmpty(request.getStudioName())) {
      List<Long> queryStudioIds = studioRpcService.queryStudioIdsByNameLike(request.getStudioName());
      if (CollectionUtils.isEmpty(queryStudioIds)) {
        return emptyVO;
      }
      studioIds.addAll(queryStudioIds);
    }
    if (null != request.getStudioId()) {
      if (CollectionUtils.isNotEmpty(studioIds)) {
        if (!studioIds.contains(request.getStudioId())) {
          return emptyVO;
        }
      }
      studioIds.add(request.getStudioId());
    }
    AggStudioAxfOrderVO vo = studioAxfOrderRpService.agg(
        Lists.newArrayList(studioIds),
        request.getMemberId(), request.getMemberName(), request.getVoucherBusinessNo(),
        request.getThirdOrderNo(), request.getSignTimeStart(), request.getSignTimeEnd(),
        request.getCancelTimeStart(), request.getCancelTimeEnd(), request.getState());
    vo.setRate(AxfConst.RATE);
    return vo;
  }

  private void appendStudioAndMemberInfo(List<StudioAxfOrderVO> vos) {
    Set<Long> studioIds = vos.stream().map(StudioAxfOrderVO::getStudioId).collect(Collectors.toSet());
    Map<Long, StudioVO> studioMap = studioRpcService.mapStudio(Lists.newArrayList(studioIds));
    List<Long> memberIds = vos.stream().map(StudioAxfOrderVO::getMemberId).distinct().collect(Collectors.toList());
    Map<Long, MemberDigestVO> memberDigestVOMap = this.getMemberList(memberIds).stream()
        .collect(Collectors.toMap(MemberDigestVO::getId, Function.identity(), (k1, k2) -> k1));
    vos.forEach(e -> {
          e.setStudioName(studioMap.getOrDefault(e.getStudioId(), new StudioVO()).getName());
          e.setMemberName(memberDigestVOMap.getOrDefault(e.getMemberId(), new MemberDigestVO()).getName());
          e.setMemberPhone(memberDigestVOMap.getOrDefault(e.getMemberId(), new MemberDigestVO()).getPhone());
          e.setMemberStatus(memberDigestVOMap.getOrDefault(e.getMemberId(), new MemberDigestVO()).getStatus());
          e.setMemberGender(memberDigestVOMap.getOrDefault(e.getMemberId(), new MemberDigestVO()).getGender());
          e.setMemberAvatar(memberDigestVOMap.getOrDefault(e.getMemberId(), new MemberDigestVO()).getAvatar());
        });
  }

  private List<StudioAxfOrderVO> toVOList(List<StudioAxfOrder> orders) {
    return orders.stream().map(e -> BeanUtil.map(e, StudioAxfOrderVO.class)).collect(Collectors.toList());
  }

  @Override
  public StudioAxfOrderVO getOrder(Long id) {
    StudioAxfOrder order = studioAxfOrderRpService.getById(id).orElse(null);
    if (null == order) {
      return null;
    }
    StudioAxfOrderVO studioAxfOrderVO = AxfConvert.toVO(order);
    appendStudioAndMemberInfo(Lists.newArrayList(studioAxfOrderVO));
    return studioAxfOrderVO;
  }

  private void doOpenAxfStoreProcess(Long id) {
    studioAxfApplyRpService.updateById(
        StudioAxfApply.builder().id(id).state(AxfStateEnums.STORE_PROCESS.getCode()).updateTime(new Date())
            .build());
  }

  private void doOpenAxfStoreFail(Long id, String failMessage) {
    studioAxfApplyRpService.updateById(
        StudioAxfApply.builder().id(id).state(AxfStateEnums.STORE_FAIL.getCode())
            .failMessage(replaceFailMessage(failMessage)).updateTime(new Date()).build());

    executor.execute(() -> this.sendAxfFailMessage(id, replaceFailMessage(failMessage)));
  }

  private void doOpenAxfSuccess(Long id, Long merchantId) {
    studioAxfApplyRpService.updateById(
        StudioAxfApply.builder().id(id).state(AxfStateEnums.AXF_SUCCESS.getCode()).merchantId(merchantId)
            .updateTime(new Date()).build());

    // 开通门店
    this.openStore(id, merchantId, SubChannelOpenTypeEnums.AXF_MERCHANT_PERIOD_PAY);

  }

  private BindCardVO bindSettleAccount(Long id, Long merchantId) {
    StudioAxfApply apply = studioAxfApplyRpService.getById(id)
        .orElseThrow(() -> new BizException(FinanceErrorEnums.NOT_EXIST));
    String cardNo = AxfConvert.decrypt(apply).getCardNo();
    MerchantBindCardRequest request = new MerchantBindCardRequest();
    request.setApplyNo(IdWorker.getIdStr());
    request.setMerchantId(merchantId);
    request.setChannelNo(AxfConst.CHANNEL_NO);
    request.setBizCode(AxfConst.BIZ_CODE);
    request.setBankCard(BankCard.builder().cardNo(cardNo).cardType(CardTypeEnums.ALI_SETTLE_ACCOUNT).build());
    BindCardVO vo;
    try {
      JsonResult<BindCardVO> result = merchantApi.bindCard(request);
      result.assertSuccess();
      vo = result.getResult();
    } catch (Exception e) {
      log.error("bind settle account fail {}", merchantId, e);
      vo = BindCardVO.builder().state(OpenStateEnums.FAIL)
          .failMessage(e instanceof BizException ? ((BizException) e).getDefaultErrorMessage() : "系统异常")
          .build();
    }

    apply.setState(vo.getState() == OpenStateEnums.SUCCESS ? AxfStateEnums.SETTLE_SUCCESS.getCode()
        : AxfStateEnums.SETTLE_FAIL.getCode());
    apply.setFailMessage(replaceFailMessage(vo.getFailMessage()));
    studioAxfApplyRpService.updateById(apply);

    return vo;
  }

  private void openStore(Long id, Long merchantId, SubChannelOpenTypeEnums openType) {
    OpenStoreVO vo;
    try {
      JsonResult<OpenStoreVO> result = storeApi.openStore(AxfConvert.toStoreRequest(merchantId, openType));
      result.assertSuccess();
      vo = result.getResult();
    } catch (Exception e) {
      log.error("open store fail {}, {}", id, merchantId, e);
      String failMessage =
          e instanceof BizException ? ((BizException) e).getDefaultErrorMessage() : "网络异常";
      vo = OpenStoreVO.builder().state(OpenStateEnums.FAIL).failMessage(failMessage).build();
    }
    switch (vo.getState()) {
      case INIT:
      case PROCESS:
        this.doOpenAxfStoreProcess(id);
        break;
      case FAIL:
        this.doOpenAxfStoreFail(id, vo.getFailMessage());
        break;
      case SUCCESS:
        this.doOpenAxfStoreSuccess(studioAxfApplyRpService.getById(id)
                .orElseThrow(() -> new BizException(FinanceErrorEnums.NOT_EXIST)),
            StoreEvent.builder().id(vo.getStoreId()).merchantId(merchantId).build());
        break;
      default:
        break;
    }
  }

  public void createCommodityTemplateTest(Long applyId, Long storeId) {
    StudioAxfApply apply = studioAxfApplyRpService.getById(applyId)
        .orElseThrow(() -> new BizException(FinanceErrorEnums.NOT_EXIST));
    StoreEvent event = StoreEvent.builder().id(storeId).merchantId(apply.getMerchantId()).build();
    this.createCommodityTemplate(apply, event, "立即扣款周期付周1", true, CycleTypeEnums.WEEK, 1);
    this.createCommodityTemplate(apply, event, "立即扣款周期付周3", true, CycleTypeEnums.WEEK, 3);
    this.createCommodityTemplate(apply, event, "立即扣款周期付月1", true, CycleTypeEnums.MONTH, 1);
    this.createCommodityTemplate(apply, event, "立即扣款周期付月15", true, CycleTypeEnums.MONTH, 15);
    this.createCommodityTemplate(apply, event, "立即扣款周期付月28", true, CycleTypeEnums.MONTH, 28);
    this.createCommodityTemplate(apply, event, "周期付周1", false, CycleTypeEnums.WEEK, 1);
    this.createCommodityTemplate(apply, event, "周期付周3", false, CycleTypeEnums.WEEK, 3);
    this.createCommodityTemplate(apply, event, "周期付月1", false, CycleTypeEnums.MONTH, 1);
    this.createCommodityTemplate(apply, event, "周期付月15", false, CycleTypeEnums.MONTH, 15);
    this.createCommodityTemplate(apply, event, "周期付月28", false, CycleTypeEnums.MONTH, 28);
  }

  private void doOpenAxfStoreSuccess(StudioAxfApply apply, StoreEvent event) {
    studioAxfApplyRpService.updateById(
        StudioAxfApply.builder().id(apply.getId()).state(AxfStateEnums.STORE_SUCCESS.getCode())
            .shopId(event.getShopId())
            .updateTime(new Date()).build());
  }

  private void createCommodityTemplate(StudioAxfApply apply, StoreEvent event, String commodityName,
      boolean chargeNow, CycleTypeEnums cycleType, Integer cycleValue) {
    CreateCommodityTemplateRequest request = AxfConvert.toCommodityRequest(apply, event, commodityName,
        chargeNow, cycleType, cycleValue);
    JsonResult<CreateCommodityTemplateVO> result = storeApi.createCommodityTemplate(request);
    StudioAxfCommodityTemplateApply templateApply = AxfConvert.toTemplateApply(apply, result, request);
    studioAxfCommodityTemplateApplyRpService.insert(templateApply);
    if (OpenStateEnums.SUCCESS == OpenStateEnums.getByCode(templateApply.getState())) {
      StudioAxfCommodityTemplate exist = this.hitTemplate(templateApply.getMerchantId(),
          request.getCycleCommodityAttr());
      if (null == exist) {
        StudioAxfCommodityTemplate template = AxfConvert.toTemplate(apply, result, request);
        studioAxfCommodityTemplateRpService.insert(template);
      } else {
        StudioAxfCommodityTemplate template = AxfConvert.toTemplate(apply, result, request);
        template.setId(exist.getId());
        studioAxfCommodityTemplateRpService.updateById(template);
      }
    }
    if (OpenStateEnums.FAIL == OpenStateEnums.getByCode(templateApply.getState())) {
      // 钉钉告警
      DingUtil.sendCreateCommodityTemplateFail(env, apply.getStudioId(), templateApply.getId());
    }
  }

  @Override
  public CreateCommodityVO createCommodity(CreateStudioAxfCommodityRequest request) {

    // 校验
    this.check(request);

    StudioAxfCommodityTemplate template = this.hitTemplate(request);
    if (null == template) {
      return CreateCommodityVO.builder().state(OpenStateEnums.FAIL).failMessage("未找到相应标品").build();
    }

    return this.createCommodity(AxfConvert.toRequest(request, template));
  }

  private void check(CreateStudioAxfCommodityRequest request) {
    BigDecimal sumOriginalPrice = request.getCycleCommodityAttr().getPeriodPriceList().stream()
        .map(PeriodPrice::getOriginalPrice).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
    BigDecimal sumSalePrice = request.getCycleCommodityAttr().getPeriodPriceList().stream()
        .map(PeriodPrice::getSalePrice).reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
    if (sumSalePrice.compareTo(request.getSalePrice()) != 0) {
      throw new BizException(FinanceErrorEnums.AXF_PERIOD_SALE_PRICE_ERROR);
    }
    if (sumOriginalPrice.compareTo(request.getOriginalPrice()) != 0) {
      throw new BizException(FinanceErrorEnums.AXF_PERIOD_SALE_PRICE_ERROR);
    }
  }

  @Override
  public CommodityVO getCommodity(String thirdCommodityId) {
    JsonResult<CommodityVO> result = storeQueryApi.getCommodityByThirdCommodityId(
        GetCommodityByThirdCommodityIdRequest.builder().channelNo(AxfConst.CHANNEL_NO)
            .thirdCommodityId(thirdCommodityId).build());
    result.assertSuccess();
    return result.getResult();
  }

  private CreateCommodityVO createCommodity(CreateCommodityRequest req) {
    try {
      JsonResult<CreateCommodityVO> result = storeApi.createCommodity(req);
      result.assertSuccess();
      return result.getResult();
    } catch (Exception e) {
      log.error("create commodity fail {}, {}", req.getMerchantId(), req.getTemplateId(), e);
      return CreateCommodityVO.builder().state(OpenStateEnums.FAIL).failMessage("网络异常").build();
    }
  }

  @Override
  public List<StudioAxfOrderDeductionVO> listDeduction(Long orderId) {
    List<StudioAxfOrderDeduction> deductions = studioAxfOrderDeductionRpService.listByOrderId(orderId);
    return AxfConvert.toDeductionVO(deductions);
  }

  @Override
  public ConfirmEnum existFailDeduction(Long orderId) {
    List<StudioAxfOrderDeduction> deductions = studioAxfOrderDeductionRpService.listByOrderId(orderId);
    return Optional.ofNullable(deductions).orElse(new ArrayList<>(1)).stream()
        .anyMatch(e -> CommodityOrderDeductionStateEnums.FAIL.getCode().equals(e.getState()))
        ? ConfirmEnum.YES
        : ConfirmEnum.NO;
  }

  @Override
  public List<StudioAxfOrderRecoveryVO> listRecovery(Long deductionId) {
    List<StudioAxfOrderRecovery> recoveries = studioAxfOrderRecoveryRpService.listByDeductionId(deductionId);
    return AxfConvert.toRecoveryVO(recoveries);
  }

  private StudioAxfCommodityTemplate hitTemplate(CreateStudioAxfCommodityRequest request) {
    PartyToMerchant merchant = partyToMerchantRpService.getByStudioId(request.getStudioId(),
        AxfConst.CHANNEL_NO);
    if (null == merchant) {
      throw new BizException(FinanceErrorEnums.NOT_OPEN_AXF);
    }
    return hitTemplate(merchant.getMerchantId(), request.getCycleCommodityAttr());
  }

  private StudioAxfCommodityTemplate hitTemplate(Long merchantId, CycleCommodityAttr attr) {
    List<StudioAxfCommodityTemplate> list = studioAxfCommodityTemplateRpService.listSuccessAll(merchantId);
    return Optional.ofNullable(list).orElse(new ArrayList<>(1)).stream()
        .filter(e -> e.getChargeNow().equals(attr.getChargeNow()))
        .filter(e -> attr.getCycleType().getCode().equals(e.getCycleType()))
        .filter(e -> attr.getCycleValue().equals(e.getCycleValue()))
        .filter(e -> attr.getCycleChargeType().getCode().equals(e.getCycleChargeType())).findFirst()
        .orElse(null);
  }

  private void doOpenAxfProcess(Long id) {
    studioAxfApplyRpService.updateById(
        StudioAxfApply.builder().id(id).state(AxfStateEnums.AXF_PROCESS.getCode()).updateTime(new Date())
            .build());
  }

  private static final List<Pair<String, String>> AXF_FAIL_REPLACE = new ArrayList<>(1);

  static {
    AXF_FAIL_REPLACE.add(Pair.of("商户名称与传参不匹配",
        "F_90001|TIHUAN|授权支付宝商户名称与上传资料的营业执照主体名称不一致"));
    AXF_FAIL_REPLACE.add(Pair.of("该账号不是个体户或者企业支付宝账号，请升级后再开通",
        "F_90001|TIHUAN|授权支付宝商户名称与上传资料的营业执照主体名称不一致"));
    AXF_FAIL_REPLACE.add(Pair.of("该商户存在特殊资质但未上传特殊资质图片",
        "F_90002|TIHUAN|商户营业执照对应的经营类目不在支付宝准许开通安心付的范围内，请通过下方办法修改经营类目后重新开通使用。"));
    AXF_FAIL_REPLACE.add(Pair.of("该账号未上传营业执照，请上传后再开通。",
        "F_90002|TIHUAN|商户营业执照对应的经营类目不在支付宝准许开通安心付的范围内，请通过下方办法修改经营类目后重新开通使用。"));
    AXF_FAIL_REPLACE.add(
        Pair.of("当前经营类目不支持开通此服务，请重新选择。;mcc不在准入白名单内;该经营类目不允许开周期卡;",
            "F_90003|TIHUAN|商户营业执照对应的经营类目不在支付宝准许开通安心付的范围内，请通过下方办法修改经营类目后重新开通使用。"));
    AXF_FAIL_REPLACE.add(
        Pair.of("当前经营类目不支持开通此服务，请重新选择。;该经营类目不允许开周期卡;mcc不在准入白名单内;",
            "F_90003|TIHUAN|商户营业执照对应的经营类目不在支付宝准许开通安心付的范围内，请通过下方办法修改经营类目后重新开通使用。"));
    AXF_FAIL_REPLACE.add(
        Pair.of("该经营类目不允许开周期卡;当前经营类目不支持开通此服务，请重新选择。;mcc不在准入白名单内;",
            "F_90003|TIHUAN|商户营业执照对应的经营类目不在支付宝准许开通安心付的范围内，请通过下方办法修改经营类目后重新开通使用。"));
    AXF_FAIL_REPLACE.add(
        Pair.of("该经营类目不允许开周期卡;mcc不在准入白名单内;当前经营类目不支持开通此服务，请重新选择。;",
            "F_90003|TIHUAN|商户营业执照对应的经营类目不在支付宝准许开通安心付的范围内，请通过下方办法修改经营类目后重新开通使用。"));
    AXF_FAIL_REPLACE.add(
        Pair.of("mcc不在准入白名单内;当前经营类目不支持开通此服务，请重新选择。;该经营类目不允许开周期卡;",
            "F_90003|TIHUAN|商户营业执照对应的经营类目不在支付宝准许开通安心付的范围内，请通过下方办法修改经营类目后重新开通使用。"));
    AXF_FAIL_REPLACE.add(
        Pair.of("mcc不在准入白名单内;该经营类目不允许开周期卡;当前经营类目不支持开通此服务，请重新选择。;",
            "F_90003|TIHUAN|商户营业执照对应的经营类目不在支付宝准许开通安心付的范围内，请通过下方办法修改经营类目后重新开通使用。"));
    AXF_FAIL_REPLACE.add(
        Pair.of("门店地址信息与高德等地图反馈的地址信息（含经纬度）偏差较大，建议填写精确到xxx路xx号的详细地址",
            "F_90004|TIHUAN|地址识别有误，请前往支付宝修改蚂蚁门店地址，门店审核成功后，功能即可开通成功。"));
    AXF_FAIL_REPLACE.add(Pair.of("商户未授权当前接口",
        "F_90005|TIHUAN|在支付宝授权时未勾选全部权限请重新授权，勾选“全权委托开发授权”"));
  }

  private void doOpenAxfFail(Long id, String failMessage) {
    studioAxfApplyRpService.updateById(StudioAxfApply.builder().id(id).state(AxfStateEnums.AXF_FAIL.getCode())
        .failMessage(replaceFailMessage(failMessage)).updateTime(new Date()).build());

    executor.execute(() -> this.sendAxfFailMessage(id, replaceFailMessage(failMessage)));
  }

  public static String replaceFailMessage(String failMessage) {
    if (null == failMessage) {
      return null;
    }
    return AXF_FAIL_REPLACE.stream().filter(e -> failMessage.contains(e.getLeft())).findFirst()
        .map(Pair::getRight).orElse(failMessage);
  }

  public static void main(String[] args) {
    System.out.println(replaceFailMessage("商户名称与传参不匹配"));
    System.out.println(replaceFailMessage(null));
  }

  private void sendAxfSuccessMessage(StudioAxfApply apply) {
    AdminVO admin = studioRpcService.getAdmin(apply.getOpId());
    BaseAdminVO parentAdmin = studioRpcService.getBossAdmin(apply.getStudioId());

    EventAgent.of(SendCommonMsgEvent.class).triggerEvent(
        SendCommonMsgEvent.builder().sourceId(apply.getStudioId()).eventId(IdWorker.getId())
            .covertTarget(false).channelEnums(Lists.newArrayList(MsgChannelEnum.SMS)).bizType("AXF_SUCCESS")
            .params(buildMessageMap(admin, MsgChannelEnum.SMS)).build());

    EventAgent.of(SendCommonMsgEvent.class).triggerEvent(
        SendCommonMsgEvent.builder().sourceId(apply.getStudioId()).eventId(IdWorker.getId())
            .covertTarget(true).channelEnums(Lists.newArrayList(MsgChannelEnum.APP_PUSH))
            .bizType("AXF_SUCCESS_SYSTEM").params(buildMessageMap(admin, MsgChannelEnum.APP_PUSH)).build());

    if (!admin.getId().equals(parentAdmin.getId())) {
      EventAgent.of(SendCommonMsgEvent.class).triggerEvent(
          SendCommonMsgEvent.builder().sourceId(apply.getStudioId()).eventId(IdWorker.getId())
              .covertTarget(false).channelEnums(Lists.newArrayList(MsgChannelEnum.SMS)).bizType("AXF_SUCCESS")
              .params(buildMessageMap(BeanUtil.map(parentAdmin, AdminVO.class), MsgChannelEnum.SMS)).build());

      EventAgent.of(SendCommonMsgEvent.class).triggerEvent(
          SendCommonMsgEvent.builder().sourceId(apply.getStudioId()).eventId(IdWorker.getId())
              .covertTarget(true).channelEnums(Lists.newArrayList(MsgChannelEnum.APP_PUSH))
              .bizType("AXF_SUCCESS_SYSTEM")
              .params(buildMessageMap(BeanUtil.map(parentAdmin, AdminVO.class), MsgChannelEnum.APP_PUSH))
              .build());
    }

  }

  private final static String SYMBOL = "|TIHUAN|";

  private String resetFailMessage(String failMessage) {
    if (null != failMessage && failMessage.contains(SYMBOL)) {
      String[] strings = failMessage.split(SYMBOL);
      if (strings.length > 1) {
        failMessage = strings[1];
      }
    }
    return failMessage;
  }

  private void sendAxfFailMessage(Long id, String failMessage) {
    failMessage = this.resetFailMessage(failMessage);
    StudioAxfApply apply = studioAxfApplyRpService.getById(id).orElse(null);

    AdminVO admin = studioRpcService.getAdmin(apply.getOpId());
    BaseAdminVO parentAdmin = studioRpcService.getBossAdmin(apply.getStudioId());

    String finalFailMessage = failMessage;
    EventAgent.of(SendCommonMsgEvent.class).triggerEvent(
        SendCommonMsgEvent.builder().sourceId(apply.getStudioId()).eventId(IdWorker.getId())
            .covertTarget(false).channelEnums(Lists.newArrayList(MsgChannelEnum.SMS)).bizType("AXF_FAIL")
            .params(buildMessageMap(admin, MsgChannelEnum.SMS).stream()
                .peek(e -> e.put("failMessage", finalFailMessage)).collect(Collectors.toList())).build());

    EventAgent.of(SendCommonMsgEvent.class).triggerEvent(
        SendCommonMsgEvent.builder().sourceId(apply.getStudioId()).eventId(IdWorker.getId())
            .covertTarget(true).channelEnums(Lists.newArrayList(MsgChannelEnum.APP_PUSH))
            .bizType("AXF_FAIL_SYSTEM").params(buildMessageMap(admin, MsgChannelEnum.APP_PUSH)).build());

    if (!admin.getId().equals(parentAdmin.getId())) {
      EventAgent.of(SendCommonMsgEvent.class).triggerEvent(
          SendCommonMsgEvent.builder().sourceId(apply.getStudioId()).eventId(IdWorker.getId())
              .covertTarget(false).channelEnums(Lists.newArrayList(MsgChannelEnum.SMS)).bizType("AXF_FAIL")
              .params(buildMessageMap(BeanUtil.map(parentAdmin, AdminVO.class), MsgChannelEnum.SMS).stream()
                  .peek(e -> e.put("failMessage", finalFailMessage)).collect(Collectors.toList())).build());

      EventAgent.of(SendCommonMsgEvent.class).triggerEvent(
          SendCommonMsgEvent.builder().sourceId(apply.getStudioId()).eventId(IdWorker.getId())
              .covertTarget(true).channelEnums(Lists.newArrayList(MsgChannelEnum.APP_PUSH))
              .bizType("AXF_FAIL_SYSTEM")
              .params(buildMessageMap(BeanUtil.map(parentAdmin, AdminVO.class), MsgChannelEnum.APP_PUSH))
              .build());
    }


  }

  private List<Map<String, Object>> buildMessageMap(AdminVO admin, MsgChannelEnum channel) {
    List<Map<String, Object>> param = Lists.newArrayList();
    Map<String, Object> map = Maps.newHashMap();
    switch (channel) {
      case SMS:
        map.put("targetId", admin.getPhone());
        break;
      case APP_PUSH:
        map.put("targetId", admin.getId());
        map.put("appUrl", pushUrl);
        break;
      default:
        break;
    }
    param.add(map);
    return param;
  }

  @Resource
  private ResourceRpcService resourceRpcService;

  private Map<ResourceTypeEnums, ResourceInfo> upload(StudioAxfApplyRequest request, StudioAxfApply apply) {
    return AxfConvert.toResourceMap(request, resourceIds -> {
      Map<Long, com.jiejing.fos.api.vo.ResourceInfoVO> urlMap = resourceRpcService.getResourceMap(
          request.getBrandId(), resourceIds);
      Map<Long, String> thirdMap = new HashMap<>(2);
      urlMap.keySet().forEach(resourceId -> {
        com.jiejing.fos.api.vo.ResourceInfoVO info = urlMap.get(resourceId);
        JsonResult<UploadVO> result = merchantApi.upload(
            UploadRequest.builder().channelNo(AxfConst.CHANNEL_NO).url(info.getUrl())
                .resourceType(ResourceTypeEnums.LICENSE).suffix(info.getSuffix())
                .fileSize(info.getFileSize()).build());
        result.assertSuccess();
        thirdMap.put(resourceId, result.getResult().getThirdId());
      });
      return thirdMap;
    });
  }

  private void checkBeforeApply(StudioAxfApplyRequest request) {
    Integer count = studioAxfApplyRpService.countProcessByStudioId(request.getStudioId());
    if (count.compareTo(0) > 0) {
      throw new BizException(FinanceErrorEnums.EXIST_PROCESS_APPLY);
    }
  }


  private void cancelOrderCallback(CommodityOrderEvent event) {
    Long voucherId = Long.parseLong(event.getOrderNo());
    StudioAxfOrder order = studioAxfOrderRpService.getByVoucherId(voucherId);
    if (null == order) {
      return;
    }

    order.setSignState(AxfOrderSignStateEnums.CANCEL.getCode());
    order.setCancelFailMessage("");
    order.setCancelTime(new Date());
    order.setUpdateTime(new Date());
    studioAxfOrderRpService.updateById(order);

    delayQueueApi.remove(RemoveDelayTaskRequest.builder().bizId(order.getId()).build());
  }

  @Resource
  private VoucherApi voucherApi;

  private void createOrderCallback(CommodityOrderEvent event) {
    Long voucherId = Long.parseLong(event.getOrderNo());
    StudioAxfOrder exist = studioAxfOrderRpService.getByVoucherId(voucherId);
    if (null != exist) {
      return;
    }

    GetCommodityByBizIdRequest request = GetCommodityByBizIdRequest.builder()
        .merchantId(event.getMerchantId()).bizId(voucherId).build();
    JsonResult<CommodityVO> result = storeQueryApi.getCommodityByBizId(request);
    result.assertSuccess();
    CommodityVO commodity = result.getResult();
    VoucherVO voucher = this.getVoucher(commodity.getBizId());
    MemberDetailVO member = this.getMemberDetail(voucher.getReceiver().getId());

    String merchantNo = null;
    if (event.getMerchantId()!=null){
      MerchantVO merchantVO = merchantRpcService.getByMerchantId(event.getMerchantId());
      if (merchantVO!=null){
        merchantNo = merchantVO.getMerchantNo();
      }
    }

    StudioAxfOrder order = AxfConvert.toEntity(event, merchantNo, commodity, voucher, member);
    studioAxfOrderRpService.insert(order);
  }

  private MemberDetailVO getMemberDetail(Long memberId) {
    MemberIdRequest request = new MemberIdRequest();
    request.setMemberId(memberId);
    JsonResult<MemberDetailVO> result = memberApi.findDetailById(request);
    return result.assertAndReturn().getResult();
  }

  private List<MemberDigestVO> getMemberList(List<Long> memberIds) {
    IdsRequest request = new IdsRequest();
    request.setIds(memberIds);
    JsonResult<List<MemberDigestVO>> result = memberApi.listDigestByIds(request);
    return result.assertAndReturn().getResult();
  }

  private VoucherVO getVoucher(Long voucherId) {
    JsonResult<VoucherVO> result = voucherApi.getVoucher(IdRequest.builder().id(voucherId).build());
    return result.assertAndReturn().getResult();
  }

  private StudioAxfApplyVO toVO(StudioAxfApply apply) {
    return AxfConvert.toVO(apply, vo -> {
      Map<Long, com.jiejing.fos.api.vo.ResourceInfoVO> map = resourceRpcService.getResourceMap(apply.getBrandId(),
          Lists.newArrayList(vo.getLicenseResourceId(), vo.getStoreHeadResourceId()));
      vo.setLicenseResourceUrl(map.get(vo.getLicenseResourceId()).getUrl());
      vo.setStoreHeadResourceUrl(map.get(vo.getStoreHeadResourceId()).getUrl());
    });
  }

  private void syncOrder(StudioAxfOrderVO order) {
    List<StudioAxfOrderDeduction> deductions = studioAxfOrderDeductionRpService.listByOrderId(order.getId())
        .stream().filter(e -> CommodityOrderDeductionStateEnums.SUCCESS.getCode().equals(e.getState()))
        .collect(Collectors.toList());
    BigDecimal amount = deductions.stream().map(StudioAxfOrderDeduction::getAmount).reduce(BigDecimal::add)
        .orElse(BigDecimal.ZERO);
    BigDecimal actualAmount = deductions.stream().map(StudioAxfOrderDeduction::getActualAmount)
        .reduce(BigDecimal::add).orElse(BigDecimal.ZERO);
    Integer receivedPeriod = deductions.stream().map(StudioAxfOrderDeduction::getPeriod).max(Integer::compare)
        .orElse(0);

    StudioAxfOrder toModify = StudioAxfOrder.builder().id(order.getId()).rate(AxfConst.RATE)
        .receivedAmount(amount).receivedActualAmount(actualAmount)
        .receivedFee(MoneyUtil.subtract(amount, actualAmount)).receivedPeriod(receivedPeriod)
        .updateTime(new Date()).build();
    studioAxfOrderRpService.updateById(toModify);
  }

  private StudioAxfOrderVO getByThirdOrderNo(Long merchantId, String thirdOrderNo) {
    StudioAxfOrder order = studioAxfOrderRpService.getByThirdOrderNo(merchantId, thirdOrderNo);
    return AxfConvert.toVO(order);
  }

  @Override
  public void cancelFail(Long id, Long opId, Date date) {
    StudioAxfOrder order = studioAxfOrderRpService.getById(id)
        .orElseThrow(() -> new BizException(FinanceErrorEnums.NOT_EXIST));
    if (!AxfOrderSignStateEnums.CANCEL_PROCESS.getCode().equals(order.getSignState())) {
      return;
    }
    this.cancelFailNoCheck(id, opId, date);
  }

  @Override
  public StudioAxfOrderVO getByVoucherId(Long voucherId) {
    StudioAxfOrder order = studioAxfOrderRpService.getByVoucherId(voucherId);
    if (order == null) {
      return null;
    }
    return AxfConvert.toVO(order);
  }

  private void cancelFailNoCheck(Long id, Long opId, Date date) {
    AdminVO admin = studioRpcService.getAdmin(opId);
    studioAxfOrderRpService.updateById(
        StudioAxfOrder.builder().id(id).signState(AxfOrderSignStateEnums.SIGN.getCode()).cancelOpId(-1L)
            .cancelOpName("").cancelFailMessage(
                DateFormatUtils.format(date, "yyyy-MM-dd HH:mm:ss") + " " + admin.getName()
                    + "发起解约，解约失败，如需解约请重新操作").build());
  }

  private void setAuthUsed(Long studioId, String merchantNo) {
    StudioAxfAuthApply apply = studioAxfAuthApplyRpService.getLatestByUserId(studioId, merchantNo);
    if (null != apply) {
      apply.setUsed(ConfirmEnum.YES.name());
      apply.setUpdateTime(new Date());
      studioAxfAuthApplyRpService.updateById(apply);
    }
    StudioAxfAuth auth = studioAxfAuthRpService.getById(studioId).orElse(null);
    if (null != auth && auth.getUserId().equals(merchantNo)) {
      auth.setUsed(ConfirmEnum.YES.name());
      auth.setUpdateTime(new Date());
      studioAxfAuthRpService.updateById(auth);
    }
  }

}
