package com.jiejing.fitness.finance.service.cashier.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.jiejing.common.exception.BizException;
import com.jiejing.common.model.PageVO;
import com.jiejing.common.utils.collection.CollectionUtil;
import com.jiejing.common.utils.convert.BeanUtil;
import com.jiejing.common.utils.http.HttpAgent;
import com.jiejing.common.utils.text.StringUtil;
import com.jiejing.fitness.enums.finance.BrandCashierTransTypeEnum;
import com.jiejing.fitness.finance.api.cashier.vo.StudioCashierRecordVO;
import com.jiejing.fitness.finance.api.cashier.vo.StudioCashierStatisticVO;
import com.jiejing.fitness.finance.api.cashier.vo.StudioSettleVO;
import com.jiejing.fitness.finance.repository.entity.GlobalConfig;
import com.jiejing.fitness.finance.repository.entity.StudioCashierRecord;
import com.jiejing.fitness.finance.repository.entity.StudioSettleRecord;
import com.jiejing.fitness.finance.repository.query.PageBrandCashierRecordQuery;
import com.jiejing.fitness.finance.repository.query.PageStudioSettleQuery;
import com.jiejing.fitness.finance.repository.service.GlobalConfigRpService;
import com.jiejing.fitness.finance.repository.service.StudioCashierRecordRpService;
import com.jiejing.fitness.finance.repository.service.StudioSettleRecordRpService;
import com.jiejing.fitness.finance.service.cashier.StudioCashierService;
import com.jiejing.fitness.finance.service.cashier.convert.CashierConvert;
import com.jiejing.fitness.finance.service.cashier.params.PageStudioCashierParams;
import com.jiejing.fitness.finance.service.cashier.params.PageStudioSettleParams;
import com.jiejing.fitness.finance.service.enums.FinanceErrorEnums;
import com.jiejing.fitness.finance.service.enums.GlobalConfigEnums;
import com.jiejing.paycenter.common.enums.common.PayChannelEnums;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

/**
 * @author chengyubing
 * @since 2024/2/29 11:08
 */
@Slf4j
@Service
public class StudioCashierServiceImpl implements StudioCashierService {

  @Resource
  private StudioCashierRecordRpService studioCashierRecordRpService;

  @Resource
  private StudioSettleRecordRpService studioSettleRecordRpService;

  @Resource
  private GlobalConfigRpService globalConfigRpService;

  @Override
  public Map<PayChannelEnums, String> getAppId() {
    GlobalConfig config = globalConfigRpService.getById(GlobalConfigEnums.CASHIER_APP_IDS.getCode())
        .orElse(new GlobalConfig());
    return JSON.parseObject(config.getConfigValue(), Map.class);
  }

  @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
  @Override
  public void bind(List<Long> ids, String orderNo, String businessNo, String subOrderNo) {
    List<StudioCashierRecord> records = studioCashierRecordRpService.getListByIds(ids);
    if (records.stream().anyMatch(e -> StringUtil.isNotBlank(e.getSubOrderNo()))) {
      throw new BizException(FinanceErrorEnums.EXIST_BIND_RECORD);
    }
    studioCashierRecordRpService.updateByIds(
        StudioCashierRecord.builder()
            .orderNo(orderNo)
            .businessNo(businessNo)
            .subOrderNo(subOrderNo)
            .updateTime(new Date()).build(), ids);
  }

  @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
  @Override
  public void unbind(List<Long> ids) {
    studioCashierRecordRpService.updateByIds(
        StudioCashierRecord.builder().orderNo("").businessNo("").subOrderNo("").updateTime(new Date())
            .build(), ids);
  }

  @Override
  public StudioCashierRecordVO getById(Long id) {
    StudioCashierRecord record = studioCashierRecordRpService.getById(id).orElse(null);
    if (null == record) {
      return null;
    }
    if (BrandCashierTransTypeEnum.PAY.getCode().equals(record.getTransType())) {
      List<StudioCashierRecord> refunds = studioCashierRecordRpService.listByRelatedTransNo(
          record.getTransNo());
      return CashierConvert.convertVO(record, refunds);
    }
    return CashierConvert.convertVO(record);
  }

  @Override
  public List<StudioCashierRecordVO> listByOrderNo(String orderNo) {
    List<StudioCashierRecord> list = studioCashierRecordRpService.listByOrderNo(orderNo);
    List<String> payNos = list.stream()
        .filter(e -> BrandCashierTransTypeEnum.PAY.getCode().equals(e.getTransType()))
        .map(StudioCashierRecord::getTransNo)
        .collect(Collectors.toList());
    Map<String, List<StudioCashierRecord>> payNoRefundMap = studioCashierRecordRpService.mapByRelatedTransNos(
        payNos);
    return CashierConvert.convertList(list, payNoRefundMap);
  }

  @Override
  public List<StudioCashierRecordVO> listByOrderNos(Set<String> orderNos) {
    List<StudioCashierRecord> list = studioCashierRecordRpService.listByOrderNos(orderNos);
    List<String> payNos = list.stream()
        .filter(e -> BrandCashierTransTypeEnum.PAY.getCode().equals(e.getTransType()))
        .map(StudioCashierRecord::getTransNo)
        .collect(Collectors.toList());
    Map<String, List<StudioCashierRecord>> payNoRefundMap = studioCashierRecordRpService.mapByRelatedTransNos(
        payNos);
    return CashierConvert.convertList(list, payNoRefundMap);
  }

  @Override
  public List<StudioCashierRecordVO> listBySubOrderNo(List<String> subOrderNos) {
    List<StudioCashierRecord> list = studioCashierRecordRpService.listBySubOrderNo(subOrderNos);
    return CashierConvert.convertList(list);
  }

  @Override
  public PageVO<StudioCashierRecordVO> page(PageStudioCashierParams params) {
    PageBrandCashierRecordQuery query = JSON.parseObject(JSON.toJSONString(params),
        PageBrandCashierRecordQuery.class);
    Page<StudioCashierRecord> page = studioCashierRecordRpService.page(query);
    return CashierConvert.convertPageVO(page);
  }

  @Override
  public PageVO<StudioSettleVO> page(PageStudioSettleParams params) {
    PageStudioSettleQuery query = BeanUtil.map(params, PageStudioSettleQuery.class);
    Page<StudioSettleRecord> page = studioSettleRecordRpService.page(query);
    return CashierConvert.convertSettlePageVO(page);
  }

  @Override
  public StudioCashierStatisticVO statistic(PageStudioCashierParams params) {
    PageBrandCashierRecordQuery query = JSON.parseObject(JSON.toJSONString(params),
        PageBrandCashierRecordQuery.class);
    return studioCashierRecordRpService.statistic(query);
  }

  @Override
  public List<StudioCashierRecordVO> listByIds(List<Long> ids) {
    List<StudioCashierRecord> records = studioCashierRecordRpService.getListByIds(ids);
    List<String> payNos = Optional.ofNullable(records).orElse(new ArrayList<>(1)).stream()
        .filter(e -> BrandCashierTransTypeEnum.PAY.getCode().equals(e.getTransType()))
        .map(StudioCashierRecord::getTransNo)
        .collect(Collectors.toList());
    Map<String, List<StudioCashierRecord>> payNoRefundMap = studioCashierRecordRpService.mapByRelatedTransNos(
        payNos);
    return CashierConvert.convertList(records, payNoRefundMap);
  }

  @Override
  public List<StudioCashierRecordVO> listByTransNo(List<String> transNos) {
    List<StudioCashierRecord> records = studioCashierRecordRpService.listByTransNo(transNos);
    List<String> payNos = records.stream()
        .filter(e -> BrandCashierTransTypeEnum.PAY.getCode().equals(e.getTransType()))
        .map(StudioCashierRecord::getTransNo)
        .collect(Collectors.toList());
    Map<String, List<StudioCashierRecord>> payNoRefundMap = studioCashierRecordRpService.mapByRelatedTransNos(
        payNos);
    return CashierConvert.convertList(records, payNoRefundMap);
  }

  @Override
  public StudioSettleVO getSettle(Long id) {
    return studioSettleRecordRpService.getById(id)
        .map(CashierConvert::convertSettleVO)
        .orElse(null);
  }

  @Override
  public List<StudioCashierRecordVO> listByRelatedTransNo(String relatedTransNo) {
    List<StudioCashierRecord> records = studioCashierRecordRpService.listByRelatedTransNo(
        relatedTransNo);
    return CashierConvert.convertList(records);
  }

  public static void main(String[] args) {
    String token = "83_5IJi4qSVAhY4leEebVWVzDuNZT5RAbwl9cEWi_iw6N5DWqWZ0jS2A1FvOn7lrxJaFqzwqXRAnowcDkoQW8vRmEr6H9lKUe-4khE8GCgXML7MSqxWmm8ZwDBWIPFEriqiAYYTfpybdKrK0FXxGSGgAEDWIW";
    String url = "https://api.weixin.qq.com/wxaapi/wxaembedded/add_embedded?access_token=%s";
    JSONObject json = HttpAgent.postAsJson(String.format(url, token),
        new JSONObject().fluentPut("appid", "wx2e1e65f65f6e257d")).asJSONObject();
    System.out.println(json);
  }

}
