package com.jiejing.fitness.finance.service.cashier.convert;

import static java.math.BigDecimal.ZERO;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.jiejing.common.model.PageVO;
import com.jiejing.common.utils.collection.CollectionUtil;
import com.jiejing.common.utils.convert.BeanUtil;
import com.jiejing.common.utils.crypt.AesUtil;
import com.jiejing.common.utils.text.SensitiveUtil;
import com.jiejing.fitness.enums.finance.BrandCashierTransStateEnum;
import com.jiejing.fitness.finance.api.cashier.vo.StudioCashierRecordVO;
import com.jiejing.fitness.finance.api.cashier.vo.StudioSettleVO;
import com.jiejing.fitness.finance.repository.entity.StudioCashierRecord;
import com.jiejing.fitness.finance.repository.entity.StudioSettleRecord;
import com.jiejing.fitness.finance.service.utils.MoneyUtil;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.BinaryOperator;
import java.util.stream.Collectors;
import org.springframework.beans.BeanUtils;
import org.springframework.data.domain.Page;

/**
 * @author chengyubing
 * @since 2024/2/29 11:09
 */
public class CashierConvert {

  public static StudioCashierRecordVO convertVO(StudioCashierRecord record) {
    StudioCashierRecordVO result = new StudioCashierRecordVO();
    BeanUtils.copyProperties(record, result, "extra");
    result.setExtra(JSON.parseObject(Optional.ofNullable(record.getExtra()).orElse("{}")));
    return result;
  }

  public static List<StudioCashierRecordVO> convertList(List<StudioCashierRecord> list) {
    if (CollectionUtil.isEmpty(list)) {
      return Lists.newArrayList();
    }
    return list.stream().map(CashierConvert::convertVO).collect(Collectors.toList());
  }

  public static List<StudioCashierRecordVO> convertList(List<StudioCashierRecord> list,
      Map<String, List<StudioCashierRecord>> payNoRefundMap) {
    if (CollectionUtil.isEmpty(list)) {
      return Lists.newArrayList();
    }
    return list.stream()
        .map(e -> convertVO(e, payNoRefundMap.getOrDefault(e.getTransNo(), new ArrayList<>(1))))
        .collect(Collectors.toList());
  }

  public static StudioCashierRecordVO convertVO(StudioCashierRecord record,
      List<StudioCashierRecord> refunds) {
    StudioCashierRecordVO result = convertVO(record);
    result.setLeftTransAmount(MoneyUtil.subtract(record.getTransAmount(),
        refunds.stream()
            .filter(e -> BrandCashierTransStateEnum.REFUND_SUCCESS.getCode().equals(e.getTransState())
                || BrandCashierTransStateEnum.REFUNDING.getCode().equals(e.getTransState()))
            .map(StudioCashierRecord::getTransAmount).reduce(BigDecimal::add).orElse(ZERO)));
    result.setLeftActualAmount(MoneyUtil.subtract(record.getActualAmount(),
        refunds.stream()
            .filter(e -> BrandCashierTransStateEnum.REFUND_SUCCESS.getCode().equals(e.getTransState())
                || BrandCashierTransStateEnum.REFUNDING.getCode().equals(e.getTransState()))
            .map(StudioCashierRecord::getActualAmount).reduce(BigDecimal::add).orElse(ZERO)));
    return result;
  }

  public static PageVO<StudioCashierRecordVO> convertPageVO(Page<StudioCashierRecord> page,
      List<StudioCashierRecordVO> list) {
    return PageVO.convert(page, convertList(page.getContent()));
  }

  public static PageVO<StudioSettleVO> convertSettlePageVO(Page<StudioSettleRecord> page) {
    return PageVO.convert(page, convertSettleList(page.getContent()));
  }

  private static List<StudioSettleVO> convertSettleList(List<StudioSettleRecord> list) {
    if (CollectionUtil.isEmpty(list)) {
      return Lists.newArrayList();
    }
    return list.stream().map(e -> {
      StudioSettleVO vo = CashierConvert.convertSettleVO(e);
      vo.setCardNo(SensitiveUtil.bankCard(AesUtil.decrypt(e.getSalt(), e.getCardNo())));
      return vo;
    }).collect(Collectors.toList());
  }

  public static StudioSettleVO convertSettleVO(StudioSettleRecord record) {
    StudioSettleVO vo = BeanUtil.map(record, StudioSettleVO.class);
    vo.setCardNo(AesUtil.decrypt(record.getSalt(), record.getCardNo()));
    return vo;
  }
}
