import axios from 'axios';
import { load as cookieLoad, save as cookieSave } from 'react-cookies';
const { webDomain, apiDomain } = require('./common/webConfigs');

// 创建axios实例
let serverPath = apiDomain;

const service = axios.create({
  baseURL: serverPath, // api 的 base_url
  timeout: 20000, // 请求超时时间
});

// request拦截器
service.interceptors.request.use(
  (config) => {
    if (config.url.indexOf('public/') === -1 && cookieLoad('token')) {
      return config;
    } else if (config.headers.module) {
      const url = JSON.parse(JSON.stringify(config.url));
      config.url = `${config.headers.module}/` + url;
      return config;
    } else {
      // config.baseURL = '';
      return config;
    }
  },
  (error) => {
    // Do something with request error
    Promise.reject(error);
  },
);

// response 拦截器
service.interceptors.response.use(
  (response) => {
    const { data } = response;
    const { code, result, message } = data;
    // 坚决禁止在response中做业务异常操作
    const rejectCode = [];
    if (code < 200 || code > 300) {
      processHttpError(response);
    } else {
      if (rejectCode.indexOf(code) > -1) {
        console.table({
          time: new Date().toString(),
          code,
          errorInfo: message || result,
          path: response.config.url,
        });
        return Promise.reject(data);
      }
      return data;
    }
  },
  (error) => {
    try {
      processHttpError(error.response);
    } catch (e) {
      if (error.toString().indexOf('Error: timeout') !== -1) {
        console.table({
          time: new Date().toString(),
          errorInfo: '网络请求超时',
        });
        return Promise.reject(error);
      }
      if (error.toString().indexOf('Error: Network Error') !== -1) {
        console.table({
          time: new Date().toString(),
          errorInfo: '网络请求错误',
        });
        return Promise.reject(error);
      }
    }

    return Promise.reject(error);
  },
);
function processHttpError(response) {
  const { data } = response;
  let message = '';
  const status = Number(data.code) > 0 ? Number(data.code) : Number(data.status);
  const code = Number(data.code);
  if (status === 404) {
    message = '404 网络错误 \n 请检查你的本地网络是否连接';
  } else if (status === 500) {
    try {
      message = data.message || 'Oops! 500错误 \n 服务器异常';
    } catch (e) {
      message = 'Oops! 500错误 \n 服务器异常';
    }
  } else if (status === 0) {
    return;
  } else {
    message = data.message;
  }
  console.table({
    code: code > 0 ? code : status,
    time: new Date().toString(),
    errorInfo: message,
    path: response.config.url,
  });
}
export default service;
