package com.xiaomai.cases.polar.training;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.xiaomai.client.DataUserInfo;
import com.xiaomai.enums.ApiModule;
import com.xiaomai.enums.RequestType;
import com.xiaomai.enums.Terminal;
import com.xiaomai.utils.*;
import org.springframework.stereotype.Component;
import org.testng.Assert;
import org.testng.Reporter;

import java.util.List;
import java.util.UUID;

/**
 * 训练营活动相关工具类
 *
 * @author adu
 * data 2024/7/3 15:12
 */
@Component("trainingTools")
public class TrainingTools extends XMBaseTest {
    String applyStartTime1 = String.valueOf(CommUtil.getWholeHourTimestamp()); //当前时间的整点
    String applyEndTime1 = CommUtil.oneKeyGetAddDay(1);//明天0点
    String openStartDate0 = applyEndTime1;
    String openEndDate0 = CommUtil.getNDayEndTimeTamp(2);//后天23：59：59

    long currentTimestamp = System.currentTimeMillis();
    // 在当前时间的基础上，加上10分钟（1000毫秒*60*10）
    long futureTimestamp = currentTimestamp + 600000;
    String futureTimestampS = String.valueOf(futureTimestamp);

    /**
     * 同一个ApiModule下的登录信息封装出一个类，减少代码冗余
     *
     * @param apiName
     * @param userInfo
     */
    public void setUP(String apiName, DataUserInfo... userInfo) {
        dataApi.setApiModule(ApiModule.Polar_Training)
                .setApiName(apiName)
                .setTerminal(Terminal.B);
        super.beforeDataRequest(userInfo);
    }

    /**
     * 新建训练营活动：保存并发布
     *
     * @param applyStartTime    报名开始时间
     * @param applyEndTime      报名结束时间
     * @param courseId          关联课程ID
     * @param openStartDate     开营开始时间
     * @param openEndDate       开营结束时间
     * @param price             售卖价格
     * @param title             训练营活动名称
     * @param totalClassHour    总课次数
     * @param traineeLowerLimit 开营人数
     * @param trainerList       上课教练
     * @param xcxSaleStatus     是否开启小程序售卖 YES，NO
     * @return 创建的训练营活动ID
     */
    public String createThenPublishTraining(String applyStartTime, String applyEndTime, String courseId, String openStartDate,
                                            String openEndDate, Double price, String title, int totalClassHour,
                                            int traineeLowerLimit, List<String> trainerList, String xcxSaleStatus) {
        setUP("API_createThenPublishTraining");
        JSONObject body = new JSONObject();
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId()); // 场馆
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("applyEndTime", applyEndTime);//报名结束时间
        body.put("applyStartTime", applyStartTime);//报名开始时间
        body.put("courseId", courseId); //关联课程ID
        body.put("creatorId", dataApi.getLoginInfo().getAdminId()); // 创建人即操作者
        body.put("openEndDate", openEndDate); //开营开始时间
        body.put("openStartDate", openStartDate); //开营结束时间
        body.put("price", price); //售卖价格
        body.put("surfaceUrl", "https://xiaomai-res.oss-cn-hangzhou.aliyuncs.com/xmfit/course_cover/training_camp_0.png"); // 默认活动封面图
        body.put("title", title); // 训练营活动名称：杭州自动化测试训练营
        body.put("totalClassHour", totalClassHour); // 总课次数
        body.put("traineeLowerLimit", traineeLowerLimit); // 开营人数
        body.put("traineeUpperLimit", 99); // 人数上限
        body.put("trainerList", trainerList); // 上课教练
        body.put("xcxSaleStatus", xcxSaleStatus); //是否开启小程序售卖
        body.put("underlinePrice", 99999.99); //划线价
        //图文介绍不设置，所以回归时check下 图文介绍
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        if (XMJSONPath.readPath(dataApi.getApi_response(), "$.message").equals("训练营名称不能重复")) {
            return "";
        }
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));
        return XMJSONPath.readPath(dataApi.getApi_response(), "$.result");
    }

    /**
     * 引用时可以继承 BaseCreateTrainingData 这个类，然后直接传值就可以
     * 价格0元，一键创建已发布且报名中的训练营活动
     *
     * @param applyStartTime 报名开始时间，要早于第二天的0点，最好是晚于当前时间半个小时
     * @param courseId       需要传入课程ID
     * @param trainerList    需要传入教练
     * @return 活动ID
     */
    public String oneKeyCreateTraining(String courseId, List<String> trainerList, String applyStartTime) {
        setUP("API_createThenPublishTraining");
        String courseName = "训练营课程-感统课";
        String trainingTitle = "三期36天训练营感统课" + RandomStringUtil.randomString(5);
        JSONObject body = new JSONObject();
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId()); // 场馆
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("applyEndTime", applyEndTime1);//报名结束时间
        body.put("applyStartTime", applyStartTime);//报名开始时间
        body.put("courseId", courseId); //关联课程ID
        body.put("creatorId", dataApi.getLoginInfo().getAdminId()); // 创建人即操作者
        body.put("openEndDate", openEndDate0); //开营结束时间
        body.put("openStartDate", openStartDate0); //开营开始时间
        body.put("price", 0); //售卖价格
        body.put("surfaceUrl", "https://xiaomai-res.oss-cn-hangzhou.aliyuncs.com/xmfit/course_cover/training_camp_0.png"); // 默认活动封面图
        body.put("title", trainingTitle);
        body.put("totalClassHour", 2); // 总课次数
        body.put("traineeLowerLimit", 1); // 开营人数
        body.put("traineeUpperLimit", 99); // 人数上限
        body.put("trainerList", trainerList); // 上课教练
        body.put("xcxSaleStatus", "YES"); //开启小程序售卖
        body.put("underlinePrice", 19.9); //划线价
        //图文介绍不设置，所以回归时check下 图文介绍
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));
        return XMJSONPath.readPath(dataApi.getApi_response(), "$.result");


    }


    /**
     * 提供训练营活动名称，课程ID和教练ID 快速创建未发布的活动
     *
     * @param title       训练营活动名称，可以传空，有默认值
     * @param courseId
     * @param trainerList
     * @return
     */
    public String createTrainingNoPublish(String title, String courseId, List<String> trainerList) {
        setUP("API_createTrainingNoPublish");
        JSONObject body = new JSONObject();
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId()); // 场馆
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("applyEndTime", applyEndTime1);//报名结束时间
        body.put("applyStartTime", applyStartTime1);//报名开始时间
        body.put("courseId", courseId); //关联课程ID
        body.put("creatorId", dataApi.getLoginInfo().getAdminId()); // 创建人即操作者
        body.put("openEndDate", openEndDate0); //开营开始时间
        body.put("openStartDate", openStartDate0); //开营结束时间
        body.put("price", 0.01); //售卖价格
        body.put("surfaceUrl", "https://xiaomai-res.oss-cn-hangzhou.aliyuncs.com/xmfit/course_cover/training_camp_0.png"); // 默认活动封面图
        body.put("title", title); // 训练营活动名称："杭州自动化测试训练营-未发布"
        if (title.isEmpty()) {
            body.put("title", "杭州自动化测试训练营-未发布");
        }

        body.put("totalClassHour", 2); // 总课次数
        body.put("traineeLowerLimit", 2); // 开营人数
        body.put("traineeUpperLimit", 99); // 人数上限
        body.put("trainerList", trainerList); // 上课教练
        body.put("xcxSaleStatus", "YES"); //开启小程序售卖
        body.put("underlinePrice", 99999.99); //划线价
        //图文介绍不设置，所以回归时check下 图文介绍
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));
        return XMJSONPath.readPath(dataApi.getApi_response(), "$.result");
    }

    /**
     * 删除训练营活动
     *
     * @param id 活动ID
     */
    public void deleteTraining(String id) {
        setUP("API_deleteTraining");
        JSONObject body = new JSONObject();
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("id", id);
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

    }

    /**
     * 训练营活动有报名订单时操作删除时会删除失败，就用这个方法
     *
     * @param id
     */
    public void delTrainingFail(String id) {
        setUP("API_deleteTraining");
        JSONObject body = new JSONObject();
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("id", id);
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs).assetsSuccess(false);
    }

    /**
     * 根据训练营活动名称全局搜索
     *
     * @param title 活动名称
     */
    public void commonSearchTraining(String title) {
        setUP("API_commonSearchTraining");
        JSONObject body = new JSONObject();
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("title", title);
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

    }

    /**
     * 查询可报名的训练营列表
     *
     * @param title 活动名称
     */
    public JSONObject fetchAppliableCampList(String title) {
        setUP("API_fetchAppliableCampList");
        JSONObject body = new JSONObject();
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("title", title);
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        JSONObject response = dataApi.getBodyInJSON();
        return response;

    }

    /**
     * 训练营活动列表搜索
     *
     * @param courseId  根据关联课程搜索，可传""
     * @param stateList 根据活动状态搜索
     *                  不指定时传空，new ArrayList<>()
     *                  未发布：UNPUBLISHED
     *                  报名未开始：APPLY_NOT_START
     *                  报名中：APPLYING
     *                  待开营：TO_BE_OPEN
     *                  进行中：OPENING
     *                  已结束：CLOSED
     *                  已取消：CANCEL
     */
    public void searchTrainingList(String courseId, String trainingName, List<String> stateList) {
        setUP("API_searchTrainingList");
        JSONObject body = new JSONObject();
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("current", 0);
        body.put("size", 500);
        body.put("courseId", courseId);
        body.put("stateList", stateList);
        if (!trainingName.isEmpty()) {
            body.put("title", trainingName);
        }

        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

    }

    /**
     * 创建/编辑是训练营名称重名校验
     *
     * @param title 活动名称
     * @return 返回校验结果 true 是有重名不通过，false 是无重名 可创建
     */
    public Boolean checkTrainingTitleDuplicate(String title) {
        setUP("API_checkTrainingTitleDuplicate");
        JSONObject body = new JSONObject();
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("title", title);
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(),"$.success"))==true,"调用接口返回结果："+XMJSONPath.readPath(dataApi.getApi_response(),"$.message"));
        return Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.result"));
    }

    /**
     * 查看训练营活动详情：基础信息
     *
     * @param id 活动ID
     */
    public void findTrainingDetailById(String id) {
        setUP("API_findTrainingDetailById");
        JSONObject body = new JSONObject();
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("id", id);
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(),"$.success"))==true,"调用接口返回结果："+XMJSONPath.readPath(dataApi.getApi_response(),"$.message"));


    }


    /**
     * 训练营详情：查看 报名会员
     *
     * @param trainingCampId 训练营ID
     * @param content        根据手机号/会员姓名搜索
     */
    public void getTrainingApplyRecordsList(String trainingCampId, String content) {
        setUP("API_getTrainingApplyRecordsList");
        JSONObject body = new JSONObject();
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("trainingCampId", trainingCampId);
        if (null != content || !content.isEmpty()) {
            body.put("content", content);
        }
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

    }

    /**
     * B 端操作：在训练营活动详情-报名学员-给学员报名
     *
     * @param memberId         会员ID
     * @param receivableAmount 支付金额
     * @param campId           训练营活动ID
     * @param userCouponId     已发放给学员的优惠券记录ID
     */
    public void doSignCampOrder(String memberId, Double receivableAmount, String campId, String userCouponId) {
        setUP("API_signOrderCamp");
        JSONObject body = new JSONObject();
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        if (userCouponId != null || !userCouponId.isEmpty()) {
            body.put("userCouponId", userCouponId);
        }

        body.put("salesmanId", dataApi.getLoginInfo().getAdminId());
        body.put("operationTime", TimeUtils.getCurrentTime());
        body.put("objectId", RandomStringUtil.randomString(20));
        body.put("memberId", memberId);
        body.put("receivableAmount", receivableAmount);

        JSONObject campInfo = new JSONObject();
        campInfo.put("campId", campId);
        body.put("camp", campInfo);
        JSONArray paymentWays = new JSONArray();
        JSONObject payObject = new JSONObject();
        payObject.put("amount", receivableAmount);
        payObject.put("paymentWay", "1");
        payObject.put("voucherResourceIds", new JSONArray());
        paymentWays.add(payObject);
        body.put("paymentWays", paymentWays);
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Reporter.log(XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

    }

    /**
     * 查看训练营活动-报名会员-报名详情
     *
     * @param trainingAccountId 即memberCardId，从查看报名会员列表返回数据中获取对应学员的报名数据（memberCardId）
     */
    public void getTrainingAccountDetailById(String trainingAccountId) {
        setUP("API_getTrainingAccountDetailById");
        JSONObject body = new JSONObject();
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("id", trainingAccountId);
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

    }

    /**
     * 退单训练营活动报名详情-退款
     *
     * @param memberId     会员ID
     * @param refundAmount 退出金额
     * @param memberCampId 会员的训练营账户ID
     * @param quantity     退出购买数量
     */
    public void refundCampOrder(String memberId, Double refundAmount, String memberCampId, int quantity) {
        setUP("API_refundCampOrder");
        JSONObject body = new JSONObject();
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());

        body.put("operationTime", TimeUtils.getCurrentTime());
        body.put("objectId", RandomStringUtil.randomString(20));
        body.put("memberId", memberId);
        body.put("receivableAmount", refundAmount);
        //退单内容
        JSONObject campInfo = new JSONObject();
        campInfo.put("memberCampId", memberCampId);
        campInfo.put("giftQuantity", 0);
        campInfo.put("quantity", quantity);
        campInfo.put("refundAmount", refundAmount);
        body.put("camp", campInfo);
        //结算方式
        JSONArray paymentWays = new JSONArray();
        JSONObject payObject = new JSONObject();
        payObject.put("amount", refundAmount);
        payObject.put("paymentWay", "1");
        payObject.put("voucherResourceIds", new JSONArray());
        paymentWays.add(payObject);
        body.put("paymentWays", paymentWays);
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));


    }

    /**
     * 根据会员姓名/手机号搜索会员
     *
     * @param content 手机号/姓名
     * @return 返回搜索的的目标数据第一条的会员ID
     */
    public String searchTargerMember(String content) {
        dataApi.setApiModule(ApiModule.Polar_Clue)
                .setApiName("API_searchMember")
                .setTerminal(Terminal.B);
        super.beforeDataRequest();
        JSONObject body = new JSONObject();
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("content", content);
        body.put("searchType", "COMMON");
        body.put("current", 0);
        body.put("size", 20);
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));
        int tatal = XMJSONPath.getJSONArrayByReadPath(dataApi.getApi_response(), "$.result.records").size();
        if (tatal > 0) {
            return XMJSONPath.getJSONArrayByReadPath(dataApi.getApi_response(), "$.result.records").getJSONObject(0).getString("id");
        }
        return "";

    }

    /**
     * 查看训练营报名记录
     *
     * @param tenantMemberCardId  训练营对应ID
     * @param purchaseChannel     来源即购买渠道 ：线下办理（B_OFF_LINE） 和 上线购买（C_ON_LINE），线上购买的case要等小程序买了之后check
     * @param memberCardStateList 报名记录状态：生效中["NORMAL"]，已结束 ["EXPIRED"]，已用尽["USED_UP"]，已退费["REFUNDED"]，已作废["DISCARD"]
     */
    public void searchTrainingAccountRecordsList(String tenantMemberCardId, String purchaseChannel, List<String> memberCardStateList) {
        setUP("API_searchTrainingAccountRecordsList");
        JSONObject body = new JSONObject();
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("accountType", "TRAINING");
        body.put("current", 0);
        body.put("size", 20);
        if (!tenantMemberCardId.isEmpty()) {
            body.put("tenantMemberCardId", tenantMemberCardId);
        }
        if (!purchaseChannel.isEmpty()) {
            body.put("purchaseChannel", purchaseChannel);
        }
        body.put("memberCardStateList", memberCardStateList);
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));


    }

    /**
     * 训练营报名记录页面中 -全局搜索（根据学员手机号/姓名）-点击目标学员-进入学员购买的训练营相关记录
     *
     * @param memberId 会员ID
     */
    public void fetchUserTrainingCardList(String memberId) {
        setUP("API_getMemberTrainingCardRecords");
        JSONObject body = new JSONObject();
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("accountType", "TRAINING");
        body.put("current", 0);
        body.put("size", 20);
        body.put("memberId", memberId);
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));


    }

    /**
     * 获取训练计划的动作部位
     * @return
     */
    public JSONObject getBodyPartList() {
        setUP("API_getBodyPartList");
        JSONObject body = new JSONObject();
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return dataApi.getBodyInJSON();
    }
    /**
     * 检验训练计划的动作重名
     * @return
     */
    public boolean trainingExerciseCheckName(String exerciseId,String exerciseName) {
        setUP("API_trainingExerciseCheckName");
        JSONObject body = new JSONObject();
        body.put("id", exerciseId);//创建时检验id传""；编辑时检验id传值
        body.put("name", exerciseName);
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return Boolean.parseBoolean(dataApi.getBodyInJSON().getString("result"));
    }
    /**
     * 创建训练计划的动作
     * @return
     */
    public String trainingExerciseCreate(String exerciseName,String bodyPartId) {
        setUP("API_trainingExerciseCreate");
        JSONArray bodyPartIds = new JSONArray();
        bodyPartIds.add(bodyPartId);
        JSONObject body = new JSONObject();
        body.put("name", exerciseName);
        body.put("bodyPartIds",bodyPartIds);
        body.put("muscle","自动化脚本生成训练肌肉");
        body.put("actionStandards","自动化脚本生成动作标准");
        body.put("notes","自动化脚本生成注意事项");
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return dataApi.getBodyInJSON().getString("result");
    }

    /**
     * 训练计划-训练动作详情
     * @param exerciseId
     * @return
     */
    public JSONObject trainingExerciseGetDetail(String exerciseId) {
        setUP("API_trainingExerciseGetDetail");
        JSONObject body = new JSONObject();
        body.put("id",exerciseId );
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return dataApi.getBodyInJSON();
    }
    /**
     * 编辑训练动作
     * @return
     */
    public String trainingExerciseEdit(String exerciseName,String exerciseId,String bodyPartId) {
        setUP("API_trainingExerciseEdit");
        JSONArray bodyPartIds = new JSONArray();
        bodyPartIds.add(bodyPartId);
        JSONObject body = new JSONObject();
        body.put("name", exerciseName);
        body.put("bodyPartIds",bodyPartIds);
        body.put("muscle","自动化脚本生成训练肌肉");
        body.put("actionStandards","自动化脚本生成动作标准");
        body.put("notes","自动化脚本生成注意事项");
        body.put("id",exerciseId);
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return dataApi.getBodyInJSON().getString("result");
    }
    /**
     * 删除训练动作
     * @param exerciseId
     */
    public void trainingExerciseDelete(String exerciseId) {
        setUP("API_trainingExerciseDelete");
        JSONObject body = new JSONObject();
        body.put("id", exerciseId);
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));
    }
    /**
     * 某个训练部位的所有动作数据
     * @param bodyPartId：胸部、肩部、背部、腿部、臀部、手臂、腹部、心肺、其他
     * @return
     */
    public JSONObject trainingExerciseList(String bodyPartId) {
        setUP("API_trainingExerciseList");
        JSONObject body = new JSONObject();
        body.put("bodyPartId",bodyPartId );
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return dataApi.getBodyInJSON();
    }
    /**
     * 搜索训练动作
     * @return
     */
    public JSONObject getListByName(String exerciseName) {
        setUP("API_getListByName");
        JSONObject body = new JSONObject();
        body.put("size",20 );
        body.put("name",exerciseName);
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return dataApi.getBodyInJSON();

    }
    /**
     * 训练模板列表、包含搜索功能
     * @return
     */
    public JSONObject trainingTemplateListPage(String templateName) {
        setUP("API_trainingTemplateListPage");
        JSONObject body = new JSONObject();
        body.put("studioId",dataApi.getLoginInfo().getStudioId());
        body.put("size",20 );
        body.put("current",0 );
        if(null != templateName || !templateName.isEmpty()) {
            body.put("trainingTemplateName",templateName);
        }
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return dataApi.getBodyInJSON();
    }
    /**
     * 训练模板重名检验
     */
    public boolean trainingTemplateCheckName(String templateId,String templateName) {
        setUP("API_trainingTemplateCheckName");
        JSONObject body = new JSONObject();
        body.put("id",templateId);//创建时检验id传""；编辑时检验id传值
        body.put("templateName",templateName);
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("studioId",dataApi.getLoginInfo().getStudioId());
        body.put("operatorId",dataApi.getLoginInfo().getAdminId());
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return Boolean.parseBoolean(dataApi.getBodyInJSON().getString("result"));

    }
    public JSONObject trainingTemplateItem(String itemType,String itemName){
        JSONObject object = new JSONObject();
        object.put("objectId", UUID.randomUUID().toString());
        object.put("itemType",itemType);//"EXERCISE"动作、"GRAPHIC"图文
        object.put("itemName",itemName);
        return object;
    }
    /**
     * 创建训练模板
     * @return
     */
    public String trainingTemplateCreate(String itemType,String itemName,String templateName) {
        setUP("API_trainingTemplateCreate");

        JSONArray itemList = new JSONArray();
        itemList.add(this.trainingTemplateItem(itemType,itemName));

        JSONObject body = new JSONObject();
        body.put("trainingTemplateName",templateName);
        body.put("trainingTemplateItemList",itemList);
        body.put("templateId","");
        body.put("studioId",dataApi.getLoginInfo().getStudioId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("creator",dataApi.getLoginInfo().getAdminId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());

        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return dataApi.getBodyInJSON().getString("result");
    }
    /**
     * 训练模板详情
     * @return
     */
    public JSONObject trainingTemplateDetail(String templateId) {
        setUP("API_trainingTemplateDetail");
        JSONObject body = new JSONObject();
        body.put("id",templateId);
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("studioId",dataApi.getLoginInfo().getStudioId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return dataApi.getBodyInJSON();
    }
    /**
     * 删除训练模板
     * @return
     */
    public void trainingTemplateDelete(String templateId) {
        setUP("API_trainingTemplateDelete");

        JSONObject body = new JSONObject();
        body.put("id",templateId);
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("studioId",dataApi.getLoginInfo().getStudioId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());

        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));
    }
    /**
     * 创建智能训练计划
     * @return
     */
    public String trainingPlanRecordCreate(String memberId,String exerciseId,String exerciseName){
        setUP("API_trainingPlanRecordCreate");
        String content ="[{\"id\":\""+exerciseId+"\",\"name\":\""+exerciseName+"\",\"trainExerciseType\":\"SYSTEM\",\"arrange\":[{\"timeAmount\":{\"num\":\"12\",\"unit\":\"times\"},\"weightAmount\":{\"num\":\"12\",\"unit\":\"RM\"}},{\"timeAmount\":{\"num\":\"12\",\"unit\":\"times\"},\"weightAmount\":{\"num\":\"12\",\"unit\":\"RM\"}},{\"timeAmount\":{\"num\":\"12\",\"unit\":\"times\"},\"weightAmount\":{\"num\":\"12\",\"unit\":\"RM\"}}]}]";

        //构建trainList
        JSONArray trainList = new JSONArray();
        JSONObject exercise = new JSONObject();
        exercise.put("trainingTemplateItemType","EXERCISE");
        exercise.put("name","训练动作");
        exercise.put("content",content);
        JSONObject graphic = new JSONObject();
        graphic.put("trainingTemplateItemType","GRAPHIC");
        graphic.put("name","训练建议");
        graphic.put("content","{\"items\":[{\"type\":\"TEXT\",\"content\":\"由于是低强度训练，建议注重动作的标准性，避免受伤。可以在训练前后进行适当拉伸，提高训练效果。\"}]}");
        trainList.add(exercise);
        trainList.add(graphic);

        //构建请求
        JSONObject body = new JSONObject();
        body.put("memberId",memberId);
        body.put("trainingPlanName","脚本生成智能训练计划"+RandomStringUtil.randomString(5));
        body.put("id","");
        body.put("sendStatus","SEND");//已发送"SEND"，未发送"UN_SEND"
        body.put("trainingTemplateDetailList",trainList);
        body.put("trainingPlanType","AUTO");
        body.put("studioId",dataApi.getLoginInfo().getStudioId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("creator",dataApi.getLoginInfo().getAdminId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        //调用请求
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return dataApi.getBodyInJSON().getString("result");
    }
    /**
     * 编辑训练计划
     * @return
     */
    public String trainingPlanRecordEdit(String memberId,String content,String planName,String recordId,String templateId,String templateName,String itemType,String itemName) {
        setUP("API_trainingPlanRecordEdit");

        JSONObject template = new JSONObject();
        template.put("templateId",templateId);
        template.put("templateName",templateName);

        JSONObject train = new JSONObject();
        train.put("name",itemName);
        train.put("trainingTemplateItemType",itemType);
        if(null != content || !content.isEmpty()) {
            train.put("content",content);
        }
        JSONArray trainList = new JSONArray();
        trainList.add(train);

        JSONObject body = new JSONObject();
        body.put("memberId",memberId);
        body.put("trainingPlanName",planName);
        body.put("template",template);
        body.put("id",recordId);
        body.put("sendStatus","SEND");//保存并发送
        body.put("templateId",templateId);
        body.put("templateName",templateName);
        body.put("trainingTemplateDetailList",trainList);
        body.put("studioId",dataApi.getLoginInfo().getStudioId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("creator",dataApi.getLoginInfo().getAdminId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());

        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return dataApi.getBodyInJSON().getString("result");

    }
    /**
     * 训练计划发送给学员：状态从未发送变动为发送
     * @param recordId
     */
    public void updateSendStatus(String recordId) {
        setUP("API_updateSendStatus");
        JSONObject body = new JSONObject();
        body.put("id",recordId);
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("studioId",dataApi.getLoginInfo().getStudioId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));
    }

    /**
     * 训练计划详情
     * @param recordId
     * @return
     */
    public JSONObject trainingPlanRecordDetail(String recordId) {
        setUP("API_trainingPlanRecordDetail");
        JSONObject body = new JSONObject();
        body.put("id",recordId);
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("studioId",dataApi.getLoginInfo().getStudioId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return dataApi.getBodyInJSON();
    }
    /**
     * 训练记录列表
     * @return
     */
    public JSONObject trainingPlanRecordPage(String memberId,String templateId) {
        setUP("API_trainingPlanRecordPage");
        JSONObject body = new JSONObject();
        body.put("size",20);
        body.put("current",0);
        if (null != memberId || !memberId.isEmpty()) {
            body.put("memberId", memberId);
        }
        if (null != templateId || !templateId.isEmpty()) {
            body.put("templateId", templateId);
        }
        body.put("studioId",dataApi.getLoginInfo().getStudioId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return dataApi.getBodyInJSON();
    }
    /**
     * 智能生成训练计划
     */
    public String generateTrainingPlan(String bodyPartId,String memberId) {
        setUP("API_generateTrainingPlan");
        JSONArray bodyPartIds = new JSONArray();
        bodyPartIds.add(bodyPartId);
        String promot ="# 你是一个健身教练，需要根据会员的一些信息，以及给出动作库，输出一份完整的动作训练计划；\n# 要求与限制：\n1.输出的动作仅可取自给定的的动作库；\n2.每一个动作都必须包含：ID(id)，动作名称(name)，组数(groupNum)，重量的单位(weightUnit)，重量的数值(weightNum)，次数的单位(timesUnit)，次数的数值(timesNum)；\nID取该动作的ID；组数仅可输出1-100以内一个正整数；重量的单位可以选择KG(KG)/LBS(LBS)/RM(RM)/空(\"\")，重量的数值只输出1-100以内一个正整数即可；次数单位可以选择次(times)/分钟(minute)/秒(second)/空(\"\")，次数多单位只输出1-100以内一个正整数即可。\n3.动作训练计划必须以json格式输出；\n4.根据会员信息中的动作强度输出不同数量的动作个数，需至少保证5个动作；（动作强度分为：高强度，中等强度，低强度，训练强度越大动作数量越大，至多输出18个动作）。\n5.全部动作输出完毕后，基于你对会员信息的理解，再输出一个字段：训练建议。\n6.不要分析，不要输出任何多余内容，直接输出json内容。\n@@\n# 输出示例：\n接收的会员信息：\n性别：男\n年龄：34\n身高：176\n体重：76kg\n体脂率：30%\n训练部位：胸部、心肺\n训练强度：低强度\n输出的内容：\n```json\n{\n\"exerciseItems\": [\n{\n\"id\": \"1\",\n\"name\": \"杠铃卧推\",\n\"groupNum\": 3,\n\"weightUnit\": \"KG\",\n\"weightNum\": \"25\",\n\"timesUnit\": \"times\",\n\"timesNum\": \"15\",\n},\n],\n\"extra\": [\n\"advise\": \"健身过程中请保持核心收紧，关注自身心率变动\",\n],\n}\n```\n\n# 现在给出的会员信息为，请直接输出json动作：\n性别：女\n年龄：26\n身高：160CM\n体重：48.3KG\n体脂率：26.1%\n训练部位：胸部\n训练强度：低强度\n";
        JSONObject body = new JSONObject();
        body.put("bodyPartIds",bodyPartIds);
        body.put("memberId",memberId);
        body.put("promot",promot);
        body.put("studioId",dataApi.getLoginInfo().getStudioId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return dataApi.getBodyInJSON().getString("result");
    }


    /**
     * 删除训练记录
     * @param recordId
     */
    public void trainingPlanRecordDelete(String recordId) {
        setUP("API_trainingPlanRecordDelete");
        JSONObject body = new JSONObject();
        body.put("id",recordId);
        body.put("studioId",dataApi.getLoginInfo().getStudioId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));
    }

    /**
     * 小程序我的训练列表
     * @param userInfo
     */
    public JSONObject listMyTrainingPlan(String memberId,String trainingPlanName,DataUserInfo...userInfo) {
        dataApi.setApiModule(ApiModule.Lunar_Training)
                .setApiName("API_listMyTrainingPlan")
                .setTerminal(Terminal.minApp);
        super.beforeDataRequest(userInfo);

        JSONObject body = new JSONObject();
        body.put("memberId",memberId);
        body.put("current",0);
        body.put("size",20);
        if (null != trainingPlanName || !trainingPlanName.isEmpty()) {
            body.put("trainingPlanName", trainingPlanName);
        }
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("studioId",dataApi.getLoginInfo().getStudioId());
        body.put("sendStatus","SEND");
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return dataApi.getBodyInJSON();
    }
    /**
     * 小程序我的训练详情
     * @param recordId
     * @param userInfo
     * @return
     */
    public JSONObject findMyTrainingPlanDetailById(String recordId,DataUserInfo...userInfo) {
        dataApi.setApiModule(ApiModule.Lunar_Training)
                .setApiName("API_findMyTrainingPlanDetailById")
                .setTerminal(Terminal.minApp);
        super.beforeDataRequest(userInfo);

        JSONObject body = new JSONObject();
        body.put("id",recordId);

        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return dataApi.getBodyInJSON();
    }
    /**
     * 小程序训练计划的动作详情
     * @return
     */
    public JSONObject findTrainExerciseDetailById(String exerciseId,String planRecordId,DataUserInfo...userInfo) {
        dataApi.setApiModule(ApiModule.Lunar_Training)
                .setApiName("API_findTrainExerciseDetailById")
                .setTerminal(Terminal.minApp);
        super.beforeDataRequest(userInfo);

        JSONObject body = new JSONObject();
        body.put("exerciseId",exerciseId);
        body.put("planRecordId",planRecordId);

        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return dataApi.getBodyInJSON();
    }
    /**
     * 小程序我的训练记录分享
     * @param recordId
     * @param userInfo
     * @return
     */
    public JSONObject getShareDetail(String recordId,DataUserInfo...userInfo) {
        dataApi.setApiModule(ApiModule.Lunar_Training)
                .setApiName("API_getShareDetail")
                .setTerminal(Terminal.minApp);
        super.beforeDataRequest(userInfo);

        JSONObject extJson = new JSONObject();
        extJson.put("studioId",dataApi.getLoginInfo().getStudioId());
        extJson.put("brandId",dataApi.getLoginInfo().getBrandId());

        JSONObject body = new JSONObject();
        body.put("bizId",recordId);
        body.put("shareType","TRAINING_PLAN");
        body.put("extJson", extJson);

        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return dataApi.getBodyInJSON();
    }
    /**
     * AI小助理列表
     * @return
     */
    public JSONObject listModelPerson(){
        setUP("API_listModelPerson");
        JSONObject body = new JSONObject();
        body.put("userId",dataApi.getLoginInfo().getAdminId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("studioId",dataApi.getLoginInfo().getStudioId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return dataApi.getBodyInJSON();
    }
    /**
     * 获取会话id
     * @return
     */
    public String getAiConversationId(String modelId,String personModelId) {
        setUP("API_getAiConversationId");
        JSONObject body = new JSONObject();
        body.put("modelId", modelId);
        body.put("personModelId", personModelId);
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("userId", dataApi.getLoginInfo().getAdminId());
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return dataApi.getBodyInJSON().getString("result");
    }
    /**
     * 获取会话历史信息
     * @return
     */
    public JSONObject getConversationMessagePage(String conversationId) {
        setUP("API_getConversationMessagePage");
        JSONObject body = new JSONObject();
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("current", 0);
        body.put("size", 10);
        body.put("conversationId",conversationId);
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));

        return dataApi.getBodyInJSON();
    }
    /**
     * 清除聊天上下文
     * @return
     */
    public void clearConversationContext(String conversationId) {
        setUP("API_clearConversationContext");
        JSONObject body = new JSONObject();
        body.put("id", conversationId);
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));
    }
    /**
     * 删除聊天记录
     * @return
     */
    public void clearConversationHistory(String conversationId) {
        setUP("API_clearConversationHistory");
        JSONObject body = new JSONObject();
        body.put("id", conversationId);
        body.put("brandId", dataApi.getLoginInfo().getBrandId());
        body.put("studioId", dataApi.getLoginInfo().getStudioId());
        body.put("operatorId", dataApi.getLoginInfo().getAdminId());
        dataApi.doRequest(RequestType.JSON, dataparams, body.toString(), dataheadrs);
        Assert.assertTrue(Boolean.valueOf(XMJSONPath.readPath(dataApi.getApi_response(), "$.success")) == true, "调用接口返回结果：" + XMJSONPath.readPath(dataApi.getApi_response(), "$.message"));
    }

}

