/*
 * @Author: 吴文洁 
 * @Date: 2020-07-14 15:41:30 
 * @Last Modified by: 吴文洁
 * @Last Modified time: 2020-07-23 13:45:16
 * @Description: 大班直播、互动班课列表的筛选组件
 */

import React from 'react';
import { withRouter } from 'react-router-dom';
import { Row, Input, Select } from 'antd';
import Bus from '@/core/bus';

import TeacherSearchSelect from "@/modules/common/TeacherSearchSelect";
import RangePicker from "@/modules/common/DateRangePicker";

import './LiveCourseFilter.less';

const { Search } = Input;
const { Option } = Select;

const defaultQuery = {
  courseName: null,
  startTime: null,
  teacherName: null,
  courseState: undefined,
}

class LiveCourseFilter extends React.Component {

  constructor(props) {
    super(props);
    this.state = {
      query: {...defaultQuery}
    }
  }

  componentWillReceiveProps(nextProps) {
    const { match: { path } } = nextProps;
    const { match: { path: curPath } } = this.props;
    if (path !== curPath) {
      this.setState({
        query: {...defaultQuery}
      })
    }
  }

  // 改变搜索条件
  handleChangeQuery = (field, value) =>  {
    this.setState({
      query: {
        ...this.state.query,
        [field]: value,
        current: 1,
      }
    }, () => {
      if (field === 'courseName') return;
      this.props.onChange(this.state.query)
    });
  }
  
  handleChangeDates = (dates) => {
    const { query } = this.state;
    if (_.isEmpty(dates)) {
      delete query.startTime;
      delete query.endTime;
    } else {
      query.startTime = dates[0].valueOf();
      query.endTime = dates[1].valueOf();
    }
    this.setState({
      query,
      current: 1,
    }, () => {
      this.props.onChange(this.state.query);
    })
  }
  
  // 选择老师
  handleSelectTeacher = (teacher) => {
    const { name: teacherName, teacherId } = teacher;
    this.setState({
      query: {
        ...this.state.query,
        teacherId,
        teacherName,
        current: 1,
      }
    }, () => {
      this.props.onChange(this.state.query);
    })
  }

  // 清空搜索条件
  handleReset = () => {
    this.setState({
      query: {
        ...this.state.query,
        courseName: null,
        startTime: null,
        endTime: null,
        teacherId: null,
        teacherName: null,
        courseState: undefined,
        current: 1,
      },
    }, () => {
      this.props.onChange(this.state.query);
    })
  }

  render() {
    const {
      courseName, startTime, endTime,
      courseState, teacherName, teacherId
    } = this.state.query;
    const { teacherId: _teahcerId } = {};
    const isTeacher = !!_teahcerId;    // 判断是否是老师身份

    return (
      <div className="live-course-filter">
        <Row type="flex" justify="space-between" align="top">
          <div className="search-condition">
            <div className="search-condition__item">
              <span className="search-name">直播名称：</span>
              <Search
                value={courseName}
                placeholder="搜索直播名称"
                onChange={(e) => { this.handleChangeQuery('courseName', e.target.value)}}
                onSearch={ () => { this.props.onChange(this.state.query) } }
                style={{ width: "calc(100% - 70px)" }}
              />
            </div>

            <div className="search-condition__item">
              <span className="search-date">上课日期：</span>
              <RangePicker
                id="course_date_picker"
                allowClear={false}
                value={ startTime ? [moment(startTime), moment(endTime)] : null }
                format={"YYYY-MM-DD"}
                onChange={(dates) => { this.handleChangeDates(dates) }}
                style={{ width: "calc(100% - 70px)" }}
              />
            </div>

            {!isTeacher && 
              <div className="search-condition__item">
                <TeacherSearchSelect
                  id="teacher_select"
                  ref="TeacherSelect"
                  label="上课老师"
                  placeholder="请选择"
                  teacherName={teacherName}
                  onSelect={this.handleSelectTeacher}
                  defaultValue={teacherId}
                />
              </div>
            }
            <div className="search-condition__item">
              <span className="select-status">上课状态：</span>
              <Select
                style={{ width: "calc(100% - 70px)" }}
                placeholder="请选择"
                allowClear={true}
                value={courseState}
                onChange={(value) => { this.handleChangeQuery('courseState', value) }}
              >
                <Option value="UN_START">待上课</Option>
                <Option value="STARTING">上课中</Option>
                <Option value="FINISH">已完成</Option>
                <Option value="EXPIRED">未成功开课</Option>
              </Select>
            </div>
          </div>
          
          <span
            className="icon iconfont"
            onClick={this.handleReset}
          >
            &#xe6a3;
          </span>
        </Row>
      </div>
    )
  }
}

export default withRouter(LiveCourseFilter);