/*
 * @Author: 吴文洁 
 * @Date: 2020-07-14 15:41:30 
 * @Last Modified by: 吴文洁
 * @Last Modified time: 2020-07-23 13:45:16
 * @Description: 大班直播、互动班课列表的筛选组件
 */

import React, { useState, useEffect } from 'react';
import { withRouter } from 'react-router-dom';
import { Row, Input, Select ,Tooltip} from 'antd';
import RangePicker from "@/modules/common/DateRangePicker";
import moment from 'moment';
import StoreService from "@/domains/store-domain/storeService";
import './PlanFilter.less';
const { Search } = Input;
const { Option } = Select;

const DEFAULT_QUERY = {
  planName: null,
  startTime: null,
  endTime:null,
  enableState:null,
  createId: null
}
const defaultCreatorQuery = {
  size: 10,
  current: 1,
  nickName:null
}
function PlanFilter(props) {

  const [expandFilter, setExpandFilter] = useState(false);
  const [query,setQuery] = useState(DEFAULT_QUERY);
  const [hasNext,setHasNext] = useState(false);
  const [creatorQuery,setCreatorQuery] = useState(defaultCreatorQuery);
  const [creatorList,setCreatorList] =  useState([]);

  
  useEffect(() => {
    getCreatorList();
  }, []);

    // 改变搜索条件
  function handleChangeQuery(field, value){
    const _query ={
      ...query,
      [field]: value,
      current: 1,
    } 
    setQuery(_query);
    if (field === 'planName') return;
    props.onChange( _query);
  }

   

  function handleChangeDates (dates){
    const _query = _.clone(query);
    if (_.isEmpty(dates)) {
      delete _query.startTime;
      delete _query.endTime;
    } else {
      _query.startTime = dates[0]?.startOf('day').valueOf()
      _query.endTime = dates[1]?.endOf('day').valueOf()
    }
    const param ={
      ..._query,
      current: 1,
    }
    setQuery(param);
    props.onChange(param);
  }

  // 重置搜索条件
  function  handleReset(){
    setQuery(DEFAULT_QUERY);
    props.onChange(DEFAULT_QUERY);
  }

  function getCreatorList(current = 1, selectList){
    const _query = {
      ...creatorQuery,
      current,
      size:10
    };
    StoreService.getStoreUserBasicPage( _query).then((res) => {
        const { result = {} } = res;
        const { records = [], hasNext } = result;
        const list = current > 1 ? creatorList.concat(records) : records;
        setHasNext(hasNext);
        setCreatorList(list);
    });
  }

   // 滑动加载更多讲师列表
  function handleScrollCreatorList(e){
    const container = e.target;
    const scrollToBottom = container &&  container.scrollHeight <= container.clientHeight + container.scrollTop;
    if (scrollToBottom && hasNext) {
      const _creatorQuery = {...creatorQuery};
      _creatorQuery.current = creatorQuery.current + 1;
      setCreatorQuery(_creatorQuery);
      getCreatorList(creatorQuery.current + 1);
    }
  }


  return (
      <div className="plan-filter">
        <Row>
            <div className="search-condition">
              <div className="search-condition__item">
                <span className="search-name">培训计划：</span>
                <Search
                  value={query.planName}
                  placeholder="搜索培训计划名称"
                  onChange={(e) => { handleChangeQuery('planName', e.target.value)}}
                  onSearch={ () => { props.onChange(query) } }
                  style={{ width: "calc(100% - 70px)" }}
                  enterButton={<span className="icon iconfont">&#xe832;</span>}
                />
              </div>
             
              <div className="search-condition__item">
                  <span>创建人：</span>
                  <Select
                    placeholder="请选择创建人"
                    style={{width:"calc(100% - 70px)"}}
                    showSearch
                    allowClear
                    filterOption={(input, option) => option}
                    suffixIcon={<span className="icon iconfont" style={{fontSize:'12px',color:'#BFBFBF'}}>&#xe835;</span>}
                    onPopupScroll={handleScrollCreatorList}
                    value={query.createId}
                    onChange={(value) => {
                      handleChangeQuery('createId', value) 
                    }}
                    onSearch={(value) => {
                      creatorQuery.nickName = value
                        setCreatorQuery(creatorQuery)
                        getCreatorList()
                      }
                    }
                    onClear ={(value)=>{
                      setCreatorQuery({
                        size: 10,
                        current: 1,
                        nickName:null
                      })
                      getCreatorList()
                    }
                  }
                  >
                    {_.map(creatorList, (item, index) => {
                      return (
                        <Select.Option value={item.id} key={item.id}>{item.nickName}</Select.Option>
                      );
                    })}
                  </Select>
              </div>
              <div className="search-condition__item">
                <span className="search-date">创建日期：</span>
                <RangePicker
                  id="course_date_picker"
                  allowClear={false}
                  value={ query.startTime ? [moment(query.startTime), moment(query.endTime)] : null }
                  format={"YYYY-MM-DD"}
                  onChange={(dates) => { handleChangeDates(dates) }}
                  style={{ width: "calc(100% - 70px)" }}
                />
              </div>
              { expandFilter &&
                <div className="search-condition__item">
                  <span className="shelf-status">当前状态：</span>
                  <Select
                    style={{ width: "calc(100% - 70px)" }}
                    placeholder="请选择当前状态"
                    allowClear={true}
                    value={query.enableState}
                    onChange={(value) => { handleChangeQuery('enableState', value) }}
                    suffixIcon={<span className="icon iconfont" style={{fontSize:'12px',color:'#BFBFBF'}}>&#xe835;</span>}
                  >
                    <Option value="YES">开启</Option>
                    <Option value="NO">关闭</Option>
                  </Select>
                </div>
              }
            </div>  

          <div className="reset-fold-area">
            <Tooltip title="清空筛选"><span className="resetBtn iconfont icon" onClick={handleReset}>&#xe61b; </span></Tooltip>
            <span style={{ cursor: 'pointer' }} className="fold-btn" onClick={() => {
              setExpandFilter(!expandFilter)
            }}>{expandFilter ? <span><span>收起</span><span className="iconfont icon fold-icon" >&#xe82d; </span> </span> : <span>展开<span className="iconfont icon fold-icon" >&#xe835; </span></span>}</span>
         </div>
        </Row>
      </div>
    )
 
}

export default withRouter(PlanFilter);