/*
 * @Author: wufan
 * @Date: 2020-07-09 14:03:09
 * @Last Modified by: mikey.zhaopeng
 * @Last Modified time: 2020-11-28 13:53:12
 * 店铺管理-员工管理
 */
import React, { useEffect, useState } from "react";
import { withRouter } from "react-router-dom";
import _ from "underscore";
import PageControl from "@/components/PageControl";
import { CheckBox } from "@/components";
import { Button, Table, Modal, message, Input } from "antd";
import { QuestionCircleOutlined } from "@ant-design/icons";
import StoreService from "@/domains/store-domain/storeService";
import EmployeeAddOrEditModal from "./EmployeeAddOrEditModal";
import User from "@/common/js/user";

import "./EmployeesManagePage.less";
import Item from "antd/lib/list/Item";
const { confirm } = Modal;
const { Search } = Input;

declare var window: any;
interface RecordTypes {
  id: string;
  role: string;
  roleCodes: Array<string>;
  userId: string;
  nickName: string;
  phone: string;
  avatar?: string;
  weChatAccount?: string;
}

interface RoleItemType {
  isChecked: boolean;
  roleCode: string;
  name: string;
  id: string;
}

interface QueryType {
  current: number;
  size: number;
  nickName?: string;
  phone?: string;
  roleCodes: Array<string>;
}

interface ChoosedItemType {
  nickName?: string;
  phone?: string;
  role: Array<string>;
  avatar?: string;
  storeUserId?: string;
  weChatAccount?: string;
}

function EmployeesManagePage() {
  const [employeeList, setEmployeeList] = useState([]);
  const [query, setQuery] = useState<QueryType>({
    current: 0,
    size: 10,
    nickName: "",
    phone: "",
    roleCodes: [],
  });

  const [valueLike, setValueLike] = useState();
  const [total, setTotal] = useState(0);
  const [model, setModel] = useState<React.ReactNode>(null);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [isManager, setIsManager] = useState(true);
  const [isNormal, setIsNormal] = useState(true);
  const [choosedItem, setChooseItem] = useState<ChoosedItemType>({
    nickName: "",
    phone: "",
    role: [],
    avatar: "",
  });
  const [roleIds, setRoleIds] = useState<Array<RoleItemType>>([]);
  const roleMap = {
    CloudManager: "管理员",
    CloudLecturer: "讲师",
    StoreManager: "店铺管理员",
  };
  const storeId = User.getStoreId();
  const StoreType = User.getStoreType();
  const isWorkWechat = !!(StoreType === "WE_CHAT_STORE");

  useEffect(() => {
    getEmployeeList();
  }, [query]);

  useEffect(() => {
    getListInfo();
  }, [storeId]);

  async function getListInfo() {
    await getStoreRole();
  }

  function getEmployeeList() {
    let _query = _.clone(query);
    _query.current = query.current + 1;
    StoreService.getEmployeeList(_query).then((res: any) => {
      setEmployeeList(res.result.records);
      setTotal(res.result.total);
    });
  }

  function getStoreRole() {
    StoreService.getStoreRole({ storeId }).then((res: any) => {
      const data = [...res.result];
      const _query = { ...query };

      let _data = _.filter(data, (_item) => {
        return _item.roleCode !== "StoreManager";
      });

      setRoleIds(_data);
    });
  }

  function parseColumn() {
    const columns = [
      {
        title: "员工",
        dataIndex: "nickName",
        render: (val: string, record: RecordTypes) => {
          return (
            <div className="employee-info">
              {isWorkWechat && (
                <img
                  src={
                    record.avatar ||
                    "https://image.xiaomaiketang.com/xm/rJeQaZxtc7.png"
                  }
                  alt=""
                />
              )}

              <span className="title">{val}</span>
            </div>
          );
        },
      },
      {
        title: "手机号",
        dataIndex: "phone",
        key: "phone",
        render: (val: string) => {
          return <div>{val}</div>;
        },
      },
      {
        title: "身份",
        dataIndex: "role",
        key: "role",
        render: (val: string) => {
          return <div>{val.split("，").join("、")}</div>;
        },
      },
      {
        title: "操作",
        dataIndex: "operation",
        render: (val: string, record: RecordTypes) => {
          return record.role === "店铺管理员" ||
            record.userId === User.getUserId() ? (
            <div className="no-operate">-</div>
          ) : (
            <div className="operation">
              <span className="edit" onClick={() => handleEditEmployee(record)}>
                编辑
              </span>
              <span className="divider-line">{" | "}</span>
              <span
                className="delete"
                onClick={() =>{
                  isWorkWechat?
                  handleDeleteWorkWechatEmployeeConfirm(record) :
                  handleDeleteEmployeeConfirm(record)
                }}
              >
                删除
              </span>
            </div>
          );
        },
      },
    ];

    if (isWorkWechat && columns) {
      const item = {
        title: "企业微信账号",
        dataIndex: "weChatAccount",
        key: "weChatAccount",
        render: (val: string) => {
          return <div>{val}</div>;
        },
      };
      columns.splice(1, 1, item);
    }
    return columns;
  }

  function handleEditEmployee(record: RecordTypes) {
    const { nickName, phone, roleCodes, avatar, id, weChatAccount } = record;
    const _choosesItem = {
      nickName: nickName,
      phone: phone,
      role: roleCodes,
      avatar: avatar,
      storeUserId: id,
      weChatAccount
    };
    setChooseItem(_choosesItem);
    const model: React.ReactNode = (
      <EmployeeAddOrEditModal
        choosedItem={_choosesItem}
        onClose={() => {
          setModel(null);
          getEmployeeList();
          setChooseItem({
            nickName: "",
            phone: "",
            role: [],
            avatar: "",
            storeUserId: "",
            weChatAccount:""
          });
        }}
        isWorkWechat={isWorkWechat}
      />
    );
    setModel(model);
  }

  function handleDeleteEmployeeConfirm(record: RecordTypes) {
    return confirm({
      title: "你确定要删除此讲师吗？",
      content: "删除后，讲师将不能登录系统，此操作不能被撤销",
      icon: (
        <span className="icon iconfont default-confirm-icon">&#xe839; </span>
      ),
      okText: "删除",
      okType: "danger",
      cancelText: "取消",
      onOk: () => {
        handleDeleteEmployee(record.id);
      },
    });
  }

  function handleDeleteWorkWechatEmployeeConfirm(record: RecordTypes) {
    return confirm({
      title: "你确定要删除此员工吗？",
      content: "管理员需在企业微信后台的“小麦企培”应用管理中移除此员工，此员工才无法继续登陆小麦企培",
      icon: (
        <span className="icon iconfont default-confirm-icon">&#xe839; </span>
      ),
      okText: "删除",
      okType: "danger",
      cancelText: "取消",
      onOk: () => {
        handleDeleteEmployee(record.id);
      },
    });
  }

  function handleToAddEmployee() {
    const model: React.ReactNode = (
      <EmployeeAddOrEditModal
        choosedItem={choosedItem}
        onClose={() => {
          setModel(null);
          getEmployeeList();
          setChooseItem({
            nickName: "",
            phone: "",
            role: [],
            avatar: "",
            storeUserId: "",
          });
        }}
        isWorkWechat={isWorkWechat}
      />
    );
    setModel(model);
  }

  function handleDeleteEmployee(storeUserId: string) {
    StoreService.deleteEmployee({ storeUserId }).then((res: any) => {
      const msg = isWorkWechat ? "员工已删除":"讲师已删除";
      message.success(msg);
      getEmployeeList();
    });
  }

  function handleQuery(name: string, value: any) {
    const _query = _.clone(query);
    setQuery(_query);
  }

  return (
    <div className=" page employee-manage-page">
        <div className="content-header">员工管理</div>
        <div className="box">
          <div className="box-header">
            <div
              style={{
                display: "flex",
                alignItems: "center",
                justifyContent: "flex-start",
                padding: "0px 0 16px",
              }}
            >
              <div>
                搜索员工：
                <Search
                  style={{
                    width: 300,
                    marginRight: 40,
                  }}
                  placeholder={
                    isWorkWechat ? "请输入员工昵称" : "搜索员工昵称/手机号"
                  }
                  onSearch={(value) => {
                    const _query = { ...query };
                    // 企业微信用户只能搜索员工昵称
                    if (isWorkWechat) {
                      _query.nickName = value;
                      _query.current = 0;
                      setQuery(_query);
                      return;
                    }

                    if (value) {
                      const isPhone = (value || "").match(/^\d+$/);
                      const name = isPhone ? "phone" : "nickName";
                      const otherName = isPhone ? "nickName" : "phone";
                      _query[name] = value;
                      _query[otherName] = "";
                      _query.current = 0;
                    } else {
                      _query.nickName = "";
                      _query.phone = "";
                      _query.current = 0;
                    }
                    setQuery(_query);
                  }}
                  enterButton={<span className="icon iconfont">&#xe832;</span>}
                />
              </div>

              <div>
                身份：
                {_.map(roleIds, (item: any) => {
                  return (
                    <CheckBox
                      text={item.name}
                      name={item.roleCode}
                      onChange={(e: any) => {
                        const { checked, name } = e.target;
                        const _query = { ...query };
                        _query.roleCodes = [];
                        _query.current = 0;
                        const _roleIds: Array<RoleItemType> = roleIds.map(
                          (_item: RoleItemType) => {
                            if (name === _item.roleCode) {
                              if (checked) {
                                _item.isChecked = true;
                              } else {
                                _item.isChecked = false;
                              }
                            }
                            if (_item.isChecked) {
                              _query.roleCodes.push(_item.roleCode);
                            }
                            return _item;
                          }
                        );

                        setRoleIds(_roleIds);
                        setQuery(_query);
                      }}
                      defaultChecked={item.isChecked}
                    />
                  );
                })}
              </div>
            </div>
            {!isWorkWechat && (
              <Button
                onClick={() => {
                  handleToAddEmployee();
                }}
                type="primary"
                className="add-show-btn"
              >
                添加员工
              </Button>
            )}
          </div>
          <div className="box-body">
            <Table
              size={"middle"}
              pagination={false}
              dataSource={employeeList}
              columns={parseColumn()}
              rowKey={(item: any) => item.id}
              bordered
            />
          </div>
          <div className="box-footer">
            <PageControl
              current={query.current}
              pageSize={query.size}
              total={total}
              toPage={(page: any) => {
                const queryStates = _.clone(query);
                queryStates.current = page;
                setQuery(queryStates);
              }}
            />
          </div>
        </div>
        {model}
    </div>
  );
}

export default withRouter(EmployeesManagePage);
