/*
 * @Author: 吴文洁
 * @Date: 2020-07-14 15:41:30
 * @Last Modified by: 吴文洁
 * @Last Modified time: 2020-07-23 13:45:16
 * @Description: 大班直播、互动班课列表的筛选组件
 */

import React, { useState, useEffect } from 'react';
import { Row, Input, Select, Tooltip } from 'antd';
import RangePicker from '@/modules/common/DateRangePicker';
import moment from 'moment';
import StoreService from '@/domains/store-domain/storeService';
import './TrainFilter.less';
const { Search } = Input;

const DEFAULT_QUERY = {
  // 头部筛选默认值
  taskName: null,
  createId: null, // 创建人
  startTime: null,
  endTime: null,
};
const DEFAULT_CREATOR_QUERY = {
  // 创建人列表筛选默认值
  size: 10,
  current: 1,
  nickName: null, // 搜索值
};

function TrainFilter(props) {
  const [query, setQuery] = useState(DEFAULT_QUERY);
  const [hasNext, setHasNext] = useState(false);
  const [creatorQuery, setCreatorQuery] = useState(DEFAULT_CREATOR_QUERY);
  const [creatorList, setCreatorList] = useState([]);

  useEffect(() => {
    getCreatorList();
  }, []);

  // 获取创建人列表
  function getCreatorList(current = 1) {
    const _query = {
      ...creatorQuery,
      current,
    };
    StoreService.getStoreUserBasicPage(_query).then((res) => {
      const { result = {} } = res;
      const { records = [], hasNext } = result;
      const list = current > 1 ? creatorList.concat(records) : records;
      setHasNext(hasNext);
      setCreatorList(list);
    });
  }

  // 滑动加载更多创建人列表（讲师）
  function handleScrollCreatorList(e) {
    const container = e.target;
    const scrollToBottom = container && container.scrollHeight <= container.clientHeight + container.scrollTop;
    if (scrollToBottom && hasNext) {
      const _creatorQuery = { ...creatorQuery };
      _creatorQuery.current = creatorQuery.current + 1;
      setCreatorQuery(_creatorQuery);
      getCreatorList(creatorQuery.current + 1);
    }
  }

  // 改变搜索条件
  function handleChangeQuery(field, value) {
    const _query = _.clone(query);
    _query.current = 1;
    if (field === 'createdDate') {
      _query.startTime = value[0]?.startOf('day').valueOf();
      _query.endTime = value[1]?.endOf('day').valueOf();
    } else {
      _query[field] = value;
    }
    setQuery(_query);
    if (field === 'taskName') return;
    props.onChange(_query);
  }

  // 重置搜索条件
  function handleReset() {
    setQuery(DEFAULT_QUERY);
    props.onChange(DEFAULT_QUERY);
  }

  return (
    <div className='train-filter-page'>
      <Row>
        <div className='search-condition'>
          <div className='search-condition__item'>
            <span>培训任务：</span>
            <Search
              value={query.taskName}
              placeholder='搜索培训任务名称'
              onChange={(e) => {
                handleChangeQuery('taskName', e.target.value.trim());
              }}
              onSearch={() => {
                props.onChange(query);
              }}
              style={{ width: 'calc(100% - 70px)' }}
              enterButton={<span className='icon iconfont'>&#xe832;</span>}
            />
          </div>
          <div className='search-condition__item'>
            <span>创建人：</span>
            <Select
              placeholder='请选择创建人'
              style={{ width: 'calc(100% - 70px)' }}
              showSearch
              allowClear
              filterOption={(input, option) => option}
              suffixIcon={
                <span className='icon iconfont' style={{ fontSize: '12px', color: '#BFBFBF' }}>
                  &#xe835;
                </span>
              }
              onPopupScroll={handleScrollCreatorList}
              value={query.createId}
              onChange={(value) => {
                handleChangeQuery('createId', value);
              }}
              onSearch={(value) => {
                creatorQuery.nickName = value;
                setCreatorQuery(creatorQuery);
                getCreatorList();
              }}
              onClear={() => {
                setCreatorQuery(DEFAULT_CREATOR_QUERY);
                getCreatorList();
              }}>
              {_.map(creatorList, (item, index) => {
                return (
                  <Select.Option value={item.id} key={item.id}>
                    {item.nickName}
                  </Select.Option>
                );
              })}
            </Select>
          </div>
          <div className='search-condition__item'>
            <span className='search-date'>创建日期：</span>
            <RangePicker
              id='train_date_picker'
              allowClear={false}
              value={query.startTime ? [moment(query.startTime), moment(query.endTime)] : null}
              format={'YYYY-MM-DD'}
              onChange={(value) => {
                handleChangeQuery('createdDate', value);
              }}
              style={{ width: 'calc(100% - 70px)' }}
            />
          </div>
        </div>

        <div className='reset-fold-area'>
          <Tooltip title='清空筛选'>
            <span className='resetBtn iconfont icon' onClick={handleReset}>
              &#xe61b;{' '}
            </span>
          </Tooltip>
        </div>
      </Row>
    </div>
  );
}

export default TrainFilter;
